#! /usr/bin/env python
# -*- coding: utf-8 -*-
# subroutines from PyTom

import numpy as np

################################
#   Rotation Relevant Stuff    #
################################

def rotation_matrix_x(angle):
    """Return the 3x3 rotation matrix around x axis.

    @param angle: rotation angle around x axis (in degree).

    @return: rotation matrix.
    """
    angle = np.deg2rad(angle)
    mtx = np.matrix(np.zeros((3,3)))
    mtx[1,1] = np.cos(angle)
    mtx[2,1] = np.sin(angle)
    mtx[2,2] = np.cos(angle)
    mtx[1,2] = -np.sin(angle)
    mtx[0,0] = 1

    return mtx


def rotation_matrix_y(angle):
    """Return the 3x3 rotation matrix around y axis.

    @param angle: rotation angle around y axis (in degree).

    @return: rotation matrix.
    """
    angle = np.deg2rad(angle)
    mtx = np.matrix(np.zeros((3,3)))
    mtx[0,0] = np.cos(angle)
    mtx[2,0] = -np.sin(angle)
    mtx[2,2] = np.cos(angle)
    mtx[0,2] = np.sin(angle)
    mtx[1,1] = 1

    return mtx


def rotation_matrix_z(angle):
    """Return the 3x3 rotation matrix around z axis.

    @param angle: rotation angle around z axis (in degree).

    @return: rotation matrix.
    """
    angle = np.deg2rad(angle)
    mtx = np.matrix(np.zeros((3,3)))
    mtx[0,0] = np.cos(angle)
    mtx[1,0] = np.sin(angle)
    mtx[1,1] = np.cos(angle)
    mtx[0,1] = -np.sin(angle)
    mtx[2,2] = 1

    return mtx


def rotation_matrix_zxz(angle):
    """Return the 3x3 rotation matrix of an Euler angle in ZXZ convention.
    Note the order of the specified angle should be [Phi, Psi, Theta], or [Z1, Z2, X] in Pytom format.

    @param angle: list of [Phi, Psi, Theta] in degree.

    @return: rotation matrix.
    """
    assert len(angle) == 3

    z1 = angle[0]
    z2 = angle[1]
    x = angle[2]

    zm1 = rotation_matrix_z(z1)
    xm = rotation_matrix_x(x)
    zm2= rotation_matrix_z(z2)
    
    res = zm2 * (xm * zm1)

    return res


def rotation_distance(ang1, ang2):
    """Given two angles (lists), calculate the angular distance (degree).

    @param ang1: angle 1. [Phi, Psi, Theta], or [Z1, Z2, X] in Pytom format.
    @param ang2: angle 2. [Phi, Psi, Theta], or [Z1, Z2, X] in Pytom format.
    
    @return: rotation distance in degree.
    """
    mtx1 = rotation_matrix_zxz(ang1)
    mtx2 = rotation_matrix_zxz(ang2)
    res = np.multiply(mtx1, mtx2) # elementwise multiplication
    trace = np.sum(res)
    
    from math import pi, acos
    temp=0.5*(trace-1.0)
    if temp >= 1.0:
        return 0.0
    if temp <= -1.0:
        return 180
    return acos(temp)*180/pi


def euclidian_distance(pos1, pos2):
    return np.linalg.norm(np.array(pos1)-np.array(pos2))

def fix_euler_ranges(orientation):
    """A script to fix the Euler angle ranges

    @param orientation: ZXZ Euler angles (Phi,Psi,Theta = Z1,Z2,X)
    
    @return: Equivalent Euler angles with phi and psi between 0-360, and theta 0-180
    """

    # Parse angles
    phi = orientation[0]
    psi = orientation[1]
    theta = orientation[2]

    # Set theta between 0-360
    if theta > 360:
        while theta > 360:
            theta = theta - 360
    elif theta < 0:
        while theta < 0:
            theta = theta + 360

    # Set theta between 0-180 and update psi to compensate
    if theta > 180:
        theta = 360 - theta
        psi = psi + 180

    
    # Set psi between 0-360
    if psi > 360:
        while psi > 360:
            psi = psi - 360
    elif psi < 0:
        while psi < 0:
            psi = psi + 360

    # Set phi between 0-360
    if phi > 360:
        while phi > 360:
            phi = phi - 360
    elif phi < 0:
        while phi < 0:
            phi = phi + 360

    return [phi,psi,theta]




