
#include <vector>
#include <iostream>
#include <map>

#include <epoxy/gl.h>
#include <epoxy/glx.h>
#include "graphical-molecule.hh"
#include "Shader.hh"
#define GLM_ENABLE_EXPERIMENTAL
#include <glm/gtx/string_cast.hpp>  // to_string()

#include "generic-display-object.hh"

// maybe should not be const
void
graphical_molecule::setup_simple_triangles(Shader *shader_p, const Material &material_in) {

   material = material_in;
   shader_p->Use();
   fill_with_simple_triangles_vertices();
   setup_buffers(shader_p->get_entity_type());

}

void
graphical_molecule::setup_rama_balls(Shader *shader_p, const Material &material_in) {

   material = material_in;
   shader_p->Use();
   fill_rama_balls();
   setup_buffers(shader_p->get_entity_type());
}

void
graphical_molecule::setup_instanced_balls(Shader *shader_p, const Material &material_in) {

   material = material_in;
   shader_p->Use();
   add_one_origin_ball();
   add_one_origin_ball();
   add_one_origin_ball();
   n_instances = 3;
   object_is_instanced = true;

   // Now that the vertices are filled, we need to fill in the model rotation and translation
   // components of each vertex.
   unsigned int n_vertices_per_object = vertices.size() / n_instances;
   for (unsigned int i=0; i<n_instances; i++) {
      for (unsigned int j=0; j<n_vertices_per_object; j++) {
         instanced_vertex &vertex = vertices[i*n_vertices_per_object + j];
         vertex.orientation = glm::mat3(1.0f);
         vertex.translation = glm::vec3(0.0f, 0.1f * static_cast<float>(i+1), 0.0f);
         if (i == 0) vertex.color = glm::vec4(0.2, 0.8, 0.2, 1.0);
         if (i == 1) vertex.color = glm::vec4(0.7, 0.7, 0.2, 1.0);
         if (i == 2) vertex.color = glm::vec4(0.8, 0.2, 0.2, 1.0);
      }
   }

   setup_buffers(shader_p->get_entity_type());
}

void
graphical_molecule::fill_one_dodec() {

   // These are dodec balls for the moment, not flat shaded real dodecs.
   // In a bit I will transform them to pentakis dodecs.

   dodec d;
   std::vector<clipper::Coord_orth> v = d.coords();
   vertices.resize(v.size()); // should be 20

   for (unsigned int i=0; i<v.size(); i++) {
      glm::vec3 a = glm::vec3(v[i].x(), v[i].y(), v[i].z());
      float f = static_cast<float>(i)/static_cast<float>(v.size());
      vertices[i].pos    = 0.2f * a;
      vertices[i].normal = a;
      vertices[i].color  = glm::vec4(0.3 + 0.4 * f, 0.8 - 0.8 * f, 0.1 + 0.9 * f, 1.0);
   }

   for (unsigned int i=0; i<12; i++) {
      const std::vector<unsigned int> &face = d.face(i);
      graphical_triangle gt_0(face[0], face[1], face[2]);
      graphical_triangle gt_1(face[0], face[2], face[3]);
      graphical_triangle gt_2(face[0], face[3], face[4]);
      triangle_vertex_indices.push_back(gt_0);
      triangle_vertex_indices.push_back(gt_1);
      triangle_vertex_indices.push_back(gt_2);
   }

}

void
graphical_molecule::add_one_ball(const glm::vec3 &centre) { // i.e. a smooth-shaded pentakis dodec

   pentakis_dodec pkdd(1.06);
   // coordinate system finagalling - baah - Note to self: convert dodecs to glm.
   clipper::Coord_orth centre_c(centre.x, centre.y, centre.z);
   coot::generic_display_object_t::pentakis_dodec_t penta_dodec(pkdd, 0.01, centre_c);
   std::vector<clipper::Coord_orth> v = penta_dodec.pkdd.d.coords();
   float scale = 0.05;

   unsigned int vertex_index_start_base = vertices.size();

   const std::vector<clipper::Coord_orth> &pv = penta_dodec.pkdd.pyrimid_vertices;
   for (unsigned int i=0; i<12; i++) {

      const std::vector<unsigned int> &face = penta_dodec.pkdd.d.face(i);

      // first the base point (tip of the triangles/pyrimid)
      //
      clipper::Coord_orth pvu(pv[i].unit());
      instanced_vertex gv;
      // scaling by 1.732 is a "pretty" star
      gv.pos    = glm::vec3(pv[i].x(), pv[i].y(), pv[i].z()) * scale * 1.08f + centre;
      gv.normal = glm::vec3(pvu.x(), pvu.y(), pvu.z());
      vertices.push_back(gv);

      for (unsigned int j=0; j<5; j++) {
         const clipper::Coord_orth &pt = v[face[j]];
         clipper::Coord_orth ptu(pt.unit());
         gv.pos    = glm::vec3(pt.x(),  pt.y(),  pt.z()) * scale + centre;
         gv.normal = glm::vec3(ptu.x(), ptu.y(), ptu.z());
         gv.color  = glm::vec4(0.5f, 0.5f, 0.5f, 1.0f);
         vertices.push_back(gv);
      }
      unsigned int idx_base = vertex_index_start_base + i*6;
      graphical_triangle gt_0(idx_base, idx_base + 1, idx_base + 2);
      graphical_triangle gt_1(idx_base, idx_base + 2, idx_base + 3);
      graphical_triangle gt_2(idx_base, idx_base + 3, idx_base + 4);
      graphical_triangle gt_3(idx_base, idx_base + 4, idx_base + 5);
      graphical_triangle gt_4(idx_base, idx_base + 5, idx_base + 1);
      triangle_vertex_indices.push_back(gt_0);
      triangle_vertex_indices.push_back(gt_1);
      triangle_vertex_indices.push_back(gt_2);
      triangle_vertex_indices.push_back(gt_3);
      triangle_vertex_indices.push_back(gt_4);
   }
}

void
graphical_molecule::add_one_origin_ball() { // i.e. a smooth-shaded pentakis dodec

   pentakis_dodec pkdd(1.06); // what does this number do?
   // coordinate system finagalling - baah - Note to self: convert dodecs to glm.
   clipper::Coord_orth centre_c(0,0,0);
   coot::generic_display_object_t::pentakis_dodec_t penta_dodec(pkdd, 0.01, centre_c);
   std::vector<clipper::Coord_orth> v = penta_dodec.pkdd.d.coords();
   float scale = 0.04;

   unsigned int vertex_index_start_base = vertices.size();

   const std::vector<clipper::Coord_orth> &pv = penta_dodec.pkdd.pyrimid_vertices;
   for (unsigned int i=0; i<12; i++) {

      const std::vector<unsigned int> &face = penta_dodec.pkdd.d.face(i);

      // first the base point (tip of the triangles/pyrimid)
      //
      clipper::Coord_orth pvu(pv[i].unit());
      instanced_vertex gv;
      gv.pos    = glm::vec3(pv[i].x(), pv[i].y(), pv[i].z()) * scale;
      gv.normal = glm::vec3(pvu.x(), pvu.y(), pvu.z());
      vertices.push_back(gv);

      for (unsigned int j=0; j<5; j++) {
         const clipper::Coord_orth &pt = v[face[j]];
         clipper::Coord_orth ptu(pt.unit());
         gv.pos    = glm::vec3(pt.x(),  pt.y(),  pt.z()) * scale;
         gv.normal = glm::vec3(ptu.x(), ptu.y(), ptu.z());
         gv.color  = glm::vec4(0.5f, 0.5f, 0.5f, 1.0f);
         vertices.push_back(gv);
      }
      unsigned int idx_base = vertex_index_start_base + i*6;
      graphical_triangle gt_0(idx_base, idx_base + 1, idx_base + 2);
      graphical_triangle gt_1(idx_base, idx_base + 2, idx_base + 3);
      graphical_triangle gt_2(idx_base, idx_base + 3, idx_base + 4);
      graphical_triangle gt_3(idx_base, idx_base + 4, idx_base + 5);
      graphical_triangle gt_4(idx_base, idx_base + 5, idx_base + 1);
      triangle_vertex_indices.push_back(gt_0);
      triangle_vertex_indices.push_back(gt_1);
      triangle_vertex_indices.push_back(gt_2);
      triangle_vertex_indices.push_back(gt_3);
      triangle_vertex_indices.push_back(gt_4);
   }
}


void
graphical_molecule::fill_rama_balls() {

   glm::vec3 centre(0,0,0.0);
   add_one_ball(centre);

   for (unsigned int i=0; i<10; i++) {
      float f = static_cast<float>(i)/10.0;
      glm::vec3 c(0.5, 0.0, f - 1.0);
      add_one_ball(c);
   }
}

void
graphical_molecule::setup_buffers(const Shader::Entity_t &entity_type) {

   glGenVertexArrays (1, &vao);
   glBindVertexArray (vao);

   glGenBuffers(1, &buffer_id);
   glBindBuffer(GL_ARRAY_BUFFER, buffer_id);
   std::cout << "debug:: sizeof(vertices) "    << sizeof(vertices) << std::endl;
   std::cout << "debug:: sizeof(vertices[0]) " << sizeof(vertices[0]) << std::endl;
   std::cout << "debug:: sizeof(instanced_vertex) " << sizeof(instanced_vertex) << std::endl;
   unsigned int n_vertices = vertices.size();
   std::cout << "debug:: glBufferData for simple_triangles_buffer_id " << buffer_id
             << " allocating with size " << n_vertices * sizeof(vertices[0]) << std::endl;

   if (false) {

      glBufferData(GL_ARRAY_BUFFER, n_vertices * sizeof(vertices[0]), &(vertices[0]), GL_STATIC_DRAW);

      glEnableVertexAttribArray(0);
      glVertexAttribPointer (0, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), 0);
      
      glEnableVertexAttribArray (1);
      glVertexAttribPointer (1, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex),
                             reinterpret_cast<void *>(sizeof(glm::vec3)));
      
      glEnableVertexAttribArray (2);
      glVertexAttribPointer (2, 4, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex),
                             reinterpret_cast<void *>(2 * sizeof(glm::vec3)));
   } else {

      glBufferData(GL_ARRAY_BUFFER, n_vertices * sizeof(vertices[0]), &(vertices[0]), GL_STATIC_DRAW);

      // "from-origin" model matrix (orientation)
      GLenum err = glGetError(); if (err) std::cout << "GL error  17c\n";
      glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), reinterpret_cast<void *>(0 * sizeof(glm::vec3)));
      err = glGetError(); if (err) std::cout << "GL error  17c\n";
      glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), reinterpret_cast<void *>(1 * sizeof(glm::vec3)));
      err = glGetError(); if (err) std::cout << "GL error  17c\n";
      glVertexAttribPointer(2, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), reinterpret_cast<void *>(2 * sizeof(glm::vec3)));
      err = glGetError(); if (err) std::cout << "GL error  17c\n";
      glEnableVertexAttribArray(0);
      glEnableVertexAttribArray(1);
      glEnableVertexAttribArray(2);

      // "from origin" translate position, 3, size 3 floats
      glEnableVertexAttribArray(3);
      glVertexAttribPointer(3, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), reinterpret_cast<void *>(3 * sizeof(glm::vec3)));
      err = glGetError(); if (err) std::cout << "GL error  17aa\n";

      // positions, 4, size 3 floats
      glEnableVertexAttribArray(4);
      err = glGetError(); if (err) std::cout << "GL error  6\n";
      glVertexAttribPointer(4, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), reinterpret_cast<void *>(4 * sizeof(glm::vec3)));
      err = glGetError(); if (err) std::cout << "GL error  7\n";

      //  normals, 5, size 3 floats
      glEnableVertexAttribArray(5);
      err = glGetError(); if (err) std::cout << "GL error  11\n";
      glVertexAttribPointer(5, 3, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), reinterpret_cast<void *>(5 * sizeof(glm::vec3)));
      err = glGetError(); if (err) std::cout << "GL error  12\n";

      //  colours, 6, size 4 floats
      glEnableVertexAttribArray(6);
      err = glGetError(); if (err) std::cout << "GL error  16\n";
      glVertexAttribPointer(6, 4, GL_FLOAT, GL_FALSE, sizeof(instanced_vertex), reinterpret_cast<void *>(6 * sizeof(glm::vec3)));
      err = glGetError(); if (err) std::cout << "GL error  17\n";

   }

   glGenBuffers(1, &index_buffer_id);
   GLenum err = glGetError(); if (err) std::cout << "GL error setup_simple_triangles()\n";
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, index_buffer_id);
   err = glGetError(); if (err) std::cout << "GL error setup_simple_triangles()\n";
   unsigned int n_triangles = triangle_vertex_indices.size();
   unsigned int n_bytes_for_indices = n_triangles * 3 * sizeof(unsigned int);
   glBufferData(GL_ELEMENT_ARRAY_BUFFER, n_bytes_for_indices, &triangle_vertex_indices[0], GL_STATIC_DRAW);
   err = glGetError(); if (err) std::cout << "GL error setup_simple_triangles()\n";

   glDisableVertexAttribArray (0);
   glBindBuffer(GL_ARRAY_BUFFER, 0);
   glUseProgram(0);

}

void
graphical_molecule::fill_with_simple_triangles_vertices() {

   vertices.resize(2 * 3);

   vertices[0].pos = glm::vec3(  0.f,   0.5f,  -0.2f);
   vertices[1].pos = glm::vec3(  0.5f, -0.36f, -0.2f);
   vertices[2].pos = glm::vec3( -0.5f, -0.36f, -0.2f);
   vertices[3].pos = glm::vec3(  0.0f,  0.5f,   0.2f);
   vertices[4].pos = glm::vec3(  0.5f, -0.36f,  0.2f);
   vertices[5].pos = glm::vec3( -0.5f, -0.36f,  0.2f);

   vertices[0].normal = glm::vec3( 0.2f, 0.2f,  0.9f);
   vertices[1].normal = glm::vec3( 0.2f, 0.9f,  0.2f);
   vertices[2].normal = glm::vec3( 0.9f, 0.3f,  0.1f);
   vertices[3].normal = glm::vec3( 0.0f, 0.9f, -0.1f);
   vertices[4].normal = glm::vec3( 0.9f, 0.3f, -0.2f);
   vertices[5].normal = glm::vec3( 0.2f, 0.6f, -0.9f);

   vertices[0].color = glm::vec4(0.0f, 0.0f, 0.0f, 1.f);
   vertices[1].color = glm::vec4(0.2f, 0.3f, 1.0f, 1.f);
   vertices[2].color = glm::vec4(0.5f, 0.9f, 0.2f, 1.f);
   vertices[3].color = glm::vec4(0.2f, 0.2f, 0.9f, 1.f);
   vertices[4].color = glm::vec4(0.1f, 0.9f, 0.2f, 1.f);
   vertices[5].color = glm::vec4(0.9f, 0.3f, 0.2f, 1.f);

   graphical_triangle gt_0(0,1,2);
   graphical_triangle gt_1(3,4,5);
   triangle_vertex_indices.push_back(gt_0);
   triangle_vertex_indices.push_back(gt_1);

}

void
graphical_molecule::draw(Shader *shader_p,
                         const glm::mat4 &mvp,
                         const std::map<unsigned int, gl_lights_info_t> &lights) {

   GLenum err = glGetError(); if (err) std::cout << "ERROR:: in GM draw() --end--\n";
   shader_p->Use();
   glUniformMatrix4fv(shader_p->mvp_uniform_location, 1, GL_FALSE, &mvp[0][0]);

   std::map<unsigned int, gl_lights_info_t>::const_iterator it;
   unsigned int light_idx = 0;
   it = lights.find(light_idx);
   if (it != lights.end())
      shader_p->setup_light(light_idx, it->second);

   // add material properties, use class built-ins this time.

   err = glGetError();
   if (err) std::cout << "   error draw() pre-setting material " << err << std::endl;
   shader_p->set_vec4_for_uniform("material.ambient",  material.ambient);
   shader_p->set_vec4_for_uniform("material.diffuse",  material.diffuse);
   shader_p->set_vec4_for_uniform("material.specular", material.specular);
   shader_p->set_float_for_uniform("material.shininess", material.shininess);

   // bind the vertices and their indices

   glBindVertexArray(vao);
   err = glGetError();
   if (err) std::cout << "   error glBindVertexArray() " << vao
                      << " with GL err " << err << std::endl;

   glBindBuffer(GL_ARRAY_BUFFER, buffer_id);
   err = glGetError(); if (err) std::cout << "   error draw() glBindBuffer() v "
                                          << err << std::endl;
   glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, index_buffer_id);
   err = glGetError(); if (err) std::cout << "   error draw() glBindBuffer() i "
                                          << err << std::endl;

   glEnableVertexAttribArray(0);
   glEnableVertexAttribArray(1);
   glEnableVertexAttribArray(2);
   glEnableVertexAttribArray(3);
   glEnableVertexAttribArray(4);
   glEnableVertexAttribArray(5);
   glEnableVertexAttribArray(6);
   unsigned int n_triangles = triangle_vertex_indices.size();
   GLuint n_verts = 3 * n_triangles;

   if (object_is_instanced) {
      glDrawElementsInstanced(GL_TRIANGLES, n_verts, GL_UNSIGNED_INT, nullptr, n_instances);
      err = glGetError(); if (err) std::cout << "ERROR:: in GM draw() " << shader_p->name
                                             << " glDrawElementsInstanced()\n";
   } else {
      glDrawElements(GL_TRIANGLES, n_verts, GL_UNSIGNED_INT, nullptr);
      err = glGetError(); if (err) std::cout << "ERROR:: in GM draw() " << shader_p->name
                                             << " glDrawElements()\n";
   }

   glDisableVertexAttribArray (0);
   glDisableVertexAttribArray (1);
   glDisableVertexAttribArray (2);
   glBindBuffer (GL_ARRAY_BUFFER, 0);
   glUseProgram (0);
   err = glGetError(); if (err) std::cout << "ERROR:: in GM draw() --end--\n";

}
