//
// This file is part of ProSMART.
//

#include "align_alignment_algorithm.h"

const int NO_MOMENTS = 1;

bool further_refine_erase(Align &alignment, Frag &f1, Frag &f2, Array2D<double> &FDM)
// this function erases clashing aligned fragment-pairs - the best scoring fragments are fixed.
// only one fragment-pair is fixed per iteration (function should be called recursively).
{
    int M = f1.len();			//fragment length
	vector<bool> fixed;
	int frag1 = 0;
	int frag2 = 0;
	int diff1 = 0;
	int diff2 = 0;
	bool made_change = 0;
    
	for(unsigned int i=1; i<=alignment.len(); i++){
		fixed.push_back(1);
	}
	for(unsigned int i=1; i<alignment.len(); i++){	//pre-filter, to identify inherently fixed (non-clashing) fragments
		frag1 = f1.idx(alignment.get(1,i));
		frag2 = f2.idx(alignment.get(2,i));
		for(int j=1; j<M; j++){
			if(i+j <= alignment.len()){
				diff1 = f1.idx(alignment.get(1,i+j)) - frag1;
				diff2 = f2.idx(alignment.get(2,i+j)) - frag2;
				if(diff1 >= M && diff2 >= M){
					break;		//sufficient seperation
				}
				if(diff1 != diff2){		//clash
					fixed[i-1] = 0;
					fixed[i-1+j] = 0;
				}		
			}
		}
		//cout << endl << f1.idx(alignment.get(1,i)) << " " << f2.idx(alignment.get(2,i)) << "\t" << FDM[alignment.get(1,i)][alignment.get(2,i)] << "  \t" << fixed[i-1];
	}
	
	made_change = 0;
	for(unsigned int i=0; i<fixed.size(); i++){		//check whether there are any clashes
		if(fixed[i]==0){
			made_change = 1;
			break;
		}
	}
	if(made_change == 0){			//if no clashes then return
		return 0;
	}
	
	while(made_change == 1){
		made_change = further_refine_erase_iterate(alignment, f1, f2, FDM, M, fixed);
		/*cout << endl;
         for(unsigned int i=1; i<=alignment.len(); i++){
         cout << endl << f1.idx(alignment.get(1,i)) << " " << f2.idx(alignment.get(2,i)) << "\t" << FDM[alignment.get(1,i)][alignment.get(2,i)] << "  \t" << fixed[i-1];
         }*/
	}
	
	return 1;
}

bool further_refine_erase_iterate(Align &alignment, Frag &f1, Frag &f2, Array2D<double> &FDM, int M, vector<bool> &fixed)
// this function is called iteratively by further_refine_erase_new()
{
    bool made_change = 0;
	int best_frag1 = 0;
	int best_frag2 = 0;
	int diff1 = 0;
	int diff2 = 0;
	double best_score = -1.0;
	int best_idx = 0;
	
	for(unsigned int i=1; i<=alignment.len(); i++){	//get fragment-pair with best score
		if(fixed[i-1]==1){
			continue;					//only consider fragment-pairs that have not yet been fixed
		}
		if(best_score < 0.0){
			best_score = FDM[alignment.get(1,i)][alignment.get(2,i)];
			best_idx = i;
		} else {
			if(FDM[alignment.get(1,i)][alignment.get(2,i)]<best_score){
				best_score = FDM[alignment.get(1,i)][alignment.get(2,i)];
				best_idx = i;
			}
		}
	}
	
	if(best_idx == 0){
		return 0;								//clash removal finished!
	}
	
	fixed[best_idx-1] = 1;		//fix fragment-pair
	
	made_change = 1;
	best_frag1 = f1.idx(alignment.get(1,best_idx));
	best_frag2 = f2.idx(alignment.get(2,best_idx));
	while(made_change==1){
		if(best_idx==1){break;}															//beginning of alignment
		diff1 = best_frag1 - f1.idx(alignment.get(1,best_idx-1));
		diff2 = best_frag2 - f2.idx(alignment.get(2,best_idx-1));
		//cout << endl << "\t" << diff1 << " " << diff2 << " " << M;
		if((diff1 < M || diff2 < M) && (diff1 != diff2)){		//clash
			alignment.erase(best_idx-2);											//erase previous alignment
			fixed.erase(fixed.begin()+best_idx-2);
			best_idx -= 1;
		} else {
			made_change = 0;
		}
	}
	made_change = 1;
	while(made_change==1){
		if(best_idx >= (int)alignment.len()){break;}							//end of alignment
		diff1 = f1.idx(alignment.get(1,best_idx+1)) - best_frag1;
		diff2 = f2.idx(alignment.get(2,best_idx+1)) - best_frag2;
		//cout << endl << "\t" << diff1 << " " << diff2 << " " << M;
		if((diff1 < M || diff2 < M) && (diff1 != diff2)){		//clash
			alignment.erase(best_idx);											//erase next alignment
			fixed.erase(fixed.begin()+best_idx);
		} else {
			made_change = 0;
		}
	}
	
	return 1;
}

bool further_refine_point(Align &alignment, Frag &f1, Frag &f2, Array2D<double> &FDM)
// only one change per discontinuity is allowed (function should be called recursively).
{
    bool upper = 0;
    bool lower = 0;
    bool made_change = 0;
    for(unsigned int i=1; i<alignment.len(); i++){		//refine alignment
        //cout << endl << "alignment: " << i << "\tscore: " << FDM[alignment.get(1,i)][alignment.get(2,i)];
        //cout << endl << "\tfrag_i: " << alignment.get(1,i) << "\t res_i: " << f1.idx(alignment.get(1,i)) << "\t orig_i: " << orig1[f1.idx(alignment.get(1,i))].res;
        //cout << endl << "\tfrag_i: " << alignment.get(2,i) << "\t res_i: " << f2.idx(alignment.get(2,i)) << "\t orig_i: " << orig2[f2.idx(alignment.get(2,i))].res;
        lower = 0;
        upper = 0;
        if( (alignment.get(1,i)+1 != alignment.get(1,i+1)) && (alignment.get(2,i)+1 != alignment.get(2,i+1)) ){		//if there is a gap in the fragment alignment
            //cout << " ***";
            if( (f1.idx(alignment.get(1,i))+1 == f1.idx(alignment.get(1,i)+1)) && (f2.idx(alignment.get(2,i))+1 == f2.idx(alignment.get(2,i)+1)) ){	//if the next fragment-pair is consecutive in residue
                //cout << endl << "\t*  " << FDM[alignment.get(1,i)+1][alignment.get(2,i)+1] << " " << FDM[alignment.get(1,i+1)][alignment.get(2,i+1)];
                if(FDM[alignment.get(1,i+1)][alignment.get(2,i+1)] > FDM[alignment.get(1,i)+1][alignment.get(2,i)+1]){
                    //cout << " ***";
                    //lower = FDM[alignment.get(1,i)+1][alignment.get(2,i)+1];
                    lower = 1;
                }
            }
            if( (f1.idx(alignment.get(1,i+1))-1 == f1.idx(alignment.get(1,i+1)-1)) && (f2.idx(alignment.get(2,i+1))-1 == f2.idx(alignment.get(2,i+1)-1)) ){	//if the previous fragment-pair is consecutive in residue
                //cout << endl << "\t** " << FDM[alignment.get(1,i)][alignment.get(2,i)] << " " << FDM[alignment.get(1,i+1)-1][alignment.get(2,i+1)-1];
                if(FDM[alignment.get(1,i+1)-1][alignment.get(2,i+1)-1] < FDM[alignment.get(1,i)][alignment.get(2,i)]){
                    //cout << " ***";
                    //upper = FDM[alignment.get(1,i+1)-1][alignment.get(2,i+1)-1];
                    upper = 1;
                }
            }
            
            if(lower == 1 && upper == 1){
                if(FDM[alignment.get(1,i)+1][alignment.get(2,i)+1] < FDM[alignment.get(1,i+1)-1][alignment.get(2,i+1)-1]){
                    upper = 0;
                } else {
                    lower = 0;
                }
            }
            if(lower == 1){
                alignment.edit_entry(i,alignment.get(1,i)+1,alignment.get(2,i)+1);
                made_change = 1;
                i++;
            } else if(upper == 1){
                alignment.edit_entry(i-1,alignment.get(1,i+1)-1,alignment.get(2,i+1)-1);
                made_change = 1;
            }
        }
    }
    
    return made_change;
}

bool further_refine_extend(Align &alignment, Frag &f1, Frag &f2, Array2D<double> &FDM)
// only one extension per discontinuity is allowed (function should be called recursively).
{
    int M = f1.len();
    bool lower = 0;
    bool upper = 0;
    bool made_change = 0;
    for(unsigned int i=1; i<alignment.len(); i++){		//extend alignment
        //cout << endl << "alignment: " << i << "\tscore: " << FDM[alignment.get(1,i)][alignment.get(2,i)];
        //cout << endl << "\tfrag_i: " << alignment.get(1,i) << "\t res_i: " << f1.idx(alignment.get(1,i)) << "\t orig_i: " << orig1[f1.idx(alignment.get(1,i))].res;
        //cout << endl << "\tfrag_i: " << alignment.get(2,i) << "\t res_i: " << f2.idx(alignment.get(2,i)) << "\t orig_i: " << orig2[f2.idx(alignment.get(2,i))].res;
        lower = 0;
		upper = 0;
        if( (alignment.get(1,i)+1 != alignment.get(1,i+1)) && (alignment.get(2,i)+1 != alignment.get(2,i+1)) ){		//if there is a gap in the fragment alignment
            //cout << " ***";
            if( (f1.idx(alignment.get(1,i))+1 == f1.idx(alignment.get(1,i)+1)) && (f2.idx(alignment.get(2,i))+1 == f2.idx(alignment.get(2,i)+1)) ){	//if the next fragment-pair is consecutive in residue
                //cout << endl << "\t " << (f1.idx(alignment.get(1,i+1)) - f1.idx(alignment.get(1,i)));
                //cout << " " << (f2.idx(alignment.get(2,i+1)) - f2.idx(alignment.get(2,i)));
                if( (f1.idx(alignment.get(1,i+1)) - f1.idx(alignment.get(1,i))) == (f2.idx(alignment.get(2,i+1)) - f2.idx(alignment.get(2,i))) ){	//if the aligned fragments are in agreement
                    //cout << " * ";
                    alignment.insert(i,alignment.get(1,i)+1,alignment.get(2,i)+1);
                    made_change = 1;
                    i++;		//skip the next alignment, so that this gap is only extended once in this refinement iteration
                    continue;
                }	
                //cout << endl << "\t\t*  " << FDM[alignment.get(1,i)+1][alignment.get(2,i)+1];
                //cout << "  " << f1.idx(alignment.get(1,i)+1)+M-1 << " " << f2.idx(alignment.get(2,i)+1)+M-1;
                //cout << "  " << f1.idx(alignment.get(1,i+1)) << " " << f2.idx(alignment.get(2,i+1));
                if( (f1.idx(alignment.get(1,i)+1)+M <= f1.idx(alignment.get(1,i+1))) && (f2.idx(alignment.get(2,i)+1)+M <= f2.idx(alignment.get(2,i+1))) ){	//if the next fragment-pair doesn't clash with the next alignment
                    //cout << " ***";
                    lower = 1;
                    //lower = FDM[alignment.get(1,i)+1][alignment.get(2,i)+1];
                }
            }
            if( (f1.idx(alignment.get(1,i+1))-1 == f1.idx(alignment.get(1,i+1)-1)) && (f2.idx(alignment.get(2,i+1))-1 == f2.idx(alignment.get(2,i+1)-1)) ){	//if the previous fragment-pair is consecutive in residue
                //cout << endl << "\t\t** " << FDM[alignment.get(1,i+1)-1][alignment.get(2,i+1)-1];
                //cout << "  " << f1.idx(alignment.get(1,i))+M-1 << " " << f2.idx(alignment.get(2,i))+M-1;
                //cout << "  " << f1.idx(alignment.get(1,i+1)-1) << " " << f2.idx(alignment.get(2,i+1)-1);
                if( (f1.idx(alignment.get(1,i))+M <= f1.idx(alignment.get(1,i+1)-1)) && (f2.idx(alignment.get(2,i))+M <= f2.idx(alignment.get(2,i+1)-1)) ){	//if the previous fragment-pair doesn't clash with the previous alignment
                    //cout << " ***";
                    upper = 1;
                    //upper = FDM[alignment.get(1,i+1)-1][alignment.get(2,i+1)-1];
                }
            }	  
            if(lower == 1 && upper == 1){
                if(FDM[alignment.get(1,i)+1][alignment.get(2,i)+1] < FDM[alignment.get(1,i+1)-1][alignment.get(2,i+1)-1]){
                    upper = 0;
                } else {
                    lower = 0;
                }
            }
            if(lower == 1){
                alignment.insert(i,alignment.get(1,i)+1,alignment.get(2,i)+1);
                made_change = 1;
                i++;	//skip the next alignment, so that this gap is only extended once in this refinement iteration
            } else if(upper == 1){
                alignment.insert(i,alignment.get(1,i+1)-1,alignment.get(2,i+1)-1);
                made_change = 1;
                i++;	//skip the next alignment, so that this gap is only extended once in this refinement iteration
            }
        }
    }
    
	//extend first and last alignments, where possible
	if(made_change == 1){	//only extend one (if no changes are made, then iteratively extend both ends to max)
		//cout << endl << endl << "first:\t" << alignment.get(1,1) << " " << alignment.get(2,1) << "\t" << f1.idx(alignment.get(1,1)) << " " << f2.idx(alignment.get(2,1));
		if(alignment.get(1,1) > 0 && alignment.get(2,1) > 0){
			//cout << endl << "\t" << alignment.get(1,1)-1 << " " << alignment.get(2,1)-1 << "\t" << f1.idx(alignment.get(1,1)-1) << " " << f2.idx(alignment.get(2,1)-1);
			if( (f1.idx(alignment.get(1,1))-1 == f1.idx(alignment.get(1,1)-1)) && (f2.idx(alignment.get(2,1))-1 == f2.idx(alignment.get(2,1)-1)) ){	//if the previous fragment-pair is consecutive in residue
				alignment.insert(0,alignment.get(1,1)-1,alignment.get(2,1)-1);
			}
		}
		//cout << endl << endl << "last:\t" << alignment.get(1,alignment.len()) << " " << alignment.get(2,alignment.len()) << "\t" << f1.idx(alignment.get(1,alignment.len())) << " " << f2.idx(alignment.get(2,alignment.len()));
		if(alignment.get(1,alignment.len()) < FDM.dim1()-1 && alignment.get(2,alignment.len()) < FDM.dim2()-1){
			//cout << endl << "\t" << alignment.get(1,alignment.len())+1 << " " << alignment.get(2,alignment.len())+1 << "\t" << f1.idx(alignment.get(1,alignment.len())+1) << " " << f2.idx(alignment.get(2,alignment.len())+1);
			if( (f1.idx(alignment.get(1,alignment.len()))+1 == f1.idx(alignment.get(1,alignment.len())+1)) && (f2.idx(alignment.get(2,alignment.len()))+1 == f2.idx(alignment.get(2,alignment.len())+1)) ){	//if the next fragment-pair is consecutive in residue
				alignment.insert(alignment.len(),alignment.get(1,alignment.len())+1,alignment.get(2,alignment.len())+1);
			}
		}
	} else {
		//cout << endl << endl << "first:\t" << alignment.get(1,1) << " " << alignment.get(2,1) << "\t" << f1.idx(alignment.get(1,1)) << " " << f2.idx(alignment.get(2,1));
		while(alignment.get(1,1) > 0 && alignment.get(2,1) > 0){
			//cout << endl << "\t" << alignment.get(1,1)-1 << " " << alignment.get(2,1)-1 << "\t" << f1.idx(alignment.get(1,1)-1) << " " << f2.idx(alignment.get(2,1)-1);
			if( (f1.idx(alignment.get(1,1))-1 != f1.idx(alignment.get(1,1)-1)) || (f2.idx(alignment.get(2,1))-1 != f2.idx(alignment.get(2,1)-1)) ){	//if the previous fragment-pair is consecutive in residue
                break;		//a gap in the sequence breaks the fragment alignment.
			}
			alignment.insert(0,alignment.get(1,1)-1,alignment.get(2,1)-1);
			made_change = 1;
		}
		//cout << endl << endl << "last:\t" << alignment.get(1,alignment.len()) << " " << alignment.get(2,alignment.len()) << "\t" << f1.idx(alignment.get(1,alignment.len())) << " " << f2.idx(alignment.get(2,alignment.len()));
		while(alignment.get(1,alignment.len()) < FDM.dim1()-1 && alignment.get(2,alignment.len()) < FDM.dim2()-1){
			//cout << endl << "\t" << alignment.get(1,alignment.len())+1 << " " << alignment.get(2,alignment.len())+1 << "\t" << f1.idx(alignment.get(1,alignment.len())+1) << " " << f2.idx(alignment.get(2,alignment.len())+1);
			if( (f1.idx(alignment.get(1,alignment.len()))+1 != f1.idx(alignment.get(1,alignment.len())+1)) || (f2.idx(alignment.get(2,alignment.len()))+1 != f2.idx(alignment.get(2,alignment.len())+1)) ){	//if the next fragment-pair is consecutive in residue
                break;		//a gap in the sequence breaks the fragment alignment.
			}
			alignment.insert(alignment.len(),alignment.get(1,alignment.len())+1,alignment.get(2,alignment.len())+1);
			made_change = 1;
		}
	}
    
    return made_change;
}

void ensure_res_corresp(Align &alignment, Frag &f1, Frag &f2, Array2D<double> &FDM, vector<res_corresp> &orig1, vector<res_corresp> &orig2)
{
	//erase clashes
	further_refine_erase(alignment, f1, f2, FDM);		//fixes best fragment-pairs, and erases clashes, iteratively
	
    //final refinement stages
    bool made_change = 1;
    while(made_change == 1){
		while(made_change == 1){
            made_change = further_refine_point(alignment, f1, f2, FDM);
		}
        made_change = further_refine_extend(alignment, f1, f2, FDM);
    }
    
    //error checking - confirm one-to-one residue correspondence has been maintained
    made_change = further_refine_erase(alignment, f1, f2, FDM);
    if(made_change==1){
        cout << endl << endl << "Error: 1-to-1 residue correspondence not maintained - please report this case to the author." 
        << endl << endl << "PROSMART ALIGN terminated." << endl << endl;
		exit(-1);
    }
    
    return;
}

Array2D<double> fragDM_CA(Frag &f1, Frag &f2, Array1D<Coords> &frag_norm1, Array1D<Coords> &frag_norm2, vector<double> &trcov1, vector<double> &trcov2)
// this function uses rotation matrix corresponding to the superposition of all mainchain atoms,
// but scores using only the C-alpha atoms.
{
    unsigned int N1 = f1.size();
    unsigned int N2 = f2.size();
    Array2D<double> DM(N1,N2);
    Array2D<double> tempcov(3,3);	//temp variable to store covariance matrix.
    Array2D<double> RM(3,3);		//temp variable to store rotation matrix.
    Array1D<Coords> fragca_norm1(N1);
    Array1D<Coords> fragca_norm2(N2);
    double sq_dist;
    int coordlen = 1;		// number of coords per fragment
    
    for(unsigned int i=0; i<N1; i++){
        frag_norm1[i] = f1.crds_norm(i);
        fragca_norm1[i] = f1.ca_crds_norm(i);
        trcov1.push_back(fragca_norm1[i].tr_cov());
    }
    for(unsigned int i=0; i<N2; i++){
        frag_norm2[i] = f2.crds_norm(i);
        fragca_norm2[i] = f2.ca_crds_norm(i);
        trcov2.push_back(fragca_norm2[i].tr_cov());
    }
    
    if(N1>0){
        coordlen = fragca_norm1[0].size();
    }
    for(unsigned int i=0; i<N1; i++){		//compare fragment i from protein 1 with fragment j from protein 2.
        for(unsigned int j=0; j<N2; j++){
            RM = rotateM(frag_norm1[i],frag_norm2[j]);	//3x3 rotation matrix
            cov(tempcov,fragca_norm2[j],fragca_norm1[i]); //3x3 covariance matrix
            sq_dist = ( trcov1[i] + trcov2[j] - 2*tr_cov(tempcov,RM) )/coordlen;
            if(sq_dist < 0.00001){	//this ensures that identical fragments are identified as identical.
                DM[i][j] = 0.0;			//sometimes rounding error causes d<0 when fragments are exactly the same!
            } else {
                DM[i][j] = sqrt(sq_dist);	//modified procrustes distance, normalised by no. of coords
            }
        }
    }
    
    return DM;
}

Array2D<double> fragDM_all(Frag &f1, Frag &f2, Array1D<Coords> &frag_norm1, Array1D<Coords> &frag_norm2, vector<double> &trcov1, vector<double> &trcov2)
{
    int N1 = f1.size();
    int N2 = f2.size();
    Array2D<double> DM(N1,N2);
    Array2D<double> tempcov(3,3);	//temp variable to store covariance matrix.
    double sq_dist;
    Array1D<double> singval;
    
    int coordlen = 1;		// number of coords per fragment
    
    
    for(int i=0; i<N1; i++){
		frag_norm1[i] = f1.crds_norm(i);
		trcov1.push_back(frag_norm1[i].tr_cov());
    }
    for(int i=0; i<N2; i++){
		frag_norm2[i] = f2.crds_norm(i);
		trcov2.push_back(frag_norm2[i].tr_cov());
    }
    
    if(N1>0){
        coordlen = frag_norm1[0].size();
    }
    
    for(int i=0; i<N1; i++){		//compare fragment i from protein 1 with fragment j from protein 2.
        for(int j=0; j<N2; j++){
			cov(tempcov,frag_norm1[i],frag_norm2[j]); //3x3 covariance matrix
			singval = SV(tempcov);
			sq_dist = ( trcov1[i] + trcov2[j] - 2*sum(singval) )/coordlen;	//normalised wrt no. of coords
			if(sq_dist < 0.00001){	//this ensures that identical fragments are identified as identical.
				DM[i][j] = 0.0;			//sometimes rounding error causes d<0 when fragments are exactly the same!
			} else {
				DM[i][j] = sqrt(sq_dist);	//Procrustes distance
			}
		}
    }
	
    return DM;
}

Array1D<double> eigenvalues(Coords &frag_norm)
{
	Array2D<double> tempcov(3,3);
	
	covariance(tempcov,frag_norm,frag_norm);	//this is the true (scaled) covariance

	Eigenvalue<double> tempEIG(tempcov);
	Array1D<double> eigen;
	tempEIG.getRealEigenvalues(eigen);
	
	Array1D<double> eigen_sort(3);
	
	int x = 0;
	for(unsigned int i=1; i<=2; i++){
		if(eigen[i] >= eigen[x]){x = i;}
	}
	eigen_sort[0] = eigen[x];
	eigen[x] = -1.0;
	x = 0;
	for(unsigned int i=1; i<=2; i++){
		if(eigen[i] >= eigen[x]){x = i;}
	}
	eigen_sort[1] = eigen[x];
	eigen[x] = -1.0;
	x = 0;
	for(unsigned int i=1; i<=2; i++){
		if(eigen[i] >= eigen[x]){x = i;}
	}
	eigen_sort[2] = eigen[x];
	eigen[x] = -1.0;
	
	
	//cout << endl;
	//cout << eigen[0] << " " << eigen[1] << " " << eigen[2] << "  ";
	//cout << tempcov[0][0] << " " << tempcov[1][1] << " " << tempcov[2][2] << "  ";
	
	return eigen_sort;
}

Array2D<double> fragDM_all(Frag &f1, Frag &f2)
{
    int N1 = f1.size();
    int N2 = f2.size();
    Array2D<double> DM(N1,N2);
    Array2D<double> tempcov(3,3);	//temp variable to store covariance matrix.
    Array1D<Coords> frag_norm1(N1);
    Array1D<Coords> frag_norm2(N2);
    vector<double> trcov1;		//trace of covariance matrix.
    vector<double> trcov2;
    double sq_dist;
    Array1D<double> singval;
    
    int coordlen = 1;		// number of coords per fragment
	
    
    for(int i=0; i<N1; i++){
		frag_norm1[i] = f1.crds_norm(i);
		trcov1.push_back(frag_norm1[i].tr_cov());
    }
    for(int i=0; i<N2; i++){
		frag_norm2[i] = f2.crds_norm(i);
		trcov2.push_back(frag_norm2[i].tr_cov());
    }
    
    if(N1>0){
        coordlen = frag_norm1[0].size();
    }
    
    for(int i=0; i<N1; i++){		//compare fragment i from protein 1 with fragment j from protein 2.
        for(int j=0; j<N2; j++){	  
			cov(tempcov,frag_norm1[i],frag_norm2[j]); //3x3 covariance matrix
			singval = SV(tempcov);
			sq_dist = ( trcov1[i] + trcov2[j] - 2*sum(singval) )/coordlen;	//normalised wrt no. of coords
			if(sq_dist < 0.00001){	//this ensures that identical fragments are identified as identical.
				DM[i][j] = 0.0;			//sometimes rounding error causes d<0 when fragments are exactly the same!
			} else {
				DM[i][j] = sqrt(sq_dist);	//procrustes distance
			}
		}
    }
	
    return DM;
}


void output_moments(Align &alignment, Frag &f1, Frag &f2, Array2D<double> &DM)
{
	int N1 = f1.size();
    int N2 = f2.size();
	Array1D<Coords> frag_norm1(N1);
    Array1D<Coords> frag_norm2(N2);
	string fileout = "Prosmart_Output/moments";
	string filename = "";
	char *PROSMART_DIR;
	PROSMART_DIR = getenv("PROSMART_DIR");
	string library = PROSMART_DIR;
	library += "Prosmart_Library/";
	
	PDBfile pdb_helix;
	pdb_helix.readPDB(library+"helix.pdb", 'A');
	pdb_helix.trim(f1.len());
	Residues res_helix;
	res_helix.create(pdb_helix);
	Frag f_helix;
	f_helix.create(res_helix,f1.len(),0);
	Array2D<double> HDM1_helix = fragDM_all(f1,f_helix);
	Array2D<double> HDM2_helix = fragDM_all(f2,f_helix);
	
#ifdef FILTER_HELIX
#define FILTER_SSE
	filename = "helix";
	fileout += "_helix";
	double CUTOFF = 1.5;
#endif
#ifdef FILTER_STRAND
#define FILTER_SSE
	filename = "strand";
	fileout += "_strand";
	double CUTOFF = 2.0;
#endif
	
#ifdef FILTER_SSE
	PDBfile pdb_sse;
	pdb_sse.readPDB(library+filename+".pdb", 'A');
	pdb_sse.trim(f1.len());
	Residues res_sse(pdb_sse);
	Frag f_sse(res_sse,f1.len(),0);
	Array2D<double> HDM1 = fragDM_all(f1,f_sse);
	Array2D<double> HDM2 = fragDM_all(f2,f_sse);
#endif
	
	
#ifdef REMOVE_HELIX
	double HELIX_CUTOFF = 2.0;
	fileout += "_RMhelix";
#endif
#ifdef REMOVE_STRAND
	double STRAND_CUTOFF = 2.0;
	fileout += "_RMstrand";
	PDBfile pdb_strand;
	pdb_strand.readPDB(library+"strand.pdb", 'A');
	pdb_strand.trim(f1.len());
	Residues res_strand(pdb_strand);
	Frag f_strand(res_strand,f1.len(),0);
	Array2D<double> HDM1_strand = fragDM_all(f1,f_strand);
	Array2D<double> HDM2_strand = fragDM_all(f2,f_strand);
#endif
	
	
#ifdef REMOVE_ALIGN
	bool VALID=0;
	fileout += "_RMalign";
#endif
	
	vector<vector<double> > M1;
	vector<vector<double> > M2;
    
#ifdef FILTER_OUTPUT
	double cent_dist=0.0;
#endif
	
	fileout += ".txt";
	
	for(int i=0; i<N1; i++){
		frag_norm1[i] = f1.crds_norm(i);
		M1.push_back(frag_norm1[i].moments(NO_MOMENTS));
	}
	for(int i=0; i<N2; i++){
		frag_norm2[i] = f2.crds_norm(i);
		M2.push_back(frag_norm2[i].moments(NO_MOMENTS));
	}
    
    ofstream outfile;
    outfile.open(fileout.c_str());
    if(outfile.is_open()){
        for(int i=0; i<N1; i++){
#ifdef FILTER_SSE
            if(HDM1[i][0] > CUTOFF){continue;}
#endif
#ifdef REMOVE_HELIX
            if(HDM1_helix[i][0] < HELIX_CUTOFF){continue;}
#endif
#ifdef REMOVE_STRAND
            if(HDM1_strand[i][0] < STRAND_CUTOFF){continue;}
#endif
            for(int j=0; j<N2; j++){
#ifdef FILTER_SSE
                if(HDM2[j][0] > CUTOFF){continue;}
#endif
#ifdef REMOVE_HELIX
                if(HDM2_helix[j][0] < HELIX_CUTOFF){continue;}
#endif
#ifdef REMOVE_STRAND
                if(HDM2_strand[j][0] < STRAND_CUTOFF){continue;}
#endif
#ifdef FILTER_OUTPUT
				//only output fragment-pairs that have a similar centroid distance (within 1A)
				cent_dist = M1[i][0]-M2[j][0];
				if(cent_dist > 0.5 || cent_dist < -0.5){
					continue;
				}
#endif
#ifdef REMOVE_ALIGN
				VALID = 1;
				for(unsigned int k=1; k<=alignment.len(); k++){
                    if( alignment.get(1,k)==i && alignment.get(2,k)==j){
						VALID = 0;
						break;
					}
				}
				if(VALID==0){
					continue;
				}
#endif
                outfile << DM[i][j];
				for(int k=0; k<NO_MOMENTS; k++){
					outfile << "  " << M1[i][k] << " " << M2[j][k];
				}
				outfile << "  " << HDM1_helix[i][0] << " " << HDM2_helix[j][0] << endl;
			}
        }
		outfile.close();
		
    } else {
        cout << "Unable to open output file " << fileout << " for writing" << endl;
    }	
	
	return;
}

void output_align_moments(Align &alignment, Frag &f1, Frag &f2, Array2D<double> &DM)
{
	int N = alignment.len();
	Array1D<Coords> frag_norm1(N);
    Array1D<Coords> frag_norm2(N);
	vector<vector<double> > M1;
	vector<vector<double> > M2;
	string fileout = "Prosmart_Output/align_moments";
	
	char *PROSMART_DIR;
	PROSMART_DIR = getenv("PROSMART_DIR");
	string library = PROSMART_DIR;
	library += "Prosmart_Library/";
	PDBfile pdb_helix;
	pdb_helix.readPDB(library+"helix.pdb", 'A');
	pdb_helix.trim(f1.len());
	Residues res_helix;
	res_helix.create(pdb_helix);
	Frag f_helix;
	f_helix.create(res_helix,f1.len(),0);
	Array2D<double> HDM1_helix = fragDM_all(f1,f_helix);
	Array2D<double> HDM2_helix = fragDM_all(f2,f_helix);
	
	for(int i=0; i<N; i++){
		frag_norm1[i] = f1.crds_norm(alignment.get(1,i+1));
		frag_norm2[i] = f2.crds_norm(alignment.get(2,i+1));
		M1.push_back(frag_norm1[i].moments(NO_MOMENTS));
		M2.push_back(frag_norm2[i].moments(NO_MOMENTS));
    }
	
#ifdef FILTER_HELIX
	double HELIX_CUTOFF = 1.5;
	fileout += "_helix";
#endif
#ifdef REMOVE_HELIX
	double HELIX_CUTOFF = 2.0;
	fileout += "_RMhelix";
#endif
	
	fileout += ".txt";
    
    ofstream outfile;
    outfile.open(fileout.c_str());
    if(outfile.is_open()){
        for(int i=0; i<N; i++){
#ifdef REMOVE_HELIX
			if(HDM1_helix[alignment.get(1,i+1)][0] < HELIX_CUTOFF){continue;}
			if(HDM2_helix[alignment.get(2,i+1)][0] < HELIX_CUTOFF){continue;}
#endif
#ifdef FILTER_HELIX
			if(HDM1_helix[alignment.get(1,i+1)][0] > HELIX_CUTOFF){continue;}
			if(HDM2_helix[alignment.get(2,i+1)][0] > HELIX_CUTOFF){continue;}
#endif
            outfile << DM[alignment.get(1,i+1)][alignment.get(2,i+1)];
            for(int k=0; k<NO_MOMENTS; k++){
                outfile << "  " << M1[i][k] << " " << M2[i][k];
            }
            outfile << "  " << i
            << "  " << HDM1_helix[alignment.get(1,i+1)][0] 
            << " " << HDM2_helix[alignment.get(2,i+1)][0] << endl;
        }
		outfile.close();
    } else {
        cout << "Unable to open output file " << fileout << " for writing" << endl;
    }
	
	return;
}

Align identicalAlign(Frag &f1, Frag &f2, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2)
{
    Align alignment;
    int N1 = f1.size();
    int N2 = f2.size();
    int j=0;
    
    for(int i=0; i<N1; i++){	//for each fragment in pdb1
        //cout << endl << i << " : " << f1.idx(i) << " " << original_res1[f1.idx(i)].res;
        if(j<N2){
            while(original_res2[f2.idx(j)].res < original_res1[f1.idx(i)].res){
                j++;
                if(j>=N2){
                    //cout << " *** gone over N2" << endl;
                    break;
                }
            }
            if(j<N2){
                //cout << " : " << f2.idx(j) << " " << original_res2[f2.idx(j)].res;
                if(original_res2[f2.idx(j)].res == original_res1[f1.idx(i)].res){	//have found (assumed) correspondence between two structures
                    //cout << " ***";
                    alignment.add(i,j);
                    j++;	//allows insertion codes to be taken into account (because of residue renumbering)
                }
            }
        }
    }
    
    return alignment;
}

Align dynamAlign(Array2D<double> &DM, Frag &f1, Frag &f2, double HELIX_CUTOFF, double PENALTY)
{
    Align alignment;
    Align final;
    int N1 = DM.dim1();
    int N2 = DM.dim2();
    Array2D<double> F(N1,N2);
    int x1;
    int x2;
    Array1D<bool> PENALTY1(N1,(bool)0);
    Array1D<bool> PENALTY2(N2,(bool)0);
    
   if(!USE_DNARNA){
      PDBfile pdb_helix = get_helix(f1.len());
      Residues res_helix;
      res_helix.create(pdb_helix);
      Frag f_helix;
      f_helix.create(res_helix,f1.len(),0);
      Array2D<double> HDM1 = fragDM_all(f1,f_helix);
      Array2D<double> HDM2 = fragDM_all(f2,f_helix);
      
      
      for(int i=0; i<N1; i++){		//construct helix penalty vectors
         if(HDM1[i][0] <= HELIX_CUTOFF){
            PENALTY1[i] = 1;
         } else {
            PENALTY1[i] = 0;
         }
      }
      for(int i=0; i<N2; i++){
         if(HDM2[i][0] <= HELIX_CUTOFF){
            PENALTY2[i] = 1;
         } else {
            PENALTY2[i] = 0;
         }
      }
      for(int i=0; i<N1-1; i++){		//require consecutive helix fragments in order to penalise
         if(PENALTY1[i]==1){
            if(PENALTY1[i]==0){
               PENALTY1[i] = 0;
            }
         }
      }
      for(int i=0; i<N2-1; i++){
         if(PENALTY2[i]==1){
            if(PENALTY2[i]==0){
               PENALTY2[i] = 0;
            }
         }
      }
   }
    
    for(int i=1; i<N1; i++){		//fill matrix with zeroes.
        for(int j=1; j<N2; j++){
            F[i][j] = 0;
        }
    }
    F[0][0] = DM[0][0];
    for(int i=1; i<N1; i++){		//fill first row and column.
        F[i][0] = F[i-1][0] + DM[i][0];
    }
    for(int j=1; j<N2; j++){
        F[0][j] = F[0][j-1] + DM[0][j];
    }
    
    double UP;
    double LEFT;
    double DIAG;
    for(int i=1; i<N1; i++){		//fill rest of matrix.
        for(int j=1; j<N2; j++){
            UP = F[i][j-1];
            LEFT = F[i-1][j];
            if(PENALTY1[i]==1 && PENALTY2[j]==1){		//if consecutive helix fragments found in both proteins, then penalise non-consecutive aligning
                UP += PENALTY;
                LEFT += PENALTY;
            }
            DIAG = F[i-1][j-1];
            F[i][j] = min(min(DIAG,LEFT),UP) + DM[i][j];
        }
    }
    
    //string tmp_out = "arr.txt";
    //fileoutput_array(F,tmp_out,"_F",0);
    
    x1 = N1-1;
    x2 = N2-1;
    alignment.add(x1,x2);
    while(x1>0 && x2>0){	//find best path through matrix
        if(F[x1-1][x2-1]<=F[x1][x2-1] && F[x1-1][x2-1]<=F[x1-1][x2]){
            x1 -= 1;
            x2 -= 1;
            alignment.add(x1,x2);
        } else if(F[x1][x2-1] < F[x1-1][x2]){
            x2 -= 1;
            alignment.add(x1,x2);
        } else {
            x1 -= 1;
            alignment.add(x1,x2);
        }
    }
    
    if(x1!=0 || x2!=0){
        if(x1==0){			//fill out the end (actually the start!) of the path
            for(int i=x2; i>0; i--){
                alignment.add(0,i);
            }
        } else if(x2==0){
            for(int i=x1-1; i>=0; i--){
                alignment.add(i,0);
            }
        }
    }
    
    /*
     //display alignment - this has a many-to-one fragment correspondence
     for(int i=1; i<=alignment.len(); i++){
     cout << endl << alignment.get(1,i) << " " << alignment.get(2,i) << " " << DM[alignment.get(1,i)][alignment.get(2,i)];
     }
     */
	
	/*ofstream outfile;
     outfile.open("align.txt");
     for(int i=1; i<=alignment.len(); i++){
     outfile << alignment.get(1,i) << " " << alignment.get(2,i) << endl;
     }
     outfile.close();
     exit(-1);*/
    
    //filter alignment to achieve 1-1 correspondence.
    int N;
    if(alignment.len()>0){
        final.add(alignment.get(1,alignment.len()),alignment.get(2,alignment.len()));
        for(int i=alignment.len()-1; i>0; i--){
            N = final.len();
            //cout << endl << alignment.get(1,i) << " " << alignment.get(2,i);
            //cout << "\t" << final.get(1,N) << " " << final.get(2,N);
            if(alignment.get(1,i)==final.get(1,N) || alignment.get(2,i)==final.get(2,N)){
                //cout << "\t" << DM[alignment.get(1,i)][alignment.get(2,i)] << " " << DM[final.get(1,N)][final.get(2,N)];
                if(DM[alignment.get(1,i)][alignment.get(2,i)] < DM[final.get(1,N)][final.get(2,N)]){
                    final.pop_back();
                    final.add(alignment.get(1,i),alignment.get(2,i));		//replace previous entry
                    //cout << " ***";
                }
            } else {
                final.add(alignment.get(1,i),alignment.get(2,i));		//add new entry
            }
        }
    }
    
    /*
     //display final alignment, before refinement
     for(int i=1; i<=final.len(); i++){
     cout << endl << final.get(1,i) << " " << final.get(2,i) << " " << DM[final.get(1,i)][final.get(2,i)];
     }
     */
    
    //refine alignment, to enhance well-matched regions
    
    return final;
}

Align align_refine(Align &final, Array2D<double> &DM)
{
    vector<int> start1;
    vector<int> start2;
    vector<int> len;
    int n;
    Align newalign;
    bool EXTRA_REFINEMENT = 1;
    
    //get fragment segments
    start1.push_back(final.get(1,1));
    start2.push_back(final.get(2,1));
    n=0;
    for(unsigned int i=1; i<final.len(); i++){
        //cout << endl << i << " " << final.get(1,i) << " " << final.get(2,i);
        n++;
        if(final.get(1,i)+1 != final.get(1,i+1) || final.get(2,i)+1 != final.get(2,i+1)){
            len.push_back(n);
            //cout << " * " << start1[start1.size()-1] << " " << start2[start2.size()-1] << " " << len[len.size()-1];
            n=0;
            start1.push_back(final.get(1,i+1));
            start2.push_back(final.get(2,i+1));
        }
    }
    len.push_back(n+1);
    //cout << endl << final.len() << " " << final.get(1,final.len()) << " " << final.get(2,final.len());
    //cout << " * " << start1[start1.size()-1] << " " << start2[start2.size()-1] << " " << len[len.size()-1];
    /*for(unsigned int i=0; i<start1.size(); i++){
     cout << endl << start1[i] << " " << start2[i] << " " << len[i];
     }*/
    
    //refine alignment of segments and segment edges
    if(start1.size()>1){
        while(EXTRA_REFINEMENT==1){
            EXTRA_REFINEMENT = align_refine_segment(DM,start1,start2,len);
            if(EXTRA_REFINEMENT==1){continue;}
            EXTRA_REFINEMENT = align_refine_point(DM,start1,start2,len);
            if(EXTRA_REFINEMENT==0){
                EXTRA_REFINEMENT = align_refine_max(DM,start1,start2,len);
            }
        }
    }
    
    //check alignment validity - check still 1-to-1
    for(unsigned int i=0; i<start1.size()-1; i++){
        //cout << endl << start1[i] << " " << start2[i] << "\t" << len[i];
        //cout << "\t" << start1[i]+len[i] << " " << start2[i]+len[i];
        //cout << "\t" << start1[i+1]-start1[i]-len[i] << " " << start2[i+1]-start2[i]-len[i];
        if(start1[i+1]-start1[i]-len[i]<0 || start2[i+1]-start2[i]-len[i]<0){
            cout << endl << endl << "Critical Warning: 1-to-1 fragment correspondence not maintained - please report this case to the author." << endl << endl;
        } else if(start1[i+1]-start1[i]-len[i]!=0 && start2[i+1]-start2[i]-len[i]!=0){
            cout << endl << endl << "Warning: maximal alignment not achieved - please report this case to the author." << endl << endl;
        }
    }
    
    //convert back to Align format
    for(unsigned int i=0; i<start1.size(); i++){
        //cout << endl << start1[i] << " " << start2[i] << "\t" << len[i];
        for(int j=0; j<len[i]; j++){
            newalign.add(start1[i]+j,start2[i]+j);
        }
    }
    
    return newalign;
}

bool align_refine_point(Array2D<double> &DM, vector<int> &start1, vector<int> &start2, vector<int> &len)
//attempt to move edge-fragments of consecutively aligned fragment segments to the adjacent segment, if this scores better.
{
    bool MADE_CHANGE = 0;
    
    /*for(unsigned int i=0; i<start1.size()-1; i++){
     cout << endl << start1[i] << " " << start2[i] << "\t" << len[i];
     }*/
    
    for(unsigned int i=0; i<start1.size()-1; i++){
        //cout << endl << start1[i] << " " << start2[i] << "\t" << len[i];
        //cout << endl << "\tstay i+1: \t" <<  DM[start1[i+1]][start2[i+1]];
        //cout << endl << "\tlengthen lower:\t" <<  DM[start1[i]+len[i]][start2[i]+len[i]];
        //cout << endl << "\tstay i: \t" <<  DM[start1[i]+len[i]-1][start2[i]+len[i]-1];
        //cout << endl << "\tlengthen upper:\t" <<  DM[start1[i+1]-1][start2[i+1]-1];
        if(DM[start1[i]+len[i]][start2[i]+len[i]] < DM[start1[i+1]-1][start2[i+1]-1]){	//lengthen lower more preferable than upper
            if(DM[start1[i]+len[i]][start2[i]+len[i]] < DM[start1[i+1]][start2[i+1]]){	//lengthen lower
                //cout << " *";
                len[i]++;
                if(len[i+1]==1){
                    start1.erase(start1.begin()+i+1);
                    start2.erase(start2.begin()+i+1);
                    len.erase(len.begin()+i+1);
                } else {
                    start1[i+1]++;
                    start2[i+1]++;
                    len[i+1]--;
                }
                MADE_CHANGE=1;
            } else if(DM[start1[i+1]-1][start2[i+1]-1] < DM[start1[i]+len[i]-1][start2[i]+len[i]-1]){	//lengthen upper
                //cout << " **";
                len[i+1]++;
                start1[i+1]--;
                start2[i+1]--;
                if(len[i]==1){
                    start1.erase(start1.begin()+i);
                    start2.erase(start2.begin()+i);
                    len.erase(len.begin()+i);
                } else {
                    len[i]--;
                }
                MADE_CHANGE=1;
            }
        } else {
            if(DM[start1[i+1]-1][start2[i+1]-1] < DM[start1[i]+len[i]-1][start2[i]+len[i]-1]){	//lengthen upper
                //cout << " ***";
                len[i+1]++;
                start1[i+1]--;
                start2[i+1]--;
                if(len[i]==1){
                    start1.erase(start1.begin()+i);
                    start2.erase(start2.begin()+i);
                    len.erase(len.begin()+i);
                } else {
                    len[i]--;
                }
                MADE_CHANGE=1;
            } else if(DM[start1[i]+len[i]][start2[i]+len[i]] < DM[start1[i+1]][start2[i+1]]){	//lengthen lower
                //cout << " ****";
                len[i]++;
                if(len[i+1]==0){
                    start1.erase(start1.begin()+i+1);
                    start2.erase(start2.begin()+i+1);
                    len.erase(len.begin()+i+1);
                } else {
                    start1[i+1]++;
                    start2[i+1]++;
                    len[i+1]--;
                }
                MADE_CHANGE=1;
            }
        }
    }
    
    /*for(unsigned int i=0; i<start1.size()-1; i++){
     cout << endl << start1[i] << " " << start2[i] << "\t" << len[i];
     }*/
    
    return MADE_CHANGE;
}

bool align_refine_segment(Array2D<double> &DM, vector<int> &start1, vector<int> &start2, vector<int> &len)
//attempt to find better alignment for segments of consecutively aligned fragments
{
    int n;
    double fs;
    double best_fs;
    int best_j;
    int best_k;
    bool MADE_CHANGE = 0;
    int lower1 = start1[0];
    int lower2 = start2[0];
    int upper1 = start1[0];
    int upper2 = start2[0];
#ifdef ALIGN_REFINE_COUTS
    cout << endl << "align_refine_segment" << endl;
    /*int tmp;
    cin >> tmp;*/
    cout << endl << "number of segments: " << start1.size() << endl;
#endif
    
    //limited by (0,0) for first segment
#ifdef ALIGN_REFINE_COUTS
    cout << endl << start1[0] << " " << start2[0] << "\t" << len[0];
#endif
    fs = frag_seq_score(start1[0],start2[0],len[0],DM);
#ifdef ALIGN_REFINE_COUTS
    cout << "\t" << fs << "\t";
#endif
    best_fs = fs;
    best_j = start1[0];
    best_k = start2[0];
    if(start1.size()>1){
        lower1 = start1[1]-len[0];
        lower2 = start2[1]-len[0];
    }
    for(int j=0; j<=lower1; j++){
        for(int k=0; k<=lower2; k++){
            fs = frag_seq_score(j,k,len[0],DM);
#ifdef ALIGN_REFINE_COUTS
            cout << endl << "\t" << j << " " << k << "\t\t" << fs;
#endif
            if(fs < best_fs){
                best_fs = fs;
                best_j = j;
                best_k = k;
#ifdef ALIGN_REFINE_COUTS
                cout << " *";
#endif
            }
        }
    }
    if(best_j != start1[0]){
#ifdef ALIGN_REFINE_COUTS
        cout << endl << "\tbest_j: " << best_j;
#endif
        start1[0] = best_j;
        MADE_CHANGE = 1;
    }
    if(best_k != start2[0]){
#ifdef ALIGN_REFINE_COUTS
        cout << endl << "\tbest_k: " << best_k;
#endif
        start2[0] = best_k;
        MADE_CHANGE = 1;
    }
    //search potential translations
    for(unsigned int i=1; i<start1.size()-1; i++){
#ifdef ALIGN_REFINE_COUTS
        cout << endl << start1[i] << " " << start2[i] << "\t" << len[i];
#endif
        fs = frag_seq_score(start1[i],start2[i],len[i],DM);
#ifdef ALIGN_REFINE_COUTS
        cout << "\t" << fs << "\t";
#endif
        best_fs = fs;
        best_j = start1[i];
        best_k = start2[i];
        for(int j=start1[i-1]+len[i-1]; j<=start1[i+1]-len[i]; j++){
            for(int k=start2[i-1]+len[i-1]; k<=start2[i+1]-len[i]; k++){
                fs = frag_seq_score(j,k,len[i],DM);
#ifdef ALIGN_REFINE_COUTS
                cout << endl << "\t" << j << " " << k << "\t\t" << fs;
#endif
                if(fs < best_fs){
                    best_fs = fs;
                    best_j = j;
                    best_k = k;
#ifdef ALIGN_REFINE_COUTS
                    cout << " *";
#endif
                }
            }
        }
        if(best_j != start1[i]){
#ifdef ALIGN_REFINE_COUTS
            cout << endl << "\tbest_j: " << best_j;
#endif
            start1[i] = best_j;
            MADE_CHANGE = 1;
        }
        if(best_k != start2[i]){
#ifdef ALIGN_REFINE_COUTS
            cout << endl << "\tbest_k: " << best_k;
#endif
            start2[i] = best_k;
            MADE_CHANGE = 1;
        }
    }
    //limited by end for last segment
    n=start1.size()-1;
#ifdef ALIGN_REFINE_COUTS
    cout << endl << start1[n] << " " << start2[n] << "\t" << len[n];
#endif
    fs = frag_seq_score(start1[n],start2[n],len[n],DM);
#ifdef ALIGN_REFINE_COUTS
    cout << "\t" << fs << "\t";
#endif
    best_fs = fs;
    best_j = start1[n];
    best_k = start2[n];
    if(n>0){
        upper1 = start1[n-1]+len[n-1];
        upper2 = start2[n-1]+len[n-1];
    }
    for(int j=upper1; j<=DM.dim1()-len[n]; j++){
        for(int k=upper2; k<=DM.dim2()-len[n]; k++){
            fs = frag_seq_score(j,k,len[n],DM);
#ifdef ALIGN_REFINE_COUTS
            cout << endl << "\t" << j << " " << k << "\t\t" << fs;
#endif
            if(fs < best_fs){
                best_fs = fs;
                best_j = j;
                best_k = k;
#ifdef ALIGN_REFINE_COUTS
                cout << " *";
#endif
            }
        }
    }
    if(best_j != start1[n]){
#ifdef ALIGN_REFINE_COUTS
        cout << endl << "\tbest_j: " << best_j;
#endif
        start1[n] = best_j;
        MADE_CHANGE = 1;
    }
    if(best_k != start2[n]){
#ifdef ALIGN_REFINE_COUTS
        cout << endl << "\tbest_k: " << best_k;
#endif
        start2[n] = best_k;
        MADE_CHANGE = 1;
    }
    
    //splice segments if appropriate
    if(MADE_CHANGE==1){
        for(unsigned int i=0; i<start1.size()-1; i++){
            if(start1[i]+len[i]==start1[i+1] && start2[i]+len[i]==start2[i+1]){
                len[i] += len[i+1];
                start1.erase(start1.begin()+i+1);
                start2.erase(start2.begin()+i+1);
                len.erase(len.begin()+i+1);
                i--;
            }
        }
    }
    
    return MADE_CHANGE;
}

bool align_refine_max(Array2D<double> &DM, vector<int> &start1, vector<int> &start2, vector<int> &len)
//maximise alignment length - this will only add one fragment. Function should be called multiple times in order to add multiple fragments.
{
    bool MADE_CHANGE = 0;
    int N = start1.size()-1;
    
    //maximise between segments
    for(int i=0; i<N; i++){
        //cout << endl << start1[i] << " " << start2[i] << "\t" << len[i];
        //cout << "\t" << start1[i]+len[i] << " " << start2[i]+len[i];
        //cout << "\t" << start1[i+1]-start1[i]-len[i] << " " << start2[i+1]-start2[i]-len[i];
        if((start1[i+1] > start1[i]+len[i]) && (start2[i+1] > start2[i]+len[i])){	//alignment can be made longer.
            if(DM[start1[i]+len[i]][start2[i]+len[i]] < DM[start1[i+1]-1][start2[i+1]-1]){
                len[i]++;
            } else {
                start1[i+1]--;
                start2[i+1]--;
                len[i+1]++;
            }
            MADE_CHANGE = 1;
        }
    }
    
    //maximise start
    if(start1[0]>0 && start2[0]>0){
        MADE_CHANGE = 1;
        start1[0] --;
        start2[0] --;
        len[0] ++;
    }
    //maximise end
    if((start1[N]+len[N] < DM.dim1()) && (start2[N]+len[N] < DM.dim2())){
        MADE_CHANGE = 1;
        len[N] ++;
    }
    
    //splice compatible segments
    if(MADE_CHANGE==1){
        for(int i=0; i<N; i++){
            if(start1[i]+len[i]==start1[i+1] && start2[i]+len[i]==start2[i+1]){
                len[i] += len[i+1];
                start1.erase(start1.begin()+i+1);
                start2.erase(start2.begin()+i+1);
                len.erase(len.begin()+i+1);
                i--;
            }
        }
    }
    
    return MADE_CHANGE;
}

double frag_seq_score(int i1, int j1, int N, Array2D<double> &FDM)
{
    double result = 0.0;
    
    for(int i=0; i<N; i++){
        result += FDM[i1+i][j1+i];
    }
    
    return result;
}

Align align_residues(Frag &f1, Frag &f2, Align &alignment, Array2D<double> &FDM, unsigned short fraglen)
{
    Array2D<int> align = alignment.get();
    int N = FDM.dim1();
    int M = align.dim1();
    Align new_alignment;
    Align temp_alignment;
    
    //get fragment alignment
    Array1D<int> frag2(N,-1);
    for(int i=0; i<M; i++){
        frag2[align[i][0]] = align[i][1];	//frag2[i] in frag2 corresponds to i in frag1
    }
    
    //get fragment distances
    Array1D<double> fragdist(N,-1.0);
    Array1D<double> fragdist_temp(N);
    for(int i=0; i<N; i++){
        if(frag2[i] > -1){
            fragdist[i] = FDM[i][frag2[i]];	//fragdist remains constant after this assignment
        }
        fragdist_temp[i] = fragdist[i];			//fragdist_temp changes during the algorithm
    }
    
    //establish new fragment 2 array
    Array1D<int> new_frag2(N,-1);
    
    //establish new residue alignment arrays
    Array1D<int> res1(f1.res_idx(f1.res_size()-1)+1,-1);	//note - residues exist in this array even if they do not exist in the pdb file.
    Array1D<int> res2(f2.res_idx(f2.res_size()-1)+1,-1);
    
    //align residues by finding lowest fragment distances
    double best_dist;
    int best_frag = 0;
    bool valid;
    
    while(best_frag >= 0){
        best_frag = -1;
        for(int i=0; i<N; i++){				//initialise best_dist and best_frag
            if(fragdist_temp[i]>=0){
                best_dist = fragdist_temp[i];
                best_frag = i;
                break;
            }
        }
        if(best_frag<0){break;}
        
        for(int i=best_frag+1; i<N; i++){	//find best_dist and best_frag
            if(fragdist_temp[i]>=0 && fragdist_temp[i]<best_dist){
                best_dist = fragdist_temp[i];
                best_frag = i;
            }
        }
        
        if(best_frag>=0){					//try to align best_frag
            fragdist_temp[best_frag] = -1;	//indicates best_frag has been considered/aligned
            valid = 1; 
            for(int j=0; j<fraglen; j++){
                if(res1[f1.idx(best_frag)+j] != -1){
                    if(res1[f1.idx(best_frag)+j] != f2.idx(frag2[best_frag])+j){
                        valid = 0;
                        break;
                    }
                }
                if(res2[f2.idx(frag2[best_frag])+j] != -1){
                    if(res2[f2.idx(frag2[best_frag])+j] != f1.idx(best_frag)+j){
                        valid = 0;
                        break;
                    }
                }
            }
            if(valid == 1){					//indicates best_frag does not conflict, so can be aligned
                new_frag2[best_frag] = frag2[best_frag];
                for(int j=0; j<fraglen; j++){
                    res1[f1.idx(best_frag)+j] = f2.idx(frag2[best_frag])+j;
                    res2[f2.idx(frag2[best_frag])+j] = f1.idx(best_frag)+j;
                }
            }
        }
    }
#ifdef DISPLAY_ALIGN  
    cout << endl << "Frag1\tFrag2\tRes1\tRes2\tNewF2\tNewR1\tNewR2\tFragDist";
    for(int i=0; i<N; i++){
        cout << endl << i << "\t";
        if(frag2[i]==-1) cout << "\t";
        else cout << frag2[i] << "\t";
        cout << f1.idx(i) << "\t";
        if(frag2[i]==-1) cout << "\t";
        else cout << f2.idx(frag2[i]) << "\t";
        if(new_frag2[i]==-1) cout << "\t";
        else cout << new_frag2[i] << "\t";
        if(res1[f1.idx(i)]==-1) cout << "\t";
        else cout << res1[f1.idx(i)] << "\t";
        if(frag2[i]==-1) cout << "\t";
        else if(res2[f2.idx(frag2[i])]==-1) cout << "\t";
        else cout << res2[f2.idx(frag2[i])] << "\t";
        cout << fragdist[i];
    }
    cout << endl << endl;
#endif
    
    for(int i=0; i<N; i++){
        if(new_frag2[i] >=0){
            new_alignment.add(i,new_frag2[i]);
        }
    }
    
    return new_alignment;
}

Align refine_alignment(Frag &f1, Frag &f2, Align &old_align, Array2D<double> &FDM, unsigned short fragLen)
{
    Align new_align = old_align;
    int i1;
    int i2;
    int i1_;
    int i2_;
    bool MOVE_START;
    bool MOVE_END;
    
#ifdef DISPLAY_ALIGN_REFINEMENT
    cout << endl << "f1\tf2\tr1\tr2\tdist";
#endif
    for(unsigned int i=0; i<new_align.len()-1; i++){
        i1 = new_align.get(1,i+1);
        i2 = new_align.get(2,i+1);
        i1_ = new_align.get(1,i+2);
        i2_ = new_align.get(2,i+2);
#ifdef DISPLAY_ALIGN_REFINEMENT
        cout << endl << i1 << "\t" << i2;
        cout << "\t" << f1.idx(i1) << "\t" << f2.idx(i2);
        cout << "\t" << FDM[i1][i2];
#endif
        if((f1.idx(i1)+fragLen == f1.idx(i1_)) || (f2.idx(i2)+fragLen == f2.idx(i2_))){	//extension would cause a conflict - refine alignment instead.
#ifdef DISPLAY_ALIGN_REFINEMENT
            cout << "  ***";
            cout << endl << "Refining alignment:";
            cout << endl << i1 << "\t" << i2;
            cout << "\t" << f1.idx(i1) << "\t" << f2.idx(i2);
            cout << "\t" << FDM[i1][i2] << "  : keep start";
            cout << endl << i1_-1 << "\t" << i2_-1;
            cout << "\t" << f1.idx(i1_-1) << "\t" << f2.idx(i2_-1);
            cout << "\t" << FDM[i1_-1][i2_-1] << "  : move start to end";
            cout << endl << i1_ << "\t" << i2_;
            cout << "\t" << f1.idx(i1_) << "\t" << f2.idx(i2_);
            cout << "\t" << FDM[i1_][i2_] << "  : keep end";
            cout << endl << i1+1 << "\t" << i2+1;
            cout << "\t" << f1.idx(i1+1) << "\t" << f2.idx(i2+1);
            cout << "\t" << FDM[i1+1][i2+1] << "  : move end to start";
            cout << endl;
#endif
            MOVE_START = 0;
            MOVE_END = 0;
            if(FDM[i1_-1][i2_-1] < FDM[i1][i2]){
                if((f1.idx(i1_-1)==f1.idx(i1_)-1)&&(f2.idx(i2_-1)==f2.idx(i2_)-1)){		//prospective residues must be consecutive in order to move!
                    MOVE_START = 1;
                }
            }
            if(FDM[i1+1][i2+1] < FDM[i1_][i2_]){
                if((f1.idx(i1)==f1.idx(i1+1)-1)&&(f2.idx(i2)==f2.idx(i2+1)-1)){		//prospective residues must be consecutive in order to move!
                    MOVE_END = 1;
                }
            }
            if((MOVE_START == 1) && (MOVE_END == 1)){
                if(FDM[i1_-1][i2_-1] < FDM[i1+1][i2+1]){
                    MOVE_END = 0;
                } else {
                    MOVE_START = 0;
                } 
            } 
            if(MOVE_START == 1){
                new_align.edit_entry(i,i1_-1,i2_-1);
                if(i>0){i-=2;}
                else{i--;}	//so that i<0 can never happen on the next interation of the loop
                continue;
            }
            if(MOVE_END == 1){
                new_align.edit_entry(i+1,i1+1,i2+1);
                continue;
            }
            continue;
        } else if(i1 != i1_-1){
            if(i2 != i2_-1){	//attempt to extend seperated alignment (until there is a conflict)
#ifdef DISPLAY_ALIGN_REFINEMENT
                cout << "  *" << endl << "Seperated fragments found:";
                cout << endl << i1+1 << "\t" << i2+1;
                cout << "\t" << f1.idx(i1+1) << "\t" << f2.idx(i2+1);
                cout << "\t" << FDM[i1+1][i2+1];
                cout << endl << i1_-1 << "\t" << i2_-1;
                cout << "\t" << f1.idx(i1_-1) << "\t" << f2.idx(i2_-1);
                cout << "\t" << FDM[i1_-1][i2_-1] << endl;
#endif
                if((f1.idx(i1+1) == f1.idx(i1)+1) && (f2.idx(i2+1) == f2.idx(i2)+1)){
                    if((f1.idx(i1_-1) == f1.idx(i1_)-1) && (f2.idx(i2_-1) == f2.idx(i2_)-1)){	//can extend segment at beginning or end (no residue clashes)
#ifdef DISPLAY_ALIGN_REFINEMENT
                        cout << "**";
#endif
                        if(FDM[i1+1][i2+1] < FDM[i1_-1][i2_-1]){
#ifdef DISPLAY_ALIGN_REFINEMENT
                            cout << "1";
#endif
                            new_align.insert(i+1,i1+1,i2+1);
                        } else {
#ifdef DISPLAY_ALIGN_REFINEMENT
                            cout << "2";
#endif
                            new_align.insert(i+1,i1_-1,i2_-1);	
                            i--;
                        }
                    } else {										//can only extend segment at beginning
#ifdef DISPLAY_ALIGN_REFINEMENT
                        cout << "*1";
#endif
                        new_align.insert(i+1,i1+1,i2+1);
                    }
                } else {										//can't extend segment at beginning
                    if((f1.idx(i1_-1) == f1.idx(i1_)-1) && (f2.idx(i2_-1) == f2.idx(i2_)-1)){	//can only extend segment at end
#ifdef DISPLAY_ALIGN_REFINEMENT
                        cout << "*2";
#endif
                        new_align.insert(i+1,i1_-1,i2_-1);
                        i--;
                    }
                }
                continue;
            }
        }
        
    }
    
    //now extend the first segment back as far as possible
    MOVE_START = 1;
    while(MOVE_START == 1){
        i1 = new_align.get(1,1);
        i2 = new_align.get(2,1);
        MOVE_START = 0;
        if(i1 > 0){
            if(i2 > 0){
                if(f1.idx(i1) == f1.idx(i1-1)+1){
                    if(f2.idx(i2) == f2.idx(i2-1)+1){
                        new_align.insert_start(i1-1,i2-1);
                        MOVE_START = 1;
                        continue;
                    }
                }
            }
        }
    }
    
    //now extend the last segment as far forward as possible
    int N1 = f1.size()-1;
    int N2 = f2.size()-1;
    MOVE_END = 1;
    while(MOVE_END == 1){
        i1 = new_align.last1();
        i2 = new_align.last2();
        MOVE_END = 0;
        if(i1 < N1){
            if(i2 < N2){
                if(f1.idx(i1) == f1.idx(i1+1)-1){
                    if(f2.idx(i2) == f2.idx(i2+1)-1){
                        new_align.add(i1+1,i2+1);
                        MOVE_END = 1;
                        continue;
                    }
                }
            }
        }
    }
    
#ifdef DISPLAY_ALIGN  
    cout << endl << "idx\tFrag1\tFrag2\tRes1\tRes2\tFragDist";
    for(int i=0; i<new_align.len(); i++){
        i1 = new_align.get(1,i+1);
        i2 = new_align.get(2,i+1);
        cout << endl << i << "\t";
        cout << i1 << "\t";
        cout << i2 << "\t";
        cout << f1.idx(i1) << "\t";
        cout << f2.idx(i2) << "\t";
        cout << FDM[i1][i2];
    }
    cout << endl << endl;
#endif
    
    return new_align;
}

Array2D<double> mod_FDM_by_sequence(Frag &f1, Frag &f2, Array2D<double> &FDM, bool NO_REWARD_SEQ, double &REWARD_SEQ)
{
    if(NO_REWARD_SEQ==0){
        Array2D<double> result(FDM.dim1(),FDM.dim2(),REWARD_SEQ);
        for(int i=0; i<FDM.dim1(); i++)
            for(int j=0; j<FDM.dim2(); j++)
                if(!compare_fragment_sequence(f1,f2,i,j))
                    result[i][j] = FDM[i][j];       //if fragments are sequence-identical, set score to REWARD_SEQ.
        return result;
    }
    return FDM;
}

bool compare_fragment_sequence(Frag &f1, Frag &f2, int &i, int &j)
{
#ifdef REQUIRE_ONLY_N_SEQID
    unsigned int ctr = 0;
    for(int k=0; k<f1.len(); k++)
        if(f1.get_resid_frag(i,k) == f2.get_resid_frag(j,k))
            ctr++;
    if(ctr >= 5)
        return 1;
    else
        return 0;
#endif
    
    for(int k=0; k<f1.len(); k++)
        if(f1.get_resid_frag(i,k) != f2.get_resid_frag(j,k))
            return 0;
    return 1;
}


