//
// This file is part of ProSMART.
//

#include "align_output_filehandling.h"

Transform output_alignment(vector<string> &info, PDBfile &pdb1, PDBfile &pdb2, vector<residue_alignment> &Res_align, vector<residue_alignment> &Res_align_sim, Frag &frag1, Frag &frag2, Residues &res1, Residues &res2, string filein1, string fileout1, char domain1, string filein2, string fileout2, char domain2, string workingdirectory, string filename1, string filename2, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2, double PYMOL_COLOR_RESOLUTION, double SIDECHAIN_COLOR_RESOLUTION, vector<int> &res_type1, vector<int> &res_type2, vector<coord> &library_colors, bool OUT_PDB, bool OUT_COLOR, bool PROC_NORM, double SUPERPOSE, bool DISPLAY_AS_COSINE_DISTANCE, bool OUT_PDB_FULL, bool USE_SIDE)
{ 
    string tempfileout1;
	string tempfileout2;
	string obj1 = get_filename(filename1) + "_" + domain1;
    string obj2 = get_filename(filename2) + "_" + domain2;
	ofstream outfile;
	Transform transformation;
    
	string ALIGN_dir = workingdirectory + "Residue_Alignment_Scores/" + obj1 + "/";
    //write alignment files
    tempfileout1 = ALIGN_dir + obj1 + "_" + obj2 + ".txt";
    write_alignment(res1,res2,info,Res_align,tempfileout1, original_res1, original_res2, DISPLAY_AS_COSINE_DISTANCE);
	
	if(OUT_COLOR==1){
        double DEGREES = 15;
        double ANGLE_RESOLUTION=1000*(1-cos(DEGREES*(atan(1.0)/45)));		//PI = 4*atan(1.0)
        //Factor of 1000 is used for numerical stability - this is also used in write_pymol_rot
        string PYMOL_dir = workingdirectory + "Colour_Scripts/" + obj1+"_"+obj2+"/";
        //write pymol color files
        
		//FORCE PARTICULAR COLOURS - for generating helix-based legend.
		/*for(unsigned int i=0; i<3; i++){
         Res_align[i].dist = -1.0;
         }
         for(unsigned int i=0; i<17-3; i++){
         Res_align[i+3].dist = i/3;
         }
         for(unsigned int i=0; i<Res_align.size(); i++){
         cout << Res_align[i].res1 << " " << Res_align[i].res1 << " " << Res_align[i].dist << " " << Res_align[i].dist2 << endl;
         }*/
		
        tempfileout1 = PYMOL_dir + "flexible.pml";
        write_pymol_minimum(info, Res_align, res1, res2, obj1, obj2, tempfileout1, domain1, domain2, PYMOL_COLOR_RESOLUTION, original_res1, original_res2);
        tempfileout1 = PYMOL_dir + "procrustes.pml";
        write_pymol_central(info, Res_align, res1, res2, obj1, obj2, tempfileout1, domain1, domain2, PYMOL_COLOR_RESOLUTION, original_res1, original_res2);
        tempfileout1 = PYMOL_dir + "hinging.pml";
        write_pymol_rot(info, Res_align, res1, res2, obj1, obj2, tempfileout1, domain1, domain2, ANGLE_RESOLUTION, original_res1, original_res2);
        tempfileout1 = PYMOL_dir + "sidechainRMSD.pml";
        write_pymol_sidechainRMSD(info, Res_align, res1, res2, obj1, obj2, tempfileout1, domain1, domain2, SIDECHAIN_COLOR_RESOLUTION, original_res1, original_res2);
        tempfileout1 = PYMOL_dir + "sidechainMAX.pml";
        write_pymol_sidechainMAX(info, Res_align, res1, res2, obj1, obj2, tempfileout1, domain1, domain2, SIDECHAIN_COLOR_RESOLUTION, original_res1, original_res2);
        tempfileout1 = PYMOL_dir + "sidechainAV.pml";
        write_pymol_sidechainAV(info, Res_align, res1, res2, obj1, obj2, tempfileout1, domain1, domain2, SIDECHAIN_COLOR_RESOLUTION, original_res1, original_res2);
        tempfileout1 = PYMOL_dir + "fragment.pml";
        write_pymol_sse(info, Res_align, res1, res2, obj1, obj2, tempfileout1, domain1, domain2, original_res1, original_res2, res_type1, res_type2, library_colors);
        
		tempfileout1 = workingdirectory + ".HTML_files/tmp/Colour/" + obj1 + "_" + obj2 + ".txt";
		outfile.open(tempfileout1.c_str());
		if(outfile.is_open()){
			outfile << "flexible" << endl;
			outfile << "procrustes" << endl;
			outfile << "hinging" << endl;
			outfile << "sidechainRMSD" << endl;
			outfile << "sidechainAV" << endl;
			outfile << "fragment" << endl;
			outfile.close();
		} else {
			cout << endl << "Cannot open " << tempfileout1 << " for writing." << endl;
		}
	}

	if(OUT_PDB==1){
		string PDBOUT_dir = workingdirectory + "Superposition/PDB_files/" + obj1 + "_" + obj2 + "/";
		//write pdb files corresponding to the full alignment
		//note - this does not require the original residue coding, as it only outputs transformed coordinates
		//superimposition uses all residues with a good minimum score
		tempfileout1 = PDBOUT_dir + obj1 + ".pdb";
		tempfileout2 = PDBOUT_dir + obj2 + ".pdb";
		//if(Res_align_sim.size() > 0){
		//	transformation = write_full_pdb(info,res1,res2,Res_align_sim,filein1,tempfileout1,domain1,filein2,tempfileout2,domain2);
		//} else { //use full alignment for superimposition if no residues are well-aligned.
		
		//TEMPORARY FILTER TO MAKE THE GLOBAL SUPERPOSITION BASED ONLY ON SUFFICIENTLY SIMILAR RESIDUES
		if(PROC_NORM==1){
			Res_align_sim.clear();
			for(unsigned int i=0; i<Res_align.size(); i++){
				if(Res_align[i].dist > 2.0){	//this value may depend on fragment length, or on overall alignment score.
					Res_align_sim.push_back(Res_align[i]);
				}
			}
		} else if(SUPERPOSE < 1000){
			Res_align_sim.clear();
			for(unsigned int i=0; i<Res_align.size(); i++){
				if(Res_align[i].dist <= SUPERPOSE){
					Res_align_sim.push_back(Res_align[i]);
				}
			}
		} else {
            Res_align_sim = Res_align;
        }
        if(Res_align_sim.size()==0){
            Res_align_sim = Res_align;
        }
		
		/*Res_align_sim.clear();
         for(unsigned int i=0; i<9; i++){
         Res_align_sim.push_back(Res_align[i]);
         Res_align_sim[i].res2 = 2+i;
         }*/
		
		transformation = write_full_pdb(info,res1,res2,Res_align_sim,filein1,tempfileout1,domain1,filein2,tempfileout2,domain2,OUT_PDB_FULL,USE_SIDE);
		//}
		
		tempfileout1 = workingdirectory + ".HTML_files/tmp/" + obj1 + "_" + obj2 + ".txt";
		outfile.open(tempfileout1.c_str());
		if(outfile.is_open()){
			outfile << obj1 << endl;
			outfile << obj2 << endl;
			outfile.close();
		} else {
			cout << endl << "Cannot open " << tempfileout1 << " for writing." << endl;
		}
	} else {
		transformation = get_transf(res1,res2,Res_align,USE_SIDE);
	}
	
	tempfileout1 = workingdirectory + "Superposition/Transformations/" + obj1 + "/" + obj1 + "_" + obj2 + ".txt";
	write_transformation(transformation, "Global", tempfileout1,obj2,1);
	
    return transformation;
}

void write_transformation(Transform &transf, string transfID, string &fileout, string &obj2, bool START)
{
	ofstream outfile;
	Array2D<double> r = transf.get_r();
	coord f = transf.get_f();
	coord p = transf.get_p();
    coord t = transf.get_t();
	coord tmp_trans = p - t;
    double trans_dist = dist(tmp_trans);
    
	if(START==1){
		outfile.open(fileout.c_str());
	} else {
		outfile.open(fileout.c_str(),ios::app);	
	}
	
	if(outfile.is_open()){
		if(START==1){
			outfile << "# ProSMART Transformations File" << endl;
		}
		outfile << endl << "CHAIN_ID: " << obj2
		<< endl << "TRANSF_ID: " << transfID
        << endl << "ANGLE_DIST: " << decimal_places(transf.get_r_angle(),4) << " " << decimal_places(trans_dist,4)
		<< endl << "ROTATION" << endl;
		for(int i=0; i<3; i++){
			outfile << r[i][0] << " " << r[i][1] << " " << r[i][2] << endl;
		}
		outfile << "TRANSLATION" << endl
		<< f.x << " " << f.y << " " << f.z << endl;
		outfile.close();
    } else {
        cout << "Unable to open output file " << fileout << " for writing" << endl;
    }
	
	return;
}

void write_pymol_screw_axis(Transform &transf, string &fileout, string id)
{
    coord screw_point;
    coord r_axis;
    double rot_angle;
    double screw = transf.get_r_axis_angle(screw_point,r_axis,rot_angle);
    cout << endl << "Axis-angle representation for: " << id << endl;
    cout << "\t" << "rotation angle: " << rot_angle << endl;
    cout << "\t" << "signed screw translation magnitude: " << screw << endl;   
    cout << "\t" << "rotation axis: " << r_axis;
    cout << "\t" << "screw origin: " << screw_point;
    
   if(rot_angle==0.0 && screw==0.0){
      return;
   }
   
	ofstream outfile;
    string filename = fileout+"_"+id+"_screw.pml";
    string fileid = get_filename(fileout);
    string pa1 = fileid+"_"+id+"_screw_1";
    string pa2 = fileid+"_"+id+"_screw_2";
    string paext1 = fileid+"_"+id+"_screw_ext1";
    string paext2 = fileid+"_"+id+"_screw_ext2";
    string pa3 = fileid+"_"+id+"_screw_3";
    string ta1 = fileid+"_"+id+"_trans_1";
    string ta2 = fileid+"_"+id+"_trans_2";
    string ta3 = fileid+"_"+id+"_trans_3";
    string ta4 = fileid+"_"+id+"_trans_4";
    string pad = fileid+"_"+id+"_screw_d";
    string ptd1 = fileid+"_"+id+"_screw_d1";
    string ptd2 = fileid+"_"+id+"_screw_d2";
    string ptd3 = fileid+"_"+id+"_screw_d3";
    string ang1 = fileid+"_"+id+"_angle1";
    
    coord t1 = transf.get_t();
    coord t2 = transf.get_p();
    
    coord tmp = t1-screw_point;
    
    coord t3 = tmp*transf.get_r();
    t3 = t3 + screw_point;
    
    coord t4 = r_axis*dotprod(tmp,r_axis);
    t4 = screw_point+t4;
    
    double axis_mod = 100.0;
    if(screw<0.0){
        axis_mod = -1.0*axis_mod;
    }
    
	outfile.open(filename.c_str());
	if(outfile.is_open()){      
		//outfile << "pseudoatom " << pa1 << ", pos=[" << screw_point.x << ", " << screw_point.y << ", " << screw_point.z << "]" << endl;
		//outfile << "pseudoatom " << pa2 << ", pos=[" << screw_point.x+(screw*r_axis.x) << ", " << screw_point.y+(screw*r_axis.y) << ", " << screw_point.z+(screw*r_axis.z) << "]" << endl;
        
      //outfile << "pseudoatom " << paext1 << ", pos=[" << screw_point.x-(axis_mod*r_axis.x) << ", " << screw_point.y-(axis_mod*r_axis.y) << ", " << screw_point.z-(axis_mod*r_axis.z) << "]" << endl;
		//outfile << "pseudoatom " << paext2 << ", pos=[" << screw_point.x+((screw+axis_mod)*r_axis.x) << ", " << screw_point.y+((screw+axis_mod)*r_axis.y) << ", " << screw_point.z+((screw+axis_mod)*r_axis.z) << "]" << endl;
      
      outfile << "pseudoatom " << paext1 << ", pos=[" << t4.x-(axis_mod*r_axis.x) << ", " << t4.y-(axis_mod*r_axis.y) << ", " << t4.z-(axis_mod*r_axis.z) << "]" << endl;
		outfile << "pseudoatom " << paext2 << ", pos=[" << t4.x+((screw+axis_mod)*r_axis.x) << ", " << t4.y+((screw+axis_mod)*r_axis.y) << ", " << t4.z+((screw+axis_mod)*r_axis.z) << "]" << endl;
      
      outfile << "pseudoatom " << ta1 << ", pos=[" << t1.x << ", " << t1.y << ", " << t1.z << "]" << endl;
		outfile << "pseudoatom " << ta2 << ", pos=[" << t2.x << ", " << t2.y << ", " << t2.z << "]" << endl;
		outfile << "pseudoatom " << ta3 << ", pos=[" << t3.x << ", " << t3.y << ", " << t3.z << "]" << endl;
		outfile << "pseudoatom " << ta4 << ", pos=[" << t4.x << ", " << t4.y << ", " << t4.z << "]" << endl;
        outfile << "distance " << pad << ", /" << paext1 << "/////1, /" << paext2 << "/////1; hide labels, " << pad << endl;
        //outfile << "distance " << ptd1 << ", /" << ta1 << "/////1, /" << pa1 << "/////1" << endl;
        //outfile << "distance " << ptd1 << ", /" << ta2 << "/////1, /" << pa2 << "/////1" << endl;
        //outfile << "distance " << ptd2 << ", /" << pa1 << "/////1, /" << pa2 << "/////1" << endl;
        outfile << "distance " << ptd3 << ", /" << ta3 << "/////1, /" << ta2 << "/////1" << endl;
        outfile << "angle " << ang1 << ", /" << ta1 << "/////1, /" << ta4 << "/////1, /" << ta3 << "/////1" << endl;
        //outfile << "show spheres, " << pa1 << " " << pa2 << " " << ta1 << " " << ta2 << endl;
      outfile << "show spheres, " << ta1 << " " << ta2 << endl;
        outfile << "color red, " << ta1 << endl;
        outfile << "color blue, " << ta2 << endl;
        //outfile << "color white, " << pa1 << " " << pa2 << endl;
		outfile.close();
      cout << endl << "Screw axis-angle PyMOL script written to: " << filename << endl;
    } else {
        cout << "Unable to open output file " << filename << " for writing" << endl;
    }
	
	return;
}

/*void write_pymol_screw_axis(Transform &transf, Transform &transf2, string &fileout, string id)
{
    coord screw_point;
    coord r_axis;
    double rot_angle;
    double screw = transf.get_r_axis_angle(screw_point,r_axis,rot_angle);
    cout << endl << "Axis-angle representation for: " << id << endl;
    cout << "\t" << "rotation angle: " << rot_angle << endl;
    cout << "\t" << "signed screw translation magnitude: " << screw << endl;
    cout << "\t" << "rotation axis: " << r_axis;
    cout << "\t" << "screw origin: " << screw_point;
    
    coord screw_point2;
    coord r_axis2;
    double rot_angle2;
    transf2.get_r_axis_angle(screw_point2,r_axis2,rot_angle2);
    
	ofstream outfile;
    string filename = fileout+"_"+id+"_screw.pml";
    string fileid = get_filename(fileout);
    string pa1 = fileid+"_"+id+"_screw_1";
    string pa2 = fileid+"_"+id+"_screw_2";
    string paext1 = fileid+"_"+id+"_screw_ext1";
    string paext2 = fileid+"_"+id+"_screw_ext2";
    string pa3 = fileid+"_"+id+"_screw_3";
    string ta1 = fileid+"_"+id+"_trans_1";
    string ta2 = fileid+"_"+id+"_trans_2";
    string ta1a = fileid+"_"+id+"_trans_1a";
    string ta2a = fileid+"_"+id+"_trans_2a";
    string ta3 = fileid+"_"+id+"_trans_3";
    string ta4 = fileid+"_"+id+"_trans_4";
    string pad = fileid+"_"+id+"_screw_d";
    string ptd1 = fileid+"_"+id+"_screw_d1";
    string ptd1a = fileid+"_"+id+"_screw_d1a";
    string ptd2 = fileid+"_"+id+"_screw_d2";
    string ptd2a = fileid+"_"+id+"_screw_d2a";
    string ptdp = fileid+"_"+id+"_screw_dp";
    string ptd3 = fileid+"_"+id+"_screw_d3";
    string ang1 = fileid+"_"+id+"_angle1";
    
    coord t1 = transf.get_t();
    coord t2 = transf.get_p();
    coord tmp = t1-screw_point;
    coord t3 = tmp*transf.get_r();
    t3 = t3 + screw_point;
    coord t4 = r_axis*dotprod(tmp,r_axis);
    t4 = screw_point+t4;
    
    coord t1a = transf2.get_t();
    coord t2a = transf2.get_p();
    
    double axis_mod = 100.0;
    if(screw<0.0){
        axis_mod = -1.0*axis_mod;
    }
    
	outfile.open(filename.c_str());
	if(outfile.is_open()){
		outfile << "pseudoatom " << pa1 << ", pos=[" << screw_point.x << ", " << screw_point.y << ", " << screw_point.z << "]" << endl;
		outfile << "pseudoatom " << pa2 << ", pos=[" << screw_point.x+(screw*r_axis.x) << ", " << screw_point.y+(screw*r_axis.y) << ", " << screw_point.z+(screw*r_axis.z) << "]" << endl;
        outfile << "pseudoatom " << paext1 << ", pos=[" << screw_point.x-(axis_mod*r_axis.x) << ", " << screw_point.y-(axis_mod*r_axis.y) << ", " << screw_point.z-(axis_mod*r_axis.z) << "]" << endl;
		outfile << "pseudoatom " << paext2 << ", pos=[" << screw_point.x+((screw+axis_mod)*r_axis.x) << ", " << screw_point.y+((screw+axis_mod)*r_axis.y) << ", " << screw_point.z+((screw+axis_mod)*r_axis.z) << "]" << endl;
        outfile << "pseudoatom " << ta1 << ", pos=[" << t1.x << ", " << t1.y << ", " << t1.z << "]" << endl;
		outfile << "pseudoatom " << ta2 << ", pos=[" << t2.x << ", " << t2.y << ", " << t2.z << "]" << endl;
        outfile << "pseudoatom " << ta1a << ", pos=[" << t1a.x << ", " << t1a.y << ", " << t1a.z << "]" << endl;
		outfile << "pseudoatom " << ta2a << ", pos=[" << t2a.x << ", " << t2a.y << ", " << t2a.z << "]" << endl;
        outfile << "pseudoatom " << ta3 << ", pos=[" << t3.x << ", " << t3.y << ", " << t3.z << "]" << endl;
		outfile << "pseudoatom " << ta4 << ", pos=[" << t4.x << ", " << t4.y << ", " << t4.z << "]" << endl;
        outfile << "distance " << pad << ", /" << paext1 << "/////1, /" << paext2 << "/////1; hide labels, " << pad << endl;
        outfile << "distance " << ptd1 << ", /" << ta1 << "/////1, /" << pa1 << "/////1" << endl;
        outfile << "distance " << ptd1a << ", /" << ta1a << "/////1, /" << pa1 << "/////1" << endl;
        outfile << "distance " << ptd2 << ", /" << ta2 << "/////1, /" << pa2 << "/////1" << endl;
        outfile << "distance " << ptd2a << ", /" << ta2a << "/////1, /" << pa2 << "/////1" << endl;
        outfile << "distance " << ptdp << ", /" << pa1 << "/////1, /" << pa2 << "/////1" << endl;
        outfile << "distance " << ptd3 << ", /" << ta3 << "/////1, /" << ta2 << "/////1" << endl;
        outfile << "distance " << ptd3 << ", /" << ta3 << "/////1, /" << ta2a << "/////1" << endl;
        outfile << "angle " << ang1 << ", /" << ta1 << "/////1, /" << ta4 << "/////1, /" << ta3 << "/////1" << endl;
        outfile << "angle " << ang1 << ", /" << ta1a << "/////1, /" << ta4 << "/////1, /" << ta3 << "/////1" << endl;
        outfile << "show spheres, " << pa1 << " " << pa2 << " " << ta1 << " " << ta2 << " " << ta1a << " " << ta2a << endl;
        outfile << "color red, " << ta1 << " " << ta1a << endl;
        outfile << "color blue, " << ta2 << " " << ta2a << endl;
        outfile << "color white, " << pa1 << " " << pa2 << endl;
		outfile.close();
    } else {
        cout << "Unable to open output file " << filename << " for writing" << endl;
    }
	
	return;
}*/

void output_scores(string filename1, string filename2, char domain1, char domain2, string workingdirectory, int align_len, double align_percent, double SeqID, double align_rmsd, double score_central, double score_min, double min_score_central, vector<string> &info, double &av_sideRMSD_score, double &av_sideMaxDist_score, vector<int> &no_large_side, double &asa_corr)
{
    string obj1 = get_filename(filename1) + "_" + domain1;
    string obj2 = get_filename(filename2) + "_" + domain2;
    
    string fileout = workingdirectory + "Global_Statistics/Single/" + obj1 + "_" + obj2 + ".txt";
    
    ofstream outfile;
    outfile.open(fileout.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Pairwise Scores File" << endl;
		for(unsigned int i=0; i<info.size(); i++){
			outfile << info[i] << endl;
		}
		outfile.precision(3);
      outfile << "NumberAligned     " << align_len << endl;
		outfile << "PercentAligned    " << align_percent << endl;
		outfile << "SequenceIdentity  " << SeqID << endl;
		outfile << "AverageProcrustes " << score_central << endl;
		outfile << "AverageFlexible   " << score_min << endl;
		outfile << "GlobalRMSD        " << align_rmsd << endl;
      outfile << "BestFragmentScore " << min_score_central << endl;
#ifdef EXTRA_GLOBAL_SCORES
      outfile << "AvLocalSideRMSD " << av_sideRMSD_score << endl;
      outfile << "AvLocalSideMaxD " << av_sideMaxDist_score << endl;
#ifdef INCLUDE_SCORE_NUMDIST
        outfile << "NoLargeSideMaxD";
        for(unsigned int i=0; i<no_large_side.size(); i++){
            outfile << " " << no_large_side[i];
        }
#endif
        outfile << endl << "ASA_MaxD_Corr " << asa_corr << endl;
        outfile << endl;
#endif
		outfile.close();
        cout << endl << "Scores file written to: " << fileout;
    } else {
        cout << "Unable to open output file " << fileout << " for writing" << endl;
    }
    
    return;
}

/*void output_scores_full(string score_id, string filename1, string filename2, char domain1, char domain2, string workingdirectory, int align_len, double align_percent, double SeqID, double align_rmsd, double score_central, double score_min, vector<string> &info)
 {
 string obj1 = get_filename(filename1) + "_" + domain1;
 string obj2 = get_filename(filename2) + "_" + domain2;
 
 string fileout = workingdirectory + "Scores/Single/" + score_id + "_" + obj1 + "_" + obj2 + ".txt";
 
 ofstream outfile;
 outfile.open(fileout.c_str());
 if(outfile.is_open()){
 outfile << "# Pairwise Scores File" << endl;
 outfile << "#" << endl;
 for(unsigned int i=0; i<info.size(); i++){
 outfile << info[i] << endl;
 }
 outfile.precision(3);
 outfile << "Aligned " << align_len << endl;
 outfile << "Percent " << align_percent << endl;
 outfile << "SeqIden " << SeqID << endl;
 outfile << "Central " << score_central << endl;
 outfile << "Minimum " << score_min << endl;
 outfile << "RMSD    " << align_rmsd << endl;
 */		/*if(NormProc > 0.00001){
         outfile << "NrmProc " << NormProc << endl;
         } else {
         outfile << "NrmProc " << 0 << endl;
         }
         outfile << "KolSmir " << KS << endl;*/
/*		outfile.close();
 cout << endl << "Scores file written to: " << fileout;
 } else {
 cout << "Unable to open output file " << fileout << " for writing" << endl;
 }
 
 return;
 }*/

/*void output_residue_scores(vector<residue_alignment> &resalign, vector<string> &res_list, vector<int> &res_corresp, string &fileout, vector<string> &info)
 {
 ofstream outfile;
 outfile.open(fileout.c_str());
 if(outfile.is_open()){
 outfile << "# Residue Sidechain RMSD" << endl;
 outfile << "#" << endl;
 for(unsigned int i=0; i<info.size(); i++){
 outfile << info[i] << endl;
 }
 for(unsigned int i=0; i<res_list.size(); i++){
 outfile << res_list[i] << " ";
 if(res_corresp[i] >= 0){
 outfile << resalign[res_corresp[i]].dist3;		//dist3 for sidechain rmsd
 } else {
 outfile << "NA";
 }
 outfile << endl;
 }
 outfile.close();
 cout << endl << "Residue scores file written to: " << fileout;
 } else {
 cout << "Unable to open output file " << fileout << " for writing" << endl;
 }
 
 return;
 }*/


vector<string> info_vector(string filein1, string filein2, char domain1, char domain2, unsigned short fragLen, bool align_cutoff, int Nres1, int Nres2, int aligned_residues, double SeqID, double avRMSD, double av_score_central, double av_score_min, double ALIGN_SCORE, int ALIGN_MODE, bool EXTRA_REFINEMENT, double SIDE_SCORE)
{
    vector<string> info;
    /*stringstream ss;
     
     ss << "# ProSMART ALIGN ";
     info.push_back(ss.str()); ss.str("");
     ss << "#";
     info.push_back(ss.str()); ss.str("");
     ss << "# Input Files:";
     info.push_back(ss.str()); ss.str("");
     ss << "# File 1:\t" << filein1 << "  \tchain " << domain1 << "\t\t" << Nres1 << " residues";
     info.push_back(ss.str()); ss.str("");
     ss << "# File 2:\t" << filein2 << "  \tchain " << domain2 << "\t\t" << Nres2 << " residues";
     info.push_back(ss.str()); ss.str("");
     ss << "#";
     info.push_back(ss.str()); ss.str("");
     ss << "# Parameters:";
     info.push_back(ss.str()); ss.str("");
     ss << "# Fragment Length: " << fragLen << " residues";
     info.push_back(ss.str()); ss.str("");
     ss << "# Alignment Mode: " << ALIGN_MODE;
     info.push_back(ss.str()); ss.str("");
     ss << "#";
     info.push_back(ss.str()); ss.str("");
     ss << "# TAG: " << delete_ext(filein1) << "_" << domain1 << "_" << delete_ext(filein2) << "_" << domain2;
     info.push_back(ss.str()); ss.str("");
     */
    return info;
}

coord prosmart_green;
coord prosmart_red;

void define_pymol_colors(coord &col1, coord &col2)
{
	prosmart_green = col1;
	prosmart_red = col2;
	return;
}


//#define UNIFORM_COLOUR_ALIGNED_RESIDUES
//#define UNIFORM_COLOUR_SIMILAR_RESIDUES
void write_pymol(vector<string> &info, string obj1, string obj2, string file, char domain1, char domain2, double score_cutoff, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2, vector<int> &All_Res1, vector<int> &All_Res2, Array1D<double> &s1, Array1D<double> &s2)
{		
	ofstream outfile;
	double val;
	double ival;
    vector<double> colors11;
    vector<double> colors12;
    vector<double> colors13;
    vector<double> colors21;
    vector<double> colors22;
    vector<double> colors23;
	Array1D<double> scores1 = decimal_places(s1,2);
    Array1D<double> scores2 = decimal_places(s2,2);
    
    for(int i=0; i<scores1.dim1(); i++){
        if(scores1[i] < 0.0){
			colors11.push_back(1);			//not aligned: color = white
			colors12.push_back(1);
			colors13.push_back(1);
		} else {
#ifdef UNIFORM_COLOUR_ALIGNED_RESIDUES
			colors11.push_back(prosmart_red.x);
			colors12.push_back(prosmart_red.y);
			colors13.push_back(prosmart_red.z);
#else
#ifdef UNIFORM_COLOUR_SIMILAR_RESIDUES
			if(scores1[i] < score_cutoff){	//green = similar  ->  red = comparitively dissimilar
				colors11.push_back(prosmart_green.x);
				colors12.push_back(prosmart_green.y);
				colors13.push_back(prosmart_green.z);
			} else {
				colors11.push_back(prosmart_red.x);		//structurally dissimilar: color = red
				colors12.push_back(prosmart_red.y);
				colors13.push_back(prosmart_red.z);
			}
#else
			if(scores1[i] < score_cutoff){	//green = similar  ->  red = comparitively dissimilar
				val = scores1[i]/score_cutoff;
				ival = 1-val;
				colors11.push_back(prosmart_green.x*ival+prosmart_red.x*val);
				colors12.push_back(prosmart_green.y*ival+prosmart_red.y*val);
				colors13.push_back(prosmart_green.z*ival+prosmart_red.z*val);
			} else {
				colors11.push_back(prosmart_red.x);		//structurally dissimilar: color = red
				colors12.push_back(prosmart_red.y);
				colors13.push_back(prosmart_red.z);
			}
#endif
#endif
		}
    }
    for(int i=0; i<scores2.dim1(); i++){
        if(scores2[i] < 0.0){
			colors21.push_back(1);
			colors22.push_back(1);
			colors23.push_back(1);
		} else {
#ifdef UNIFORM_COLOUR_ALIGNED_RESIDUES
			colors21.push_back(prosmart_red.x);
			colors22.push_back(prosmart_red.y);
			colors23.push_back(prosmart_red.z);
#else
#ifdef UNIFORM_COLOUR_SIMILAR_RESIDUES
			if(scores2[i] < score_cutoff){	//green = similar  ->  red = comparitively dissimilar
				colors21.push_back(prosmart_green.x);
				colors22.push_back(prosmart_green.y);
				colors23.push_back(prosmart_green.z);
			} else {
				colors21.push_back(prosmart_red.x);		//structurally dissimilar: color = red
				colors22.push_back(prosmart_red.y);
				colors23.push_back(prosmart_red.z);
			}
#else
			if(scores2[i] < score_cutoff){
				val = scores2[i]/score_cutoff;
				ival = 1-val;
				colors21.push_back(prosmart_green.x*ival+prosmart_red.x*val);
				colors22.push_back(prosmart_green.y*ival+prosmart_red.y*val);
				colors23.push_back(prosmart_green.z*ival+prosmart_red.z*val);
			} else {
				colors21.push_back(prosmart_red.x);		//structurally dissimilar: color = red
				colors22.push_back(prosmart_red.y);
				colors23.push_back(prosmart_red.z);
			}
#endif
#endif
		}
    }
    
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
		for(unsigned int i=0; i<info.size(); i++){
			outfile << info[i] << endl;
		}
		outfile << "color white, (" << obj1 << "//" << domain1 << "/*/*)" << endl;
		for(int i=0; i<scores1.dim1(); i++){
			outfile << "set_color newcolor" << i << " = [" << colors11[i] << "," << colors12[i] << "," << colors13[i] << "]; color newcolor" << i << ", (" << obj1 << "//" << domain1 << "/";
			if(original_res1[All_Res1[i]].res<0){
				outfile << "\\";
			}
			outfile << original_res1[All_Res1[i]].res;
			if(original_res1[All_Res1[i]].ins != ' '){
				outfile << original_res1[All_Res1[i]].ins; 
			}
			outfile << "/*)" << endl;
		}
		outfile << "color white, (" << obj2 << "//" << domain2 << "/*/*)" << endl;
		for(int i=0; i<scores2.dim1(); i++){
			outfile << "set_color newcolorA" << i << " = [" << colors21[i] << "," << colors22[i] << "," << colors23[i] << "]; color newcolorA" << i << ", (" << obj2 << "//" << domain2 << "/";
			if(original_res2[All_Res2[i]].res<0){
				outfile << "\\";
			}
			outfile << original_res2[All_Res2[i]].res;
			if(original_res2[All_Res2[i]].ins != ' '){
				outfile << original_res2[All_Res2[i]].ins;
			}
			outfile << "/*)" << endl;
		}
        outfile.close();
		cout << endl << "Colour script written to: " << file;
    } else {
        cout << "Unable to open output file " << file << " for writing" << endl;
    }
   
   //write chimera colour script
   string chimera_file = get_folder(file) + "Chimera/" + get_filename(file) + ".com";
   outfile.open(chimera_file.c_str());
   if(outfile.is_open()){
      outfile << "# ProSMART Colour File - Chimera Command Script\n";
      outfile << "color white prosmart1; color white prosmart2; ";
      for(int i=0; i<scores1.dim1(); i++){
         outfile << "select prosmart1 & :" << original_res1[All_Res1[i]].res;
         if(original_res1[All_Res1[i]].ins != ' '){
				outfile << original_res1[All_Res1[i]].ins; 
			}
			outfile << "; color " << colors11[i] << "," << colors12[i] << "," << colors13[i] << " sel; ";
		}
      for(int i=0; i<scores2.dim1(); i++){
         outfile << "select prosmart2 & :" << original_res2[All_Res2[i]].res;
         if(original_res2[All_Res2[i]].ins != ' '){
				outfile << original_res2[All_Res2[i]].ins; 
			}
         outfile << "; color " << colors21[i] << "," << colors22[i] << "," << colors23[i] << " sel; ";
		}
      outfile << "~select";
      outfile.close();
		cout << endl << "Colour script written to: " << chimera_file;
   } else {
      cout << "Unable to open output file " << chimera_file << " for writing" << endl;
   }
	
	return;
}

void write_pymol_minimum(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, Residues &res2, string obj1_, string obj2_, string file, char domain1, char domain2, double score_cutoff, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2)
{
	string obj1 = obj1_ + "*";
	string obj2 = obj2_ + "*";			//this means that all files starting with obj2 will be coloured (i.e. original and all clusters)
    
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    vector<int> All_Res2 = res2.get_res();
    Array1D<double> s1(All_Res1.size(),-1.0);		//will store the scores corresponding to the residues
    Array1D<double> s2(All_Res2.size(),-1.0);		//if a residue is unaligned then it scores -1.0
    int idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){        
        if(All_Res1[idx]==Res_align[i].res1){
            s1[idx] = Res_align[i].dist;
        } else {
            i--;
        }
        idx++;
    }
    idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res2[idx]==Res_align[i].res2){
            s2[idx] = Res_align[i].dist;
        } else {
            i--;
        }
        idx++;
    }
    
	write_pymol(info, obj1, obj2, file, domain1, domain2, score_cutoff, original_res1, original_res2, All_Res1, All_Res2, s1, s2);
	
	return;
}

void write_pymol_central(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, Residues &res2, string obj1_, string obj2_, string file, char domain1, char domain2, double score_cutoff, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2)
{
    ofstream outfile;
	string obj1 = obj1_ + "*";
	string obj2 = obj2_ + "*";			//this means that all files starting with obj2 will be coloured (i.e. original and all clusters)
    
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    vector<int> All_Res2 = res2.get_res();
    Array1D<double> s1(All_Res1.size(),-1.0);		//will store the scores corresponding to the residues
    Array1D<double> s2(All_Res2.size(),-1.0);		//if a residue is unaligned then it scores -1.0
    int idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res1[idx]==Res_align[i].res1){
            s1[idx] = Res_align[i].dist2;
        } else {
            i--;
        }
        idx++;
    }
    idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res2[idx]==Res_align[i].res2){
            s2[idx] = Res_align[i].dist2;
        } else {
            i--;
        }
        idx++;
    }
    
    write_pymol(info, obj1, obj2, file, domain1, domain2, score_cutoff, original_res1, original_res2, All_Res1, All_Res2, s1, s2);
    
    return;
}


void write_pymol_rot(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, Residues &res2, string obj1_, string obj2_, string file, char domain1, char domain2, double score_cutoff, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2)
{
    ofstream outfile;
	string obj1 = obj1_ + "*";
	string obj2 = obj2_ + "*";			//this means that all files starting with obj2 will be coloured (i.e. original and all clusters)
	
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    vector<int> All_Res2 = res2.get_res();
    Array1D<double> s1(All_Res1.size(),-1.0);		//will store the scores corresponding to the residues
    Array1D<double> s2(All_Res2.size(),-1.0);		//if a residue is unaligned then it scores -1.0
    int idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res1[idx]==Res_align[i].res1){
			s1[idx] = 1000*Res_align[i].rot;
		} else {
			i--;
		}
		idx++;
    }
    idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res2[idx]==Res_align[i].res2){
			s2[idx] = 1000*Res_align[i].rot;
		} else {
			i--;
		}
		idx++;
    }
	
    write_pymol(info, obj1, obj2, file, domain1, domain2, score_cutoff, original_res1, original_res2, All_Res1, All_Res2, s1, s2);
    
    return;
}

void write_pymol_sidechainRMSD(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, Residues &res2, string obj1_, string obj2_, string file, char domain1, char domain2, double score_cutoff, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2)
{
    ofstream outfile;
	string obj1 = obj1_ + "*";
	string obj2 = obj2_ + "*";			//this means that all files starting with obj2 will be coloured (i.e. original and all clusters)
	
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    vector<int> All_Res2 = res2.get_res();
    Array1D<double> s1(All_Res1.size(),-1.0);		//will store the scores corresponding to the residues
    Array1D<double> s2(All_Res2.size(),-1.0);		//if a residue is unaligned then it scores -1.0
    int idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res1[idx]==Res_align[i].res1){
			s1[idx] = Res_align[i].dist3;
		} else {
			i--;
		}
		idx++;
    }
    idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res2[idx]==Res_align[i].res2){
			s2[idx] = Res_align[i].dist3;
		} else {
			i--;
		}
		idx++;
    }
	
    write_pymol(info, obj1, obj2, file, domain1, domain2, score_cutoff, original_res1, original_res2, All_Res1, All_Res2, s1, s2);
	
    return;
}

void write_pymol_sidechainMAX(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, Residues &res2, string obj1_, string obj2_, string file, char domain1, char domain2, double score_cutoff, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2)
{
    ofstream outfile;
	string obj1 = obj1_ + "*";
	string obj2 = obj2_ + "*";			//this means that all files starting with obj2 will be coloured (i.e. original and all clusters)
	
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    vector<int> All_Res2 = res2.get_res();
    Array1D<double> s1(All_Res1.size(),-1.0);		//will store the scores corresponding to the residues
    Array1D<double> s2(All_Res2.size(),-1.0);		//if a residue is unaligned then it scores -1.0
    int idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res1[idx]==Res_align[i].res1){
			s1[idx] = Res_align[i].dist5;
		} else {
			i--;
		}
		idx++;
    }
    idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res2[idx]==Res_align[i].res2){
			s2[idx] = Res_align[i].dist5;
		} else {
			i--;
		}
		idx++;
    }
	
    write_pymol(info, obj1, obj2, file, domain1, domain2, score_cutoff, original_res1, original_res2, All_Res1, All_Res2, s1, s2);
	
    return;
}

void write_pymol_sidechainAV(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, Residues &res2, string obj1_, string obj2_, string file, char domain1, char domain2, double score_cutoff, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2)
{
    ofstream outfile;
	string obj1 = obj1_ + "*";
	string obj2 = obj2_ + "*";			//this means that all files starting with obj2 will be coloured (i.e. original and all clusters)
	
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    vector<int> All_Res2 = res2.get_res();
    Array1D<double> s1(All_Res1.size(),-1.0);		//will store the scores corresponding to the residues
    Array1D<double> s2(All_Res2.size(),-1.0);		//if a residue is unaligned then it scores -1.0
    int idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res1[idx]==Res_align[i].res1){
			s1[idx] = Res_align[i].dist4;
		} else {
			i--;
		}
		idx++;
    }
    idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res2[idx]==Res_align[i].res2){
			s2[idx] = Res_align[i].dist4;
		} else {
			i--;
		}
		idx++;
    }
	
    write_pymol(info, obj1, obj2, file, domain1, domain2, score_cutoff, original_res1, original_res2, All_Res1, All_Res2, s1, s2);
    
    return;
}

void write_pymol_helix(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, string obj1_, string file, char domain1, double score_cutoff, vector<res_corresp> &original_res1)
{
    string obj1 = obj1_ + "*";
	ofstream outfile;
    
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    Array1D<double> s1(All_Res1.size(),-1.0);		//will store the scores corresponding to the residues
    int idx = 0;
    for(unsigned int i=0; i<Res_align.size(); i++){
        if(All_Res1[idx]==Res_align[i].res1){
			s1[idx] = Res_align[i].dist;
		} else {
			i--;
		}
		idx++;
    }
	
    Array1D<double> scores1 = decimal_places(s1,2);
	double val;
	double ival;
    vector<double> colors11;
    vector<double> colors12;
    vector<double> colors13;
    vector<double> colors21;
    vector<double> colors22;
    vector<double> colors23;
    for(int i=0; i<scores1.dim1(); i++){
        if(scores1[i] < 0.0){
			colors11.push_back(1);			//not aligned: color = white
			colors12.push_back(1);
			colors13.push_back(1);
		} else {
			if(scores1[i] < score_cutoff){	//green = similar  ->  red = comparitively dissimilar
				val = scores1[i]/score_cutoff;
				ival = 1-val;
				colors11.push_back(prosmart_green.x*ival+prosmart_red.x*val);
				colors12.push_back(prosmart_green.y*ival+prosmart_red.y*val);
				colors13.push_back(prosmart_green.z*ival+prosmart_red.z*val);
			} else {
				colors11.push_back(prosmart_red.x);		//structurally dissimilar: color = red
				colors12.push_back(prosmart_red.y);
				colors13.push_back(prosmart_red.z);
			}
		}
    }
    
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
		for(unsigned int i=0; i<info.size(); i++){
			outfile << info[i] << endl;
		}
		outfile << "color white, (" << obj1 << "//" << domain1 << "/*/*)" << endl;
		for(int i=0; i<scores1.dim1(); i++){
			outfile << "set_color newcolor" << i << " = [" << colors11[i] << "," << colors12[i] << "," << colors13[i] << "]; color newcolor" << i << ", (" << obj1 << "//" << domain1 << "/" << original_res1[All_Res1[i]].res;
			if(original_res1[All_Res1[i]].ins != ' '){
				outfile << original_res1[All_Res1[i]].ins; 
			}
			outfile << "/*)" << endl;
		}
        outfile.close();
		cout << endl << "Colour script written to: " << file;
    } else {
        cout << "Unable to open output file " << file << " for writing" << endl;
    }
    
    return;
}

void write_pymol_sse(vector<string> &info, vector<residue_alignment> &Res_align, Residues &res1, Residues &res2, string obj1_, string obj2_, string file, char domain1, char domain2, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2, vector<int> &res_type1, vector<int> &res_type2, vector<coord> &library_colors)
{
    ofstream outfile;
	string obj1 = obj1_ + "*";
	string obj2 = obj2_ + "*";			//this means that all files starting with obj2 will be coloured (i.e. original and all clusters)
    
    vector<int> All_Res1 = res1.get_res();		//stores the residue numbers for all residues.
    vector<int> All_Res2 = res2.get_res();
    Array1D<int> s1(All_Res1.size(),-1);		//stores the types of aligned residues
    Array1D<int> s2(All_Res2.size(),-1);		//if a residue is unaligned then it is -1
	
	for(unsigned int i=0; i<All_Res1.size(); i++){
        if(All_Res1[i] >= (int)res_type1.size()){
			break;
		}
		s1[i] = res_type1[All_Res1[i]];
    }
	for(unsigned int i=0; i<All_Res2.size(); i++){
        if(All_Res2[i] >= (int)res_type2.size()){
			break;
		}
		s2[i] = res_type2[All_Res2[i]];
    }
	
    vector<double> colors11;
    vector<double> colors12;
    vector<double> colors13;
    vector<double> colors21;
    vector<double> colors22;
    vector<double> colors23;
    for(int i=0; i<s1.dim1(); i++){
		if(s1[i] < 0){			//unaligned: white
			colors11.push_back(1.0);
			colors12.push_back(1.0);
			colors13.push_back(1.0);
		} else {
			colors11.push_back(library_colors[s1[i]].x);
			colors12.push_back(library_colors[s1[i]].y);
			colors13.push_back(library_colors[s1[i]].z);
		}
    }
    for(int i=0; i<s2.dim1(); i++){
		if(s2[i] < 0){			//unaligned: white
			colors21.push_back(1.0);
			colors22.push_back(1.0);
			colors23.push_back(1.0);
		} else {
			colors21.push_back(library_colors[s2[i]].x);
			colors22.push_back(library_colors[s2[i]].y);
			colors23.push_back(library_colors[s2[i]].z);
		}
    }
    
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
		for(unsigned int i=0; i<info.size(); i++){
			outfile << info[i] << endl;
		}
		outfile << "color white, (" << obj1 << "//" << domain1 << "/*/*)" << endl;
		for(int i=0; i<s1.dim1(); i++){
			outfile << "set_color newcolor" << i << " = [" << colors11[i] << "," << colors12[i] << "," << colors13[i] << "]; color newcolor" << i << ", (" << obj1 << "//" << domain1 << "/" << original_res1[All_Res1[i]].res;
			if(original_res1[All_Res1[i]].ins != ' '){
				outfile << original_res1[All_Res1[i]].ins; 
			}
			outfile << "/*)" << endl;
		}
		outfile << "color white, (" << obj2 << "//" << domain2 << "/*/*)" << endl;
		for(int i=0; i<s2.dim1(); i++){
			outfile << "set_color newcolorA" << i << " = [" << colors21[i] << "," << colors22[i] << "," << colors23[i] << "]; color newcolorA" << i << ", (" << obj2 << "//" << domain2 << "/" << original_res2[All_Res2[i]].res;
			if(original_res2[All_Res2[i]].ins != ' '){
				outfile << original_res2[All_Res2[i]].ins;
			}
			outfile << "/*)" << endl;
		}
        outfile.close();
		cout << endl << "Colour script written to: " << file;
    } else {
        cout << "Unable to open output file " << file << " for writing" << endl;
    }
   
   //write chimera colour script
   string chimera_file = get_folder(file) + "Chimera/" + get_filename(file) + ".com";
   outfile.open(chimera_file.c_str());
   if(outfile.is_open()){
      outfile << "# ProSMART Colour File - Chimera Command Script\n";
      outfile << "color white prosmart1; color white prosmart2; ";
      for(int i=0; i<s1.dim1(); i++){
         outfile << "select prosmart1 & :" << original_res1[All_Res1[i]].res;
         if(original_res1[All_Res1[i]].ins != ' '){
				outfile << original_res1[All_Res1[i]].ins; 
			}
			outfile << "; color " << colors11[i] << "," << colors12[i] << "," << colors13[i] << " sel; ";
		}
      for(int i=0; i<s2.dim1(); i++){
         outfile << "select prosmart2 & :" << original_res2[All_Res2[i]].res;
         if(original_res2[All_Res2[i]].ins != ' '){
				outfile << original_res2[All_Res2[i]].ins; 
			}
         outfile << "; color " << colors21[i] << "," << colors22[i] << "," << colors23[i] << " sel; ";
		}
      outfile << "~select";
      outfile.close();
		cout << endl << "Colour script written to: " << chimera_file;
   } else {
      cout << "Unable to open output file " << chimera_file << " for writing" << endl;
   }
    
    return;
}

Transform write_full_pdb(vector<string> &info, Residues &r1, Residues &r2, vector<residue_alignment> &Res_align, string filein1, string fileout1, char domain1, string filein2, string fileout2, char domain2, bool OUT_PDB_FULL, bool USE_SIDE)
{
    //string tempfileout1 = file_append_name(fileout1,"_");
    //string tempfileout2 = file_append_name(fileout2,"_");
    
	Transform transf = get_transf(r1,r2,Res_align,USE_SIDE);
	if(fileout1.size()>0){		//first chain not needed for clusters
		output_pdb(info,filein1,fileout1,domain1, OUT_PDB_FULL);	//whole chain, protein 1
	}
    output_pdb(info,filein2,fileout2,domain2,transf, OUT_PDB_FULL);	//whole chain, protein 2
    
    return transf;
}

Transform get_transf(Residues &r1, Residues &r2, vector<residue_alignment> &Res_align, bool USE_SIDE)
{
	Transform result;
	Coords c1;
	Coords c2;
    
	if(Res_align.size()==0){	//if alignment is length zero then do nothing.
		result.create();
	} else {
      if(USE_SIDE){
        get_coords_inc_side(r1,r2,Res_align,c1,c2);
      } else {
         c1 = r1.crds1(Res_align);
         c2 = r2.crds2(Res_align);
      }
		result = get_transf(c1,c2);
	}
      
    return result;
}

void get_coords_inc_side(Residues &res1, Residues &res2, vector<residue_alignment> &Res_align, Coords &c1, Coords &c2)
{
    //accounts for fact that side chain atoms may be in different orders. Does not fix nomenclature errors or flips.
    c1.clear();
    c2.clear();
    
    for(unsigned int i=0; i<Res_align.size(); i++){
        //Get list of atoms in residue
        vector<string> side_atoms1 = res1.side_atoms_idx(Res_align[i].res1);
        vector<string> side_atoms2 = res2.side_atoms_idx(Res_align[i].res2);
        side_atoms1.insert(side_atoms1.begin()," O  ");
        side_atoms2.insert(side_atoms2.begin()," O  ");
        side_atoms1.insert(side_atoms1.begin()," C  ");
        side_atoms2.insert(side_atoms2.begin()," C  ");
        side_atoms1.insert(side_atoms1.begin()," CA ");
        side_atoms2.insert(side_atoms2.begin()," CA ");
        side_atoms1.insert(side_atoms1.begin()," N  ");
        side_atoms2.insert(side_atoms2.begin()," N  ");
        
        //get coords corresponding to atoms
        Coords side1 = res1.crd_side_idx(Res_align[i].res1,1);
        Coords side2 = res2.crd_side_idx(Res_align[i].res2,1);
        
        //require atoms to have the same nomenclature.
        //remove atoms not found in both side chains - modify 'side1'.
        //identify corresponding atoms - modify 'side2'.
        //store atom names corresponding to coordinates 'side1' and 'side2' in 'atoms'.
        Coords side1_temp;
        Coords side2_temp;
        vector<string> atoms;
        vector<int> side_idx;
        if(side_atoms1 != side_atoms2){		//either there are missing atoms, or the atoms are in a different order in the PDB file.
            for(unsigned int i=0; i<side_atoms1.size(); i++){
                side_idx.push_back(-1);			//fill vector with false values (-1 in this case)
            }
            for(unsigned int i=0; i<side_atoms1.size(); i++){
                for(unsigned int j=0; j<side_atoms2.size(); j++){
                    if(side_atoms1[i]==side_atoms2[j]){     //require the atom names to be identical
                        side_idx[i] = j;
                        break;
                    }
                }
            }
            for(unsigned int i=0; i<side_idx.size(); i++){
                if(side_idx[i]>=0){ //atom present in both side chains
                    //atoms.push_back(side_atoms1[i]);
                    side1_temp.add(side1.get_crd(i));
                    side2_temp.add(side2.get_crd(side_idx[i]));
                }
            }
            side1 = side1_temp;
            side2 = side2_temp;
        } /*else {
            //atoms = side_atoms1;
        }*/
        
        if(side1.size()>0){
            c1.append(side1);
            c2.append(side2);
        }
    }
    
    return;
}

//This outputs pdb, but doesn't transform any coords.
void output_pdb(vector<string> &info, string filein, string fileout, char chain, bool OUT_PDB_FULL) 
{
    Transform transform;
    output_pdb(info, filein, fileout, chain, transform, 0, OUT_PDB_FULL);     //transform is 'empty', but is not used since DO_TRANSFORM=0.
    return;
}

//This function outputs a whole protein, in a defined coordinate frame specified
//by a transformation. This is used for placing the whole protein chain in the
//coordinate frame of a superimposed cluster.
void output_pdb(vector<string> &info, string filein, string fileout, char chain, Transform &transform, bool OUT_PDB_FULL) 
{
    output_pdb(info, filein, fileout, chain, transform, 1, OUT_PDB_FULL);
    return;
}

void output_pdb(vector<string> &info, string &filein, string &fileout, char chain, Transform &transform, bool DO_TRANSFORM, bool OUT_PDB_FULL) 
{ 
    string line;
    ifstream infile(filein.c_str(), ios::in);
    ofstream outfile;
    if(infile.is_open()){
        outfile.open(fileout.c_str());
		if(outfile.is_open()){
			outfile << "# ProSMART PDB File" << endl;
			for(unsigned int i=0; i<info.size(); i++){
				outfile << info[i] << endl;
			}
            while(!infile.eof()){
                line.clear();
                getline(infile,line);
                
                if(line.size()<66)continue;                     //minimum length to populate a 'pdbline' entry
                
                if(!OUT_PDB_FULL && chain!=' ' && line[21]!=chain)continue;		//only take this chain, unless chain is not specified
                
                if(line.substr(0,6)!="ATOM  "){
                    if(line.substr(0,6)!="HETATM"){
                        continue;
                    } //else if(line.substr(17,3)!="MSE")continue;
                }
                
                /*stringstream ss;      //this renames all residue numbers by -600 (useful if need to combine two chains into one)
                 ss.str("");
                 ss << (atoi(line.substr(22,4).c_str())-600);
                 while(ss.str().size()!=3)ss << " ";
                 line.replace(23,3,ss.str());*/
                /*stringstream ss;      //this renumbers all atom names (useful if need to reorder chains)
                 ss.str("");
                 idx++;
                 ss << idx;
                 while(ss.str().size()!=4)ss << " ";
                 line.replace(7,4,ss.str());*/
                
                if(DO_TRANSFORM){
                    coord crd_orig, crd_new;
                    crd_orig.x = atof(line.substr(30,8).c_str());
                    crd_orig.y = atof(line.substr(38,8).c_str());
                    crd_orig.z = atof(line.substr(46,8).c_str());
                    crd_new = transform.transform(crd_orig);            //transform coords
                    
                    string tempx = standardise_double(d_to_str(crd_new.x),8,3);              //format coords correctly
                    string tempy = standardise_double(d_to_str(crd_new.y),8,3);
                    string tempz = standardise_double(d_to_str(crd_new.z),8,3);
                    
                    line.replace(30,24,"                        ");
                    line.replace(38-tempx.size(),tempx.size(),tempx);   //replace only coords in the line
                    line.replace(46-tempy.size(),tempy.size(),tempy);
                    line.replace(54-tempz.size(),tempz.size(),tempz);
                }
                
                outfile << line << endl;
            }
			outfile.close();
			cout << endl << "PDB written to: " << fileout.c_str();
        }
		else cout << "Unable to open output file " << fileout.c_str() << " for writing" << endl;
        infile.close();
    } else {
        cout << "Unable to open " << filein << " for reading" << endl;
    }
	
    return;
}

void write_alignment(Residues &res1, Residues &res2, vector<string> &info, vector<residue_alignment> &final_residue_alignment, string fileout_string, vector<res_corresp> &original_res1, vector<res_corresp> &original_res2, bool DISPLAY_AS_COSINE_DISTANCE)
{
    ofstream outfile;
    int idx1;
    int idx2;
    double costheta_to_degrees = 45.0/atan(1.0);
	
#ifdef OUTPUT_JS_FILES
    string js_file = fileout_string+".js";
    outfile.open(js_file.c_str());
    if(outfile.is_open()){
        outfile.precision(4);
        outfile << "var dat = new Array();" << endl;
		for(unsigned int i=0; i<final_residue_alignment.size(); i++){
			idx1 = final_residue_alignment[i].res1;
			idx2 = final_residue_alignment[i].res2;
            outfile << "dat[" << i << "] = {";
            
            outfile << "Res1:'" << original_res1[idx1].res << original_res1[idx1].ins << "',Res2:'"
            << original_res2[idx2].res << original_res2[idx2].ins << "',AA1:'";
			
			//amino acid types
			outfile << res1.get_resid_indexed(idx1) << "',AA2:'"
			<< res2.get_resid_indexed(idx2) << "',Type1:'";
			
			//fragment types
			outfile << final_residue_alignment[i].type1 << "',Type2:'"
            << final_residue_alignment[i].type2 << "',SideRMS:+'";
			
			//side chain scores
			if(final_residue_alignment[i].dist3 < 0){
				outfile << "NA',SideAV:+'";
			} else {
				outfile << final_residue_alignment[i].dist3 << "',SideAV:+'";
			}
			if(final_residue_alignment[i].dist4 < 0){
				outfile << "NA',Min:+'";
			} else {
				outfile << final_residue_alignment[i].dist4 << "',Min:+'";
			}
			
			//backbone scores
			outfile << final_residue_alignment[i].dist << "',Central:+'";
			if(final_residue_alignment[i].dist2 < 0){	
				outfile << "NA',Rotate:+'NA";
			} else {
				outfile << final_residue_alignment[i].dist2 << "',Rotate:+'";
                if(DISPLAY_AS_COSINE_DISTANCE){
                    outfile << decimal_places(final_residue_alignment[i].rot,5);
                } else {
                    outfile << decimal_places(acos(1-final_residue_alignment[i].rot)*costheta_to_degrees,2);
                }
			}
            
            //more side chain scores
            outfile << "',MaxDist:+'";
			if(final_residue_alignment[i].dist5 < 0){
				outfile << "NA";
			} else {
				outfile << final_residue_alignment[i].dist5;
			}
            
			outfile << "'};" << endl;
        }
        outfile << "window.data = [];" << endl;
        outfile << "window.data[0] = {align:dat};" << endl;
        outfile.close();
    }
#endif
    
    outfile.open(fileout_string.c_str());
    if(outfile.is_open()){
		outfile << "# ProSMART Alignment File" << endl;
		for(unsigned int i=0; i<info.size(); i++){
			outfile << info[i] << endl;
		}
		outfile << "Res1\tRes2\tAA1\tAA2\tType1\tType2\tFlexi\tProcr\tHinge\tSideRMS\tSideAV\tMaxDist"
#ifdef INCLUDE_SCORE_NUMDIST
        << "\tNumDist"
#endif
#ifdef RUN_AREAIMOL
        << "\tASA1\tASA2"
#endif
#ifdef INCLUDE_SCORE_NUMTOTAL
       << "\tNumTot"
#endif
#ifdef OUTPUT_RESIDUE_COMPLETENESS
        << "\tC1\tC2"
#endif
        << endl;
        outfile.precision(4);
		for(unsigned int i=0; i<final_residue_alignment.size(); i++){
			idx1 = final_residue_alignment[i].res1;
			idx2 = final_residue_alignment[i].res2;
			
			//residue indexes
			outfile << original_res1[idx1].res << original_res1[idx1].ins << "\t"
            << original_res2[idx2].res << original_res2[idx2].ins << "\t";
			
			//average centroid size
			/*if(final_residue_alignment[i].dist2 < 0){
             outfile << "NA\tNA\t";
             } else {
             outfile << final_residue_alignment[i].other[0] << "\t"
             << final_residue_alignment[i].other[1] << "\t";
             }*/
			
			//average B-factors
			/*outfile << res1.get_bfact(idx1) << "\t"
             << res2.get_bfact(idx2) << "\t";*/
			
			//amino acid types
			outfile << res1.get_resid_indexed(idx1) << "\t"
			<< res2.get_resid_indexed(idx2) << "\t";
			
			//fragment types
			outfile << final_residue_alignment[i].type1 << "\t"
            << final_residue_alignment[i].type2 << "\t";
			
         //backbone scores
			outfile << final_residue_alignment[i].dist << "\t";
			if(final_residue_alignment[i].dist2 < 0){	
				outfile << "NA\tNA\t";
			} else {
				outfile << final_residue_alignment[i].dist2 << "\t";
            if(DISPLAY_AS_COSINE_DISTANCE){
               outfile << decimal_places(final_residue_alignment[i].rot,5) << "\t";
            } else {
               outfile << decimal_places(acos(1-final_residue_alignment[i].rot)*costheta_to_degrees,2) << "\t";
            }
			}
         
			//side chain scores
			if(final_residue_alignment[i].dist3 < 0){
				outfile << "NA\t";
			} else {
				outfile << final_residue_alignment[i].dist3 << "\t";
			}
			if(final_residue_alignment[i].dist4 < 0){
				outfile << "NA\t";
			} else {
				outfile << final_residue_alignment[i].dist4 << "\t";
			}
			if(final_residue_alignment[i].dist5 < 0){
				outfile << "NA";
			} else {
				outfile << final_residue_alignment[i].dist5;
			}
            
#ifdef INCLUDE_SCORE_NUMDIST
			if(final_residue_alignment[i].dist6 < 0){
				outfile << "\tNA";
			} else {
				outfile << "\t" << final_residue_alignment[i].dist6;
			}
#endif
#ifdef RUN_AREAIMOL
            outfile << "\t" << final_residue_alignment[i].asa1;
            outfile << "\t" << final_residue_alignment[i].asa2;
#endif
#ifdef INCLUDE_SCORE_NUMTOTAL
         if(final_residue_alignment[i].dist7 < 0){
				outfile << "\tNA";
			} else {
				outfile << "\t" << final_residue_alignment[i].dist7;
			}
#endif
#ifdef OUTPUT_RESIDUE_COMPLETENESS
            outfile << "\t" << final_residue_alignment[i].complete1;
            outfile << "\t" << final_residue_alignment[i].complete2;
#endif
			outfile << endl;
		}
		cout << endl << "Alignment file written to: " << fileout_string;
		outfile.close();
    } else {
		cout << endl << "Could not write alignment file: " << fileout_string;
	}
	
    return;
}

void write_residue_types(vector<string> &info, vector<residue_alignment> &final_residue_alignment, string fileout_string, vector<res_corresp> &original_res1)
{
    ofstream outfile;
    int idx1 = 0;
	int idx2 = 0;
	
    outfile.open(fileout_string.c_str());
    outfile.precision(4);
	if(outfile.is_open()){
		outfile << "# ProSMART Fragment Scores File" << endl;
		outfile << "# Res\tScore" << endl;
        for(unsigned int i=0; i<final_residue_alignment.size(); i++){
			idx1 = final_residue_alignment[i].res1;
			
			while(idx2 < idx1){
				if(original_res1[idx2].res != -1){
					outfile << original_res1[idx2].res << original_res1[idx2].ins << "\t-1" << endl;
				}
				idx2++;
			}
			
			outfile << original_res1[idx1].res << original_res1[idx1].ins << "\t"
			<< final_residue_alignment[i].dist;
			outfile << endl;
			idx2++;
		}
		while(idx2 < (int)original_res1.size()){	//ensure that all files pertaining to this target are same length - always reaches the last residue.
			if(original_res1[idx2].res != -1){
				outfile << original_res1[idx2].res << original_res1[idx2].ins << "\t-1" << endl;
			}
			idx2++;
		}		
		cout << endl << "Scores file written to: " << fileout_string;
		outfile.close();
    } else {
		cout << endl << "Could not write residue type file: " << fileout_string;
	}
    return;
}

vector<string> get_orig_res1(vector<residue_alignment> &resalign, vector<res_corresp> &orig_res)
{
    vector<string> result;
    stringstream ss;
    string empty;
    
    for(unsigned int i=0; i<resalign.size(); i++){
        empty="";
        ss << orig_res[resalign[i].res1].res << orig_res[resalign[i].res1].ins;
        result.push_back(ss.str());
        ss.str(empty);
    }
    
    return result;
}

vector<string> get_orig_res2(vector<residue_alignment> &resalign, vector<res_corresp> &orig_res)
{
    vector<string> result;
    stringstream ss;
    string empty;
    
    for(unsigned int i=0; i<resalign.size(); i++){
        empty="";
        ss << orig_res[resalign[i].res2].res << orig_res[resalign[i].res2].ins;
        result.push_back(ss.str());
        ss.str(empty);
    }
    
    return result;
}

vector<int> get_resnum_corresp(vector<string> &res_list, vector<string> &align_list)
{
    vector<int> result;
    
    unsigned int align_idx=0;
    for(unsigned int i=0; i<res_list.size(); i++){
        if(align_list.size()>align_idx){
            if(res_list[i]==align_list[align_idx]){
                result.push_back(align_idx);
                align_idx++;
            } else {
                result.push_back(-1);
            } 
        } else {
            result.push_back(-1);
        }
    }
    
    /*for(int i=0; i<res_list.size(); i++){
     cout << endl << i << " " << res_list[i] << "\t" << result[i] << " ";
     if(result[i] >= 0){
     cout << align_list[result[i]];
     }
     }*/
    
    return result;
}


