//
// This file is part of ProSMART.
//

#include "prosmartClass_Transform.h"

bool USE_SIEVE = 0;
double SIEVE_PARAMETER = 0.5;

Array2D<double> reflectM(3,3);	//global variable
Array2D<double> identityM(3,3);	//global variable

void initialise_identityM()
{
	identityM[0][0] = 1;
	identityM[0][1] = 0;
	identityM[0][2] = 0;
	identityM[1][0] = 0;
	identityM[1][1] = 1;
	identityM[1][2] = 0;
	identityM[2][0] = 0;
	identityM[2][1] = 0;
	identityM[2][2] = 1;
	return;
}

void initialise_reflectM()
{
	reflectM[0][0] = 1;
	reflectM[0][1] = 0;
	reflectM[0][2] = 0;
	reflectM[1][0] = 0;
	reflectM[1][1] = 1;
	reflectM[1][2] = 0;
	reflectM[2][0] = 0;
	reflectM[2][1] = 0;
	reflectM[2][2] = -1;
	return;
}

Array2D<double> get_identityM()
{
	return identityM;
}

ostream& operator<<(ostream& out, Transform& transform)
{
    coord t = transform.get_t();
    Array2D<double> r = transform.get_r();
    coord p = transform.get_p();
	coord f = transform.get_f();
	coord pre = transform.get_pre();
    cout << endl;
    cout << "Pre-translation:   " << t;
    cout << endl;
	cout << "Post-translation:  " << p;
    cout << endl;
	cout << "Final pre-translation: " << pre;
    cout << endl;
    cout << "Final post-translation: " << f;
    cout << endl;
	cout << "Rotation: " << endl;
	for(int i=0; i<3; i++){
		cout << r[i][0] << " " << r[i][1] << " " << r[i][2] << endl;
	}
    cout << endl;
    cout << "Angle: " << transform.get_r_angle() << endl;
    
    return out;
}

vector<Transform> coord_transf(vector<Coords> &crds)
{
  vector<Transform> result;
  Transform transf1;
  Transform transf2;
  coord mean1;
  coord mean2;
  Coords crds1rm;
  Coords crds2rm;
  Array2D<double> RM(3,3);
  
  mean1 = crds[0].mean();
  mean2 = crds[1].mean();
  crds1rm = crds[0] - mean1;
  crds2rm = crds[1] - mean2;
  
  RM = rotateM(crds1rm,crds2rm);

  transf1.create(mean1);
  transf2.create(mean2,RM);
  
  result.push_back(transf1);
  result.push_back(transf2);
  
  return result;
}

Transform get_transf(Coords c1, Coords c2)
{
   Transform result;
   //check whether coords are identical, if so return identity
   double rmsd_tmp = RMSD(c1,c2);
   if(rmsd_tmp<0.00001){
      result.create();
      return result;
   }
   
   coord mean1 = c1.mean();
   coord mean2 = c2.mean();
   Coords c1n = c1-mean1;
   Coords c2n = c2-mean2;
   Array2D<double> RM = rotateM(c1n,c2n);
   
   if(USE_SIEVE){
      Coords c2nR = c2n*RM;
      rmsd_tmp = RMSD(c1n,c2nR);
      if(rmsd_tmp>SIEVE_PARAMETER){
         cout << endl << endl << "Sieve-fitting to optimise superposition...\nInitial RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms";
         while(rmsd_tmp>SIEVE_PARAMETER){
            unsigned int maxi = 0;
            double maxd = dist(c1n.get(0),c2nR.get(0));
            for(int i=1; i<c1n.size(); i++){
               double d_tmp = dist(c1n.get(i),c2nR.get(i));
               if(d_tmp>maxd){
                  maxd = d_tmp;
                  maxi = i;
               }
            }
            c1.erase(maxi);
            c2.erase(maxi);
            mean1 = c1.mean();
            mean2 = c2.mean();
            c1n = c1-mean1;
            c2n = c2-mean2;
            RM = rotateM(c1n,c2n);
            c2nR = c2n*RM;
            rmsd_tmp = RMSD(c1n,c2nR);
            //cout << endl << "i: " << maxi << "\tdist: " << maxd << "\tRMSD: " << rmsd_tmp;
         }
         cout << endl << "Final RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms" << endl; 
      } else {
         cout << endl << "Sieve-fitting not required - RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms";
      }
   }   
   result.create(mean2,RM,mean1);		//translate to origin, rotate to agree with chain1, translate onto chain1.
   return result;
}

vector<unsigned int> sieve_fit(Coords &c1, Coords &c2)
{
   vector<unsigned int> result;
   coord mean1 = c1.mean();
   coord mean2 = c2.mean();
   Coords c1n = c1-mean1;
   Coords c2n = c2-mean2;
   Array2D<double> RM = rotateM(c1n,c2n);
   
   Coords c2nR = c2n*RM;
   double rmsd_tmp = RMSD(c1n,c2nR);
   if(rmsd_tmp>SIEVE_PARAMETER){
      cout << endl << endl << "Sieve-fitting to optimise superposition...\nInitial RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms";
      while(RMSD(c1n,c2nR)>SIEVE_PARAMETER){
         unsigned int maxi = 0;
         double maxd = dist(c1n.get(0),c2nR.get(0));
         for(int i=1; i<c1n.size(); i++){
            double d_tmp = dist(c1n.get(i),c2nR.get(i));
            if(d_tmp>maxd){
               maxd = d_tmp;
               maxi = i;
            }
         }
         c1.erase(maxi);
         c2.erase(maxi);
         result.push_back(maxi);
         mean1 = c1.mean();
         mean2 = c2.mean();
         c1n = c1-mean1;
         c2n = c2-mean2;
         RM = rotateM(c1n,c2n);
         c2nR = c2n*RM;
         rmsd_tmp = RMSD(c1n,c2nR);
         //cout << endl << "i: " << maxi << "\tdist: " << maxd << "\tRMSD: " << rmsd_tmp;
      }
      cout << endl << "Final RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms" << endl; 
   } else {
      cout << endl << "Sieve-fitting not required - RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms";
   }
   
   return result;
}

Transform get_transf(Coords c1, Coords c2, Array2D<double> &RM)
//force the use of a particular rotation matrix; optimise translation
{
   coord mean1 = c1.mean();
   coord mean2 = c2.mean();
   Coords c1n = c1-mean1;
   Coords c2n = c2-mean2;
   
   if(USE_SIEVE){
      Coords c2nR = c2n*RM;
      double rmsd_tmp = RMSD(c1n,c2nR);
      if(rmsd_tmp>SIEVE_PARAMETER){
         cout << endl << endl << "Sieve-fitting to optimise superposition...\nInitial RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms";
         while(RMSD(c1n,c2nR)>SIEVE_PARAMETER){
            unsigned int maxi = 0;
            double maxd = dist(c1n.get(0),c2nR.get(0));
            for(int i=1; i<c1n.size(); i++){
               double d_tmp = dist(c1n.get(i),c2nR.get(i));
               if(d_tmp>maxd){
                  maxd = d_tmp;
                  maxi = i;
               }
            }
            c1.erase(maxi);
            c2.erase(maxi);
            mean1 = c1.mean();
            mean2 = c2.mean();
            c1n = c1-mean1;
            c2n = c2-mean2;
            c2nR = c2n*RM;
            rmsd_tmp = RMSD(c1n,c2nR);
            //cout << endl << "i: " << maxi << "\tdist: " << maxd << "\tRMSD: " << rmsd_tmp;
         }
         cout << endl << "Final RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms" << endl; 
      } else {
         cout << endl << "Sieve-fitting not required - RMSD: " << rmsd_tmp << " using " << c1.size() << " atoms";
      }
   }
   
   Transform result;
   result.create(mean2,RM,mean1);		//translate to origin, rotate to agree with chain1, translate onto chain1.
   
   return result;
}

vector<quarternion> get_quarternions(vector<Array2D<double> > &RM)
{
	vector<quarternion> result;
	quarternion q;
	double tmp1;
	double tmp2;
	double tmp3;
	double tmp4;
	for(unsigned int i=0; i<RM.size(); i++){
		tmp1 = 1 + RM[i][0][0] + RM[i][1][1] + RM[i][2][2];
		tmp2 = 1 + RM[i][0][0] - RM[i][1][1] - RM[i][2][2];
		tmp3 = 1 - RM[i][0][0] + RM[i][1][1] - RM[i][2][2];
		tmp4 = 1 - RM[i][0][0] - RM[i][1][1] + RM[i][2][2];
		if(tmp1 > tmp2 && tmp1 > tmp3 && tmp1 > tmp4){
			if(tmp1<0.0){tmp1=0.0;}
			q.q1 = 0.5*sqrt(tmp1);
			q.q2 = (RM[i][2][1] - RM[i][1][2])/(4*q.q1);
			q.q3 = (RM[i][0][2] - RM[i][2][0])/(4*q.q1);
			q.q4 = (RM[i][1][0] - RM[i][0][1])/(4*q.q1);
		} else if(tmp2 > tmp3 && tmp2 > tmp4){
			if(tmp2<0.0){tmp2=0.0;}
			q.q2 = 0.5*sqrt(tmp2);
			q.q1 = (RM[i][2][1] - RM[i][1][2])/(4*q.q2);
			q.q3 = (RM[i][1][0] + RM[i][0][1])/(4*q.q2);
			q.q4 = (RM[i][0][2] + RM[i][2][0])/(4*q.q2);
		} else if(tmp3 > tmp4){
			if(tmp3<0.0){tmp3=0.0;}
			q.q3 = 0.5*sqrt(tmp3);
			q.q1 = (RM[i][0][2] - RM[i][2][0])/(4*q.q3);
			q.q2 = (RM[i][1][0] + RM[i][0][1])/(4*q.q3);
			q.q4 = (RM[i][2][1] + RM[i][1][2])/(4*q.q3);
		} else{
			if(tmp4<0.0){tmp4=0.0;}
			q.q4 = 0.5*sqrt(tmp4);
			q.q1 = (RM[i][1][0] - RM[i][0][1])/(4*q.q4);
			q.q2 = (RM[i][0][2] + RM[i][2][0])/(4*q.q4);
			q.q3 = (RM[i][2][1] + RM[i][1][2])/(4*q.q4);
		}

		/*//old method:
		 tmp1 = 0.5*sqrt( 1 + RM[i][0][0] - RM[i][1][1] - RM[i][2][2] );
		 //tmp4 = 0.5*sqrt( 1 + RM[i][0][0] + RM[i][1][1] + RM[i][2][2] );
		 //if(tmp1 > tmp4){
		 q.q1 = tmp1;
		 q.q2 = (RM[i][0][1] + RM[i][1][0])/(4*q.q1);
		 q.q3 = (RM[i][0][2] + RM[i][2][0])/(4*q.q1);
		 q.q4 = (RM[i][2][1] - RM[i][1][2])/(4*q.q1);
		 } else {
		 q.q4 = tmp4;
		 q.q1 = (RM[i][1][2] - RM[i][2][1])/(4*q.q4);
		 q.q2 = (RM[i][2][0] - RM[i][0][2])/(4*q.q4);
		 q.q3 = (RM[i][0][1] - RM[i][1][0])/(4*q.q4);
		 }*/
		
		result.push_back(q);
	}
	return result;
}

quarternion average_quarternion(vector<quarternion> &quart, vector<int> &cluster)
{
	quarternion result;
	quarternion tmp;
	
	tmp.q1 = 0.0;
	tmp.q2 = 0.0;
	tmp.q3 = 0.0;
	tmp.q4 = 0.0;
	for(unsigned int i=0; i<cluster.size(); i++){
		tmp.q1 += quart[cluster[i]].q1;
		tmp.q2 += quart[cluster[i]].q2;
		tmp.q3 += quart[cluster[i]].q3;
		tmp.q4 += quart[cluster[i]].q4;
	}
	
	double norm = sqrt(tmp.q1*tmp.q1 + tmp.q2*tmp.q2 + tmp.q3*tmp.q3 + tmp.q4*tmp.q4);
	
	result.q1 = tmp.q1/norm;
	result.q2 = tmp.q2/norm;
	result.q3 = tmp.q3/norm;
	result.q4 = tmp.q4/norm;

	return result;
}

quarternion average_quarternion(vector<quarternion> &quart)
{
	quarternion result;
	quarternion tmp;
	
	tmp.q1 = 0.0;
	tmp.q2 = 0.0;
	tmp.q3 = 0.0;
	tmp.q4 = 0.0;
	for(unsigned int i=0; i<quart.size(); i++){
		tmp.q1 += quart[i].q1;
		tmp.q2 += quart[i].q2;
		tmp.q3 += quart[i].q3;
		tmp.q4 += quart[i].q4;
	}
	
	double norm = sqrt(tmp.q1*tmp.q1 + tmp.q2*tmp.q2 + tmp.q3*tmp.q3 + tmp.q4*tmp.q4);
	
	result.q1 = tmp.q1/norm;
	result.q2 = tmp.q2/norm;
	result.q3 = tmp.q3/norm;
	result.q4 = tmp.q4/norm;
    
	return result;
}

vector<quarternion> get_average_quarternions(vector<quarternion> &quart, vector<vector<int> > &clusters)
{
	vector<quarternion> result;
	
	for(unsigned int i=0; i<clusters.size(); i++){
		result.push_back(average_quarternion(quart,clusters[i]));
	}	
	
	return result;
}

quarternion multiply_quarternions(quarternion &A, quarternion &B)
{
	quarternion result;
	quarternion tmp;
	
	tmp.q1 = A.q1*B.q1 - A.q2*B.q2 - A.q3*B.q3 - A.q4*B.q4;
	tmp.q2 = A.q1*B.q2 + A.q2*B.q1 + A.q3*B.q4 - A.q4*B.q3;
	tmp.q3 = A.q1*B.q3 - A.q2*B.q4 + A.q3*B.q1 + A.q4*B.q2;
	tmp.q4 = A.q1*B.q4 + A.q2*B.q3 - A.q3*B.q2 + A.q4*B.q1;

	double norm = sqrt(tmp.q1*tmp.q1 + tmp.q2*tmp.q2 + tmp.q3*tmp.q3 + tmp.q4*tmp.q4);

	result.q1 = tmp.q1/norm;
	result.q2 = tmp.q2/norm;
	result.q3 = tmp.q3/norm;
	result.q4 = tmp.q4/norm;
	
	return result;
}

quarternion multiply_quarternion_conj(quarternion &A, quarternion &B)
{
	quarternion result;
	quarternion tmp;
	
	tmp.q1 = A.q1*B.q1 + A.q2*B.q2 + A.q3*B.q3 + A.q4*B.q4;
	tmp.q2 = A.q1*B.q2 - A.q2*B.q1 - A.q3*B.q4 + A.q4*B.q3;
	tmp.q3 = A.q1*B.q3 + A.q2*B.q4 - A.q3*B.q1 - A.q4*B.q2;
	tmp.q4 = A.q1*B.q4 - A.q2*B.q3 + A.q3*B.q2 - A.q4*B.q1;

	double norm = sqrt(tmp.q1*tmp.q1 + tmp.q2*tmp.q2 + tmp.q3*tmp.q3 + tmp.q4*tmp.q4);

	result.q1 = tmp.q1/norm;
	result.q2 = tmp.q2/norm;
	result.q3 = tmp.q3/norm;
	result.q4 = tmp.q4/norm;
	
	return result;
}


Array2D<double> get_rot_from_quart_single(quarternion &q)
{
	Array2D<double> RM(3,3);

	RM[0][0] = 1 - 2*q.q3*q.q3 - 2*q.q4*q.q4;
	RM[1][1] = 1 - 2*q.q2*q.q2 - 2*q.q4*q.q4;
	RM[2][2] = 1 - 2*q.q2*q.q2 - 2*q.q3*q.q3;
	RM[0][1] = 2*(q.q2*q.q3 - q.q4*q.q1);
	RM[0][2] = 2*(q.q2*q.q4 + q.q3*q.q1);
	RM[1][0] = 2*(q.q2*q.q3 + q.q4*q.q1);
	RM[1][2] = 2*(q.q3*q.q4 - q.q1*q.q2);
	RM[2][0] = 2*(q.q2*q.q4 - q.q3*q.q1);
	RM[2][1] = 2*(q.q3*q.q4 + q.q2*q.q1);
	
	/*//old method:
	RM[0][0] = 1 - 2*q.q2*q.q2 - 2*q.q3*q.q3;
	RM[1][1] = 1 - 2*q.q1*q.q1 - 2*q.q3*q.q3;
	RM[2][2] = 1 - 2*q.q1*q.q1 - 2*q.q2*q.q2;
	RM[0][1] = 2*(q.q1*q.q2 - q.q3*q.q4);
	RM[0][2] = 2*(q.q1*q.q3 + q.q2*q.q4);
	RM[1][0] = 2*(q.q1*q.q2 + q.q3*q.q4);
	RM[1][2] = 2*(q.q2*q.q3 - q.q1*q.q4);
	RM[2][0] = 2*(q.q1*q.q3 - q.q2*q.q4);
	RM[2][1] = 2*(q.q1*q.q4 + q.q2*q.q3);*/
	
	return RM;
}

vector<Array2D<double> > get_rot_from_quart(vector<quarternion> &q)
{
	vector<Array2D<double> > result;
	for(unsigned int i=0; i<q.size(); i++){
		result.push_back(get_rot_from_quart_single(q[i]));
	}
	return result;
}

void Transform::create(Coords &f1, Coords &f2, coord &m1, coord &m2)
//this is specifically for storing info about aligned fragments, ready for clustering.
{
  translate = m1;
	post_translate = m2;
  rotate = rotateM(f1,f2);
	final_translation = translate*rotate;
	final_translation = post_translate - final_translation;
  return;
}

void Transform::create(coord trans, Array2D<double> rot)
{
  translate = trans;
  rotate = rot;
  post_translate.x = 0.0;
  post_translate.y = 0.0;
  post_translate.z = 0.0;
	final_translation = translate*rotate;
	final_translation = post_translate - final_translation;
  return;
}

void Transform::create(coord trans)
{
  translate = trans;
  rotate = identityM;			//global var
  post_translate.x = 0.0;
  post_translate.y = 0.0;
  post_translate.z = 0.0;
	final_translation = post_translate - translate;
  return;
}

void Transform::create()
{
  translate.x = 0.0;
  translate.y = 0.0;
  translate.z = 0.0;
	rotate = identityM;			//global var
  post_translate = translate;
	final_translation = translate;
  return;
}

void Transform::create(coord trans, Array2D<double> rot, coord post_trans)
{
  translate = trans;
  rotate = rot;
  post_translate = post_trans;
	final_translation = translate*rotate;
	final_translation = post_translate - final_translation;
  return;
}

void Transform::create(coord trans, coord post_trans)
{
  translate = trans;
  rotate = identityM;			//global var
  post_translate = post_trans;
	final_translation = post_translate - translate;
  return;
}

/*Coords Transform::transform(Coords &crds)
{
  Coords newcrds;
  newcrds = crds - translate;
  newcrds = newcrds*rotate;
  newcrds = newcrds + post_translate;
  return newcrds;
}

coord Transform::transform(coord crd)
{
  coord newcrd;
  newcrd = crd - translate;
  newcrd = newcrd*rotate;
  newcrd = newcrd + post_translate;
  return newcrd;
}*/

Coords Transform::transform(Coords &crds)
{
  Coords newcrds = crds*rotate;
	return (newcrds + final_translation);
}

coord Transform::transform(coord crd)
{
	coord newcrd = crd*rotate;
	return (newcrd + final_translation);
}

Coords Transform::pre_translate(Coords &crds)
{
    Array2D<double> tmp = t(rotate);
    coord tmp1 = final_translation*tmp;
    return (crds + tmp1);
}

double transdist(Coords &crds1,Coords &crds2)
// Returns Euclidean distance between the centers of the two coordinate sets.
{
  coord temp1 = crds1.mean();
  coord temp2 = crds2.mean();
  coord temp3 = temp2-temp1;
  return sqrt(dist2(temp3));
}

Array2D<double> rotateM(Coords& crds1,Coords& crds2)
// Returns matrix for rotating crds2 onto crds1.
// Such that crds1 corresponds to crds2*RM.
// crds1 and crds2 should be normalised (centered at zero) BEFORE being passed to this function.
{
    Array2D<double> temp = crds2*crds1;
    return rotateM(temp);
}

Array2D<double> rotateM(Array2D<double> &temp)
// Returns rotation matrix corresponding to temp.
{
  Array2D<double> SU(3,3);
  Array2D<double> SV(3,3);
  
  SVDuv(temp,SU,SV);
  
  Array2D<double> RM;
  if(det3x3(SU)*det3x3(SV)>=0){
    RM = crossprod_tt(SU,SV);
  } else {
    Array2D<double> rSU = crossprod_tt(SU,reflectM);	//reflectM is defined globally (above)
    RM = crossprod_tt(rSU,SV);
  }

  return RM;
}

coord& Transform::get_t()
{
  return translate;
}

coord& Transform::get_p()
{
  return post_translate;
}

coord& Transform::get_f()
{
  return final_translation;
}

coord Transform::get_pre()
{
    Array2D<double> tmp = t(rotate);
    return final_translation*tmp;
}

Array2D<double>& Transform::get_r()
{
  return rotate;
}

double Transform::get_r_angle()
{
    return get_angle(rotate);
}

void Transform::apply(Transform &transf)
{
    //apply another transformation to the present transformation
    rotate = crossprod(transf.get_r(),rotate);
    post_translate = post_translate - transf.get_t();
    post_translate = post_translate*transf.get_r();
    post_translate = post_translate + transf.get_p();
    final_translation = translate*rotate;
	final_translation = post_translate - final_translation;
    
    /*rotate = crossprod(transf.get_r(),rotate);
    final_translation = final_translation*transf.get_r();
    final_translation = final_translation + transf.get_f();
    coord tR = translate*rotate;
    post_translate = final_translation - tR;*/
    
    return;
}

void Transform::create_diff(Transform &transf1, Transform &transf2)
{
    rotate = crossprod(transf1.get_r(),transf2.get_r());
    translate = transf1.get_p();
    
    post_translate = transf1.get_t()-transf2.get_t();
    post_translate = post_translate*transf2.get_r();
    post_translate = post_translate + transf2.get_p();

    final_translation = translate*rotate;
	final_translation = post_translate - final_translation;
	return;
}

void Transform::set_r(Array2D<double> &R)
{
	rotate = R;
	final_translation = translate*rotate;
	final_translation = post_translate - final_translation;
	return;
}

void Transform::set_r(Array2D<double> &R, coord pre)
{
	rotate = R;
    final_translation = translate+pre;
    post_translate = final_translation*rotate;
	final_translation = translate*rotate;
	final_translation = post_translate - final_translation;
	return;
}

void Transform::set_r_90()
{
	Array2D<double> RM90(3,3);
	RM90[0][0] = 1;
	RM90[0][1] = 0;
	RM90[0][2] = 0;
	RM90[1][0] = 0;
	RM90[1][1] = -1;
	RM90[1][2] = 0;
	RM90[2][0] = 0;
	RM90[2][1] = 0;
	RM90[2][2] = -1;
	
	rotate = rotate*RM90;
  return;
}

Transform get_average_transformation(vector<Transform> &transf_v)
{
    //get rotations
    vector<Array2D<double> > RM_v;        
    for(unsigned int j=0; j<transf_v.size(); j++){
        RM_v.push_back(transf_v[j].get_r());
    }
    
    //get quaternions
    vector<quarternion> quats = get_quarternions(RM_v);
    /*for(unsigned int j=0; j<transf_v.size(); j++){
        //cout << endl << "\t" << j << "\t" << transf_v[j];
        cout << endl << "\t" << j << "\t" << transf_v[j].get_r_angle();
        //cout << endl << "\t" << quats[j].q1 << "\t" << quats[j].q2 << "\t" << quats[j].q3 << "\t" << quats[j].q4;
    }*/
    
    //get average quaternion
    quarternion quat_av = average_quarternion(quats);
    //cout << endl << "\t" << quat_av.back().q1 << "\t" << quat_av.back().q2 << "\t" << quat_av.back().q3 << "\t" << quat_av.back().q4 << endl;
    
    //get rotation corresponding to average quaternion
    Array2D<double> final_RM_v = get_rot_from_quart_single(quat_av);
    
    //get average pre and post translations, and create final transformation
    Coords pre;
    Coords post;
    for(unsigned int j=0; j<transf_v.size(); j++){
        pre.add(transf_v[j].get_t());
        post.add(transf_v[j].get_p());
    }
    Transform result;
    result.create(pre.mean(),final_RM_v,post.mean());
    return result;
}

double Transform::get_r_axis_angle(coord &screw_point, coord &r_axis, double &rot_angle)
{    
    double r_cosangle = ((rotate[0][0] + rotate[1][1] + rotate[2][2])-1.0)/2.0;
    double r_angle = 0.0;
   double norm_fact = 0.0;
    if(r_cosangle<=-1.0){
        r_angle = acos(-1.0);
        rot_angle = 180.0;
    } else if(r_cosangle>=1.0){
        r_angle = 0.0;
        rot_angle = 0.0;
    } else {
        r_angle = acos(r_cosangle);
        //rotation angle (degrees)
        rot_angle = r_angle*45.0/atan(1.0);
       norm_fact = 0.5/sin(r_angle);
    }

    //rotation axis
    r_axis.x = (rotate[1][2]-rotate[2][1])*norm_fact;
    r_axis.y = (rotate[2][0]-rotate[0][2])*norm_fact;
    r_axis.z = (rotate[0][1]-rotate[1][0])*norm_fact;
    
    //length of screw translation along rotation axis
    double screw_trans=r_axis.x*final_translation.x+r_axis.y*final_translation.y+r_axis.z*final_translation.z;
    
    coord rodrigues_vector = r_axis*tan(r_angle/2.0);
    coord tmp_v1 = crossprod(rodrigues_vector,final_translation);
    coord tmp_v2 = crossprod(rodrigues_vector,tmp_v1);
    coord tmp_v3 = tmp_v1 - tmp_v2;
    double tmp_d = 2.0*tr_cov(rodrigues_vector);
    //point on the rotation axis defining orgin of screw
   if(tmp_d==0.0){
      screw_point = r_axis;
   } else {
      screw_point = tmp_v3/tmp_d;
   }
   
    /*Array2D<double> tmp = t(rotate);
     coord rotated_point = tmp*translate;
     rotated_point = screw_point + final_translation;*/
    
    /*bestfit_screw_point.x=0.5*(final_translation.x-(screw_trans*r_axis.x));
    bestfit_screw_point.y=0.5*(final_translation.y-(screw_trans*r_axis.y));
    bestfit_screw_point.z=0.5*(final_translation.z-(screw_trans*r_axis.z));*/
    
    return screw_trans;
}

