//
// This file is part of ProSMART.
//

#include "prosmart_array_tools.h"

double sum(Array1D<double> &A)
{
  double ctr=0;
  for(int i=0; i<A.dim(); i++)
    ctr+=A[i];
  return ctr;
}

double average(Array1D<double> &A)
{
  double result = sum(A);
  result = result/A.dim();
  return result;
}

vector<double> average_v(vector<double> &A, vector<double> &B)
{
  vector<double> result;
	double tmp;
	for(unsigned int i=0; i<A.size(); i++){
		tmp = (A[i]+B[i])/2;
		result.push_back(tmp);
	}
  return result;
}

double stddev(vector<double> &A, double M)
{
  double ctr=0.0;
	double temp=0.0;
  for(unsigned int i=0; i<A.size(); i++){
    temp = (A[i]-M);
		ctr += temp*temp;
	}
  return sqrt(ctr/(A.size()-1));
}

Array1D<double> SV(Array2D<double> &M)
{
  SVD<double> tempsvd(M,0);
  Array1D<double> singval;
  tempsvd.getSingularValues(singval);
  return singval;
}

void SVDuv(Array2D<double> &M, Array2D<double> &u, Array2D<double> &v)
{
  SVD<double> tempsvd(M,1);
  tempsvd.getU(u);
  tempsvd.getV(v);
  return;
}

double get_angle(Array2D<double> &rotate)
{
    double cosine_dist = (3.0 - (rotate[0][0] + rotate[1][1] + rotate[2][2]))/2.0;
    
    if(cosine_dist<0)cosine_dist=0;
    if(cosine_dist>2)cosine_dist=2;
    
    return acos(1-cosine_dist)*45.0/atan(1.0);     // 45.0/atan(1.0) converts radians to degrees
}

Array2D<double> crossprod_tt(Array2D<double> &A, Array2D<double> &B)
{
  int nA = A.dim1();
  int nB = B.dim1();
  Array2D<double> temp(nA,nB);
  int N = A.dim2();
  
  for(int i=0; i<nA; i++){
    for(int j=0; j<nB; j++){
	  temp[i][j] = 0;
	  for(int k=0; k<N; k++){
	    temp[i][j] += A[i][k]*B[j][k];
	  }
	}
  }
  
  return temp;
}

Array2D<double> crossprod_weight(Array2D<double> &A, Array2D<double> &B, vector<double> &w)
{
    int nA = A.dim1();
    int nB = B.dim1();
    Array2D<double> temp(nA,nB);
    int N = A.dim2();
    
    for(int i=0; i<nA; i++){
        for(int j=0; j<nB; j++){
            temp[i][j] = 0;
            for(int k=0; k<N; k++){
                temp[i][j] += A[i][k]*B[j][k]*w[k];
            }
        }
    }
    
    return temp;
}

double tr_cov(const Array2D<double> &A, const Array2D<double> &B)
//fast calculation of trace of covariance matrix between A and B
{
  int m = A.dim1();
  int n = A.dim2();
  double result=0.0;
  
  for(int i=0; i<m; i++){
    for(int j=0; j<n; j++){
	  result += A[i][j]*B[i][j];
	}
  }
  
  return result;
}

template <class T>
void cov(Array2D<T> &M, const Array2D<T> &A, const Array2D<T> &B)
{
  int m = A.dim1();
  int n = A.dim2();
  
  for(int i=0; i<n; i++){
    for(int j=0; j<n; j++){
	  M[i][j] = 0.0;
	  for(int k=0; k<m; k++){
	    M[i][j] += A[k][i]*B[k][j];
	  }
    }
  }
  
  return;
}

double det3x3(const Array2D<double> &M)
{
  double result = 0.0;
  result += M[0][0]*((M[1][1]*M[2][2])-(M[1][2]*M[2][1]));
  result += M[1][0]*((M[2][1]*M[0][2])-(M[2][2]*M[0][1]));
  result += M[2][0]*((M[0][1]*M[1][2])-(M[0][2]*M[1][1]));
  return result;
}

template <class T>
T absol(const T &x1, const T &x2)
{
  if(x1>=x2){
    return x1-x2;
  } else {
    return x2-x1;
  }
}

double absol(const double &x)
{
  if(x>=0){
    return x;
  } else {
    return -x;
  }
}

Array2D<double> absol(Array2D<double> &x)
{
  Array2D<double> result(x.dim1(),x.dim2());
  for(int i=0; i<x.dim1(); i++){
    for(int j=0; j<x.dim2(); j++){
	  if(x[i][j]>=0){
        result[i][j] = x[i][j];
      } else {
        result[i][j] = -x[i][j];
      }
	}
  }
  return result;
}

Array1D<double> decimal_places (Array1D<double> &vect, int dp)
{
  Array1D<double> result(vect.dim1());
  int tempi;
  double tempd;
  
  for(int i=0; i<vect.dim1(); i++){
    tempd = vect[i];
	for(int j=0; j<dp; j++){
      tempd = tempd*10.0;
	}
	tempi = (int)tempd;
	tempd = (double)tempi;
	for(int j=0; j<dp; j++){
      tempd = tempd/10.0;
	}
	result[i] = tempd;
  }
  return result;
}

double decimal_places(double x, int dp)
{
    int tempi;
    double tempd = x;
    for(int j=0; j<dp; j++){
        tempd = tempd*10.0;
    }
    tempi = (int)tempd;
    tempd = (double)tempi;
    for(int j=0; j<dp; j++){
        tempd = tempd/10.0;
    }
    
    return tempd;
}

void fileoutput_array(Array2D<double> matrix1, string fileout, string filename_addon, bool sym)
{
  ofstream outfile;
  char *fileout1;
  
  string newfileout(fileout);
  newfileout.replace(newfileout.size()-3, 3, "arr");
  newfileout.insert(newfileout.size()-4, filename_addon);
  fileout1 = new char [newfileout.size()+1];
  strcpy(fileout1, newfileout.c_str());
  outfile.open(fileout1);
  if(outfile.is_open()){
	if(sym==1){
	  for(int i=0; i<matrix1.dim1(); i++){
	    for(int j=0; j<matrix1.dim2(); j++){
	      if(j>=i){
		    outfile << "NA ";
		  } else {
		    outfile << matrix1[i][j] << " ";
	      }
	    }
	    outfile << endl;
	  }
	} else {
	  for(int i=0; i<matrix1.dim1(); i++){
	    for(int j=0; j<matrix1.dim2(); j++){
		  outfile << matrix1[i][j] << " ";
	    }
	    outfile << endl;
	  }
	}
	cout << "2D array written to: " << fileout1 << endl;
  }

  outfile.close();
  return;
}

double spearman_rank_corr_coef(vector<double> &X, vector<double> &Y)
{
    unsigned int N = X.size();
    if(N!=Y.size() || N==1) return 0.0;
    
    vector<double> x = prosmart_rank(X);
    vector<double> y = prosmart_rank(Y);
    double mx = average(x);
    double my = average(y);
    double sxx = 0.0;
    double syy = 0.0;
    double sxy = 0.0;
    for(unsigned int i=0; i<N; i++){
        double xmx = (double)x[i] - mx;
        double ymy = (double)y[i] - my;
        sxx += xmx*xmx;
        syy += ymy*ymy;
        sxy += xmx*ymy;
    }
    if(sxx==0.0 || syy==0.0) return 0.0;
    return (sxy/sqrt(sxx*syy));
}

vector<double> prosmart_rank(vector<double> &X)
{
    vector<double> x;
    vector<LIST_ELEMENT> v;
    LIST_ELEMENT tmp;
    
    for(unsigned int i=0; i<X.size(); i++){
        tmp.val = X[i];
        tmp.idx = i;
        v.push_back(tmp);
    }
    
    sort(v.begin(),v.end(),v[0]);
    
    for(unsigned int i=0; i<v.size(); i++){
        x.push_back(0.0);
    }
    for(unsigned int i=0; i<v.size(); i++){
        unsigned int minx = i;
        unsigned int maxx = i;
        if(i>0)for(int j=i-1; j>=0; j--){
            if(j<0)break;
            if(v[i].val==v[j].val){
                minx = j;
            } else {
                break;
            }
        }
        if(i<v.size()-1)for(unsigned int j=i+1; j<v.size(); j++){
            if(v[i].val==v[j].val){
                maxx = j;
            } else {
                break;
            }
        }
        //cout << endl << i << "\t" << v[i].idx << "\t" << v[i].val << "\t" << minx << "\t" << maxx;
        unsigned int dminmax = maxx - minx;
        if(dminmax==0){
            x[v[i].idx] = (double)i + 1.0;  // the +1.0 accounts for indexes starting at 0.
        } else {
            x[v[i].idx] = (double)(maxx+minx)/2.0 + 1.0;
        }
    }
    return x;
}



