//
// This file is part of ProSMART.
//

#include "prosmart_external_programs.h"

coord prosmart_green;
coord prosmart_red;

void global_score_coloring(coord &col1, coord &col2)
{
	prosmart_green = col1;
	prosmart_red = col2;
	return;
}

vector<alignment> read_alignment(string &filename)
{
  string line;
  vector<string> line_v;
  string tmp;
  vector<alignment> align_v;
  alignment align_temp;
  
  const char *filein = filename.c_str();
  ifstream infile(filein, ios::in);
  if(infile.is_open()){
		while(!infile.eof()){
			line.clear();
			getline(infile,line);
			if(line[0] != '#'){
				
				line_v.clear();
				tmp.clear();
				for(unsigned int i=0; i<line.size(); i++){
					if(line[i]=='\t'){
						if(tmp.size()>0){
							line_v.push_back(tmp);
							tmp.clear();
						}
					} else {
						tmp += line[i];
					}
				}
				if(tmp.size()>0){
					line_v.push_back(tmp);
					tmp.clear();
				}
				if(line_v.size()!=0){	//if not end of file
					if(line_v.size()==6){	//valid number of entries in the line vector
						align_temp.res1 = delete_spaces(line_v[0]);
						align_temp.res2 = delete_spaces(line_v[1]);
						align_temp.central = atof(line_v[2].c_str());
						align_temp.minimum = atof(line_v[3].c_str());
						align_temp.rmsd = atof(line_v[4].c_str());
						align_temp.av = atof(line_v[5].c_str());
						align_v.push_back(align_temp);
					}
				}
			}
    }
		
    infile.close();
  }
  
  return align_v;	//align_v.size()==0 indicates either invalid lines in file, or file not found.
}

vector<vector<string> > get_extended_concensus_alignment(Array2D<vector<res_alignment> > &align_m, vector<res_alignment> &res_scores)
{        
    //initialise alignment with first chain-pair
    vector<string> res0;
    for(unsigned int i=0; i<align_m[0][1].size(); i++){
        res0.push_back(align_m[0][1][i].res1);
    }
    //#define REMOVE_RESIDUE_RANGE
#ifdef REMOVE_RESIDUE_RANGE
    for(unsigned int i=0; i<res0.size(); i++){
        //delete up to 82
        if(res0[0]=="82"){
            break;
        }
        res0.erase(res0.begin());
        //delete after to 82
        /*if(res0[i]=="82"){
         while(res0.size()>i){
         res0.pop_back();
         }
         }*/
    }
#endif
    
    vector<vector<string> > res_m;
    res_m.push_back(res0);

    //fill out by comparing against first structure (first pair is used as reference at the start - this bias is removed later).
    vector<string> empty;
    for(int i=1; i<align_m.dim2(); i++){
        if(align_m[0][i].size()>0){     //only consider triangular matrix
            res_m.push_back(empty);
            vector<res_alignment> tmp = align_m[0][i];
            for(unsigned int j=0; j<res_m[0].size(); j++){  //compare with first chain only
                bool FOUND = 0;
                for(unsigned int k=0; k<tmp.size(); k++){
                    if(res_m[0][j]==tmp[k].res1){
                        res_m[i].push_back(tmp[k].res2);
                        tmp.erase(tmp.begin()+k);
                        FOUND = 1;
                        break;
                    }
                }
                if(!FOUND){
                    res_m[i].push_back("NA");
                }
            }
        }
    }
    
    //check for concensus
    for(int i=1; i<align_m.dim1(); i++){
        for(int j=2; j<align_m.dim2(); j++){
            if(align_m[i][j].size()>0){     //only consider triangular matrix
                for(unsigned int k=0; k<res_m[i].size(); k++){
                    if(res_m[i][k]=="NA" || res_m[j][k]=="NA"){
                        continue;   //residue is not aligned to the first structure (i.e. missing) so don't worry about it.
                    }
                    bool FOUND = 0;
                    for(unsigned int l=0; l<align_m[i][j].size(); l++){
                        if(res_m[i][k]==align_m[i][j][l].res1){           //found res1
                            if(res_m[j][k]==align_m[i][j][l].res2){       //res2 agrees with existing chain-pairs
                                FOUND = 1;
                            }
                            break;
                        }
                    }
                    if(!FOUND){     //either residues are not aligned, or alignment is not in concensus
                        for(unsigned int l=0; l<res_m.size(); l++){
                            res_m[l].erase(res_m[l].begin()+k);
                        }
                        k--;
                    }
                }
            }
        }
    }
    
#define COUT_EXTENDED_CONSENSUS
#ifdef COUT_EXTENDED_CONSENSUS
    cout << endl << endl << "Initial alignment allowing missing residues:";
    for(unsigned int k=0; k<res_m[0].size(); k++){
        cout << endl;
        for(int i=0; i<align_m.dim1(); i++){
            cout << "\t" << res_m[i][k];
        }
    }
    cout << endl << endl;
#endif
    
    //attempt to reconstruct regions that are not aligned in the first structure pair
    vector<vector<string> > res_missing;
    vector<bool> clash;
    vector<string> empty_na(align_m.dim2(),"NA");
    
    //fill out by comparing against first structure
    for(int i=1; i<align_m.dim2(); i++){    //for each chain
        if(align_m[0][i].size()>0){
            for(unsigned int j=0; j<align_m[0][i].size(); j++){     //original data
                bool EXISTS = 0;
                for(unsigned int x=0; x<res_m[0].size(); x++){      //consensus list
                    if(align_m[0][i][j].res1 == res_m[0][x] && align_m[0][i][j].res2 == res_m[i][x]){
                        EXISTS = 1;
                        break;
                    }
                }
                if(!EXISTS){
                    for(unsigned int x=0; x<res_missing.size(); x++){
                        if(res_missing[x][0]==align_m[0][i][j].res1){
                            res_missing[x][i] = align_m[0][i][j].res2;
                            EXISTS = 1;
                            break;
                        }
                    }
                    if(!EXISTS){
                        clash.push_back(0);
                        res_missing.push_back(empty_na);
                        res_missing.back()[0] = align_m[0][i][j].res1;
                        res_missing.back()[i] = align_m[0][i][j].res2;
                    }
                }
            }
        }
    }
    
#ifdef COUT_EXTENDED_CONSENSUS
    cout << endl << endl << "Additional aligned residues to be added that are present in first chain (with clashes):";
    for(unsigned int i=0; i<res_missing.size(); i++){
        cout << endl;
        for(unsigned int j=0; j<res_missing[i].size(); j++){
            cout << "\t" << res_missing[i][j];
        }
    }
    cout << endl << endl;
#endif
    
    //compare with all structure-pairs to add new alignments and detect clashes
    for(int i=1; i<align_m.dim1(); i++){
        for(int j=2; j<align_m.dim2(); j++){
            for(unsigned int k=0; k<align_m[i][j].size(); k++){     //original data
                bool EXISTS = 0;
                for(unsigned int x=0; x<res_m[0].size(); x++){      //consensus list
                    if(align_m[i][j][k].res1 == res_m[i][x] && align_m[i][j][k].res2 == res_m[j][x]){
                        EXISTS = 1;
                        break;
                    }
                }
                if(EXISTS){continue;}   //residue is aligned in consensus
                for(unsigned int x=0; x<res_missing.size(); x++){
                    if(res_missing[x][i]==align_m[i][j][k].res1){
                        EXISTS = 1;
                        if(res_missing[x][j]=="NA"){
                            res_missing[x][j] = align_m[i][j][k].res2;
                        } else if(res_missing[x][j]!=align_m[i][j][k].res2){
                            clash[x] = 1;
                        }
                        break;
                    }
                }
                if(EXISTS){continue;}   //residue has already been added to res_missing
                clash.push_back(0);
                res_missing.push_back(empty_na);
                res_missing.back()[i] = align_m[i][j][k].res1;
                res_missing.back()[j] = align_m[i][j][k].res2;
            }
        }
    }
    
#ifdef COUT_EXTENDED_CONSENSUS
    cout << endl << endl << "Additional aligned residues to be added (with clashes):";
    for(unsigned int i=0; i<res_missing.size(); i++){
        cout << endl << clash[i];
        for(unsigned int j=0; j<res_missing[i].size(); j++){
            cout << "\t" << res_missing[i][j];
        }
    }
    cout << endl << endl;
#endif
    
    //further check for clashes
    for(unsigned int x=0; x<empty_na.size(); x++){
        for(unsigned int y=0; y<res_missing.size(); y++){
            if(clash[y]){continue;}
            if(res_missing[y][x]=="NA"){continue;}
            for(unsigned int z=0; z<y; z++){
                if(res_missing[y][x]==res_missing[z][x]){
                    clash[y] = 1;
                    clash[z] = 1;
                }
            }
        }
    }
    
    
#ifdef COUT_EXTENDED_CONSENSUS
    cout << endl << endl << "Additional aligned residues to be added (with final clashes):";
    for(unsigned int i=0; i<res_missing.size(); i++){
        cout << endl << clash[i];
        for(unsigned int j=0; j<res_missing[i].size(); j++){
            cout << "\t" << res_missing[i][j];
        }
    }
    cout << endl << endl;
#endif
    
    //remove clashes
    for(unsigned int x=1; x<=res_missing.size(); x++){
        if(clash[res_missing.size()-x]){
            res_missing.erase(res_missing.end()-x);
            clash.erase(clash.end()-x);
            x--;
        }
    }
    
    //require sufficient aligned residues
    for(int x=(int)res_missing.size()-1; x>=0; x--){
        int ctr=0;
        for(unsigned int y=0; y<res_missing[x].size(); y++){
            if(res_missing[x][y]!="NA"){
                ctr++;
            }
        }
        if(ctr<(int)(0.5*res_missing[x].size())){
            res_missing.erase(res_missing.begin()+x);
        }
    }
    
#ifdef COUT_EXTENDED_CONSENSUS
    cout << endl << endl << "Additional aligned residues to be added:";
    for(unsigned int i=0; i<res_missing.size(); i++){
        cout << endl;
        for(unsigned int j=0; j<res_missing[i].size(); j++){
            cout << "\t" << res_missing[i][j];
        }
    }
    cout << endl << endl;
#endif
    
    //scan res_m against align_m, inserting missing alignments into res_m where appropriate.
    for(unsigned int x=0; x<res_missing.size(); x++){
        for(unsigned int y=0; y<res_missing[x].size()-1; y++){
            if(res_missing[x][y]!="NA"){            //for each missing alignment, consider the first chain that contains that atom
                for(unsigned int z=y+1; z<res_missing[x].size(); z++){  
                    if(res_missing[x][z]!="NA"){    //find another chain that contains that atom
                        unsigned int ctr = 0;
                        for(unsigned int i=0; i<align_m[y][z].size(); i++){
                            //find next non-NA value in res_m (which we know must also be in align_m)
                            if(ctr < res_m[y].size()){
                                while(res_m[y][ctr]=="NA" || res_m[z][ctr]=="NA"){
                                    ctr++;
                                    if(ctr >= res_m[y].size()){
                                        break;
                                    }
                                }
                            }
                            //check ctr hasn't run off the end
                            if(ctr >= res_m[y].size()){
                                for(unsigned int j=0; j<res_m.size(); j++){
                                    res_m[j].push_back(res_missing[x][j]);
                                }
                                //cout << endl << "*** put at end";
                                break;
                            }
                            //cout << endl << x << " " << y << "\t" << align_m[y][z][i].res1 << "\t" << res_missing[x][y] << "\t" << res_m[y][ctr] << "\t" << ctr;
                            //check whether location has been found
                            if(align_m[y][z][i].res1 == res_missing[x][y]){                                
                                for(unsigned int j=0; j<res_m.size(); j++){
                                    res_m[j].insert(res_m[j].begin()+ctr,res_missing[x][j]);
                                }
                                //cout << endl << "*** insert in position: " << ctr;
                                break;
                            }
                            //if found the next (ctr) residue from res_m in align_m, increment to the next residue
                            if(align_m[y][z][i].res1 == res_m[y][ctr]){
                                ctr++;
                            }
                        }
                        break; 
                    }
                }
                break;
            }
        }
    }
    
    int MIN_NO_ATOMS = (int)(0.5*align_m.dim1());     //require residue to be present in at least half of chains in order to be included
    if(MIN_NO_ATOMS<=1){
        MIN_NO_ATOMS=2;     //if only comparing two structures
    }
    
    //remove residues that only have few observations
    if(MIN_NO_ATOMS>1){
        for(unsigned int k=0; k<res_m[0].size(); k++){
            int nRes = align_m.dim1();
            for(int i=0; i<align_m.dim1(); i++){
                if(res_m[i][k]=="NA"){
                    nRes--;
                }
            }
            if(nRes<MIN_NO_ATOMS){
#ifdef COUT_EXTENDED_CONSENSUS
                cout << endl << "Removing aligned residue " << k << " due to being present in only " << nRes << " chains";
#endif
                for(int i=0; i<align_m.dim1(); i++){
                    res_m[i].erase(res_m[i].begin()+k);
                }
            }
        }
    }
    
#ifdef COUT_EXTENDED_CONSENSUS
    cout << endl << endl << "Final alignment:";
    for(unsigned int k=0; k<res_m[0].size(); k++){
        cout << endl;
        for(int i=0; i<align_m.dim1(); i++){
            cout << "\t" << res_m[i][k];
        }
    }
    
#endif

    //get other scores
    res_scores.clear();
    res_alignment empty_ra;
    empty_ra.min=-1.0;
    empty_ra.MaxDist=-1.0;
    empty_ra.NumDist=-1;
    empty_ra.asa1=-1.0;
    vector<double> empty_vd;
    for(unsigned int l=0; l<res_m[0].size(); l++){
        res_scores.push_back(empty_ra);
        //asa_by_res.push_back(empty_vd);
    }
    for(int i=0; i<align_m.dim1(); i++){
        for(int j=1; j<align_m.dim2(); j++){
            if(align_m[i][j].size()>0){
                vector<res_alignment> tmp = align_m[i][j];
                for(unsigned int l=0; l<res_m[i].size(); l++){
                    for(unsigned int k=0; k<tmp.size(); k++){
                        if(res_m[i][l]==tmp[k].res1){
                            if(res_scores[l].min<tmp[k].min){
                                res_scores[l].min = tmp[k].min;
                            }
                            if(res_scores[l].MaxDist<tmp[k].MaxDist){
                                res_scores[l].MaxDist = tmp[k].MaxDist;
                            }
                            if(res_scores[l].NumDist<tmp[k].NumDist){
                                res_scores[l].NumDist = tmp[k].NumDist;
                            }
                            if(res_scores[l].asa1<tmp[k].asa1){
                                res_scores[l].asa1 = tmp[k].asa1;
                            }
                            if(res_scores[l].asa1<tmp[k].asa2){
                                res_scores[l].asa1 = tmp[k].asa2;
                            }
                            /*if(tmp[k].asa1 > tmp[k].asa2){
                                asa_by_res[l].push_back(tmp[k].asa1);
                            } else {
                                asa_by_res[l].push_back(tmp[k].asa2);
                            }*/
                            break;
                        }
                    }
                }
            }
        }
    }
    /*for(unsigned int i=0; i<res_m[0].size(); i++){
     cout << endl << i;
     for(unsigned int j=0; j<res_m.size(); j++){
     cout << "\t" << res_m[j][i];
     }
     cout << "\t" << res_scores[i].min;
     cout << "\t" << res_scores[i].MaxDist;
     cout << "\t" << res_scores[i].NumDist;
     cout << "\t" << res_scores[i].asa1;
     }*/
    
    //filter align_m to agree with res_m
    /*//this code is broken, and shouldn't be needed anyway
    for(int i=0; i<align_m.dim1(); i++){
        for(int j=0; j<align_m.dim2(); j++){
            if(align_m[i][j].size()>0){
                unsigned int ctr = 0;
                for(unsigned int k=0; k<align_m[i][j].size(); k++){
                    //check that align_m[i][j][k] is in res_m[i] (res1)
                    if(ctr>=res_m[i].size()){                           //end of alignment concensus
                        align_m[i][j].erase(align_m[i][j].begin()+k);
                        k--;
                    } else if(align_m[i][j][k].res1==res_m[i][ctr]){    //residue-pair is aligned in concensus
                        ctr++;
                    } else {                                            //residue-pair is not aligned in concensus
                        //cout << endl << i << "\t" << j << "\t" << k << "\t" << align_m[i][j][k].res1 << "\t" << res_m[i][ctr];
                        align_m[i][j].erase(align_m[i][j].begin()+k);
                        k--;
                    }
                }
            }
        }
    }*/
    
    string file = "extended_pairwise";
#ifdef SIDE_SCORES_GLOBAL_COORD_FRAME
    file += "_global.txt";
#else
    file += "_local.txt";
#endif
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<res_scores.size(); i++){
            outfile << res_scores[i].min;
            outfile << "\t" << res_scores[i].MaxDist;
            outfile << "\t" << res_scores[i].NumDist;
            outfile << "\t" << res_scores[i].asa1 << endl;
        }
        outfile.close();
    }
    
    file = "final_multiple_";
#ifdef SIDE_SCORES_GLOBAL_COORD_FRAME
    file += "pairwise";
#else
    file += "local";
#endif
    file += "_res_maxchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "local\tmax\tn_g1\tmaxsumasa" << endl;
        outfile.precision(3);
        for(unsigned int i=0; i<res_scores.size(); i++){
            outfile << res_scores[i].min;
            outfile << "\t" << res_scores[i].MaxDist;
            outfile << "\t" << res_scores[i].NumDist;
            outfile << "\t" << res_scores[i].asa1 << endl;
        }
        outfile.close();
    }

    /*for(int i=0; i<align_m.dim1(); i++){
        for(int j=0; j<align_m.dim2(); j++){
            for(unsigned int k=0; k<align_m[i][j].size(); k++){
                cout << endl << i << "\t" << j << "\t" << k << "\t" << align_m[i][j][k].res1 << "\t" << align_m[i][j][k].asa1 << "\t" << align_m[i][j][k].asa2;
            }
        }
    }*/

    return res_m;
}

vector<vector<string> > get_concensus_alignment(Array2D<vector<res_alignment> > &align_m, vector<res_alignment> &res_scores)
{    
    //initialise alignment with first chain-pair
    vector<string> res0;
    for(unsigned int i=0; i<align_m[0][1].size(); i++){
        res0.push_back(align_m[0][1][i].res1);
    }
//#define REMOVE_RESIDUE_RANGE
#ifdef REMOVE_RESIDUE_RANGE
    for(unsigned int i=0; i<res0.size(); i++){
        //delete up to 82
        if(res0[0]=="82"){
            break;
        }
        res0.erase(res0.begin());
        //delete after to 82
        /*if(res0[i]=="82"){
            while(res0.size()>i){
                res0.pop_back();
            }
        }*/
    }
#endif
    
    
    vector<vector<string> > res_m;
    res_m.push_back(res0);
    
    //fill out by comparing against first structure.
    vector<string> empty;
    for(int i=1; i<align_m.dim2(); i++){
        if(align_m[0][i].size()>0){     //only consider triangular matrix
            res_m.push_back(empty);
            vector<res_alignment> tmp = align_m[0][i];
            for(unsigned int j=0; j<res_m[0].size(); j++){  //compare with first chain only
                bool FOUND = 0;
                for(unsigned int k=0; k<tmp.size(); k++){
                    if(res_m[0][j]==tmp[k].res1){
                        res_m[i].push_back(tmp[k].res2);
                        tmp.erase(tmp.begin()+k);
                        FOUND = 1;
                        break;
                    }
                }
                if(!FOUND){
                    for(unsigned int k=0; k<res_m.size()-1; k++){
                        res_m[k].erase(res_m[k].begin()+j);
                    }
                    j--;
                }
            }            
        }
    }
    
    //check for concensus
    for(int i=1; i<align_m.dim1(); i++){
        for(int j=2; j<align_m.dim2(); j++){
            if(align_m[i][j].size()>0){     //only consider triangular matrix
                for(unsigned int k=0; k<res_m[i].size(); k++){
                    bool FOUND = 0;
                    for(unsigned int l=0; l<align_m[i][j].size(); l++){
                        if(res_m[i][k]==align_m[i][j][l].res1){           //found res1
                            if(res_m[j][k]==align_m[i][j][l].res2){       //res2 agrees with existing chain-pairs
                                FOUND = 1;
                            }
                            break;
                        }
                    }
                    if(!FOUND){     //either residues are not aligned, or alignment is not in concensus
                        for(unsigned int l=0; l<res_m.size(); l++){
                            res_m[l].erase(res_m[l].begin()+k);
                        }
                        k--;
                    }
                }
            }
        }
    }
    
    //get other scores
    res_scores.clear();
    res_alignment empty_ra;
    empty_ra.min=-1.0;
    empty_ra.MaxDist=-1.0;
    empty_ra.NumDist=-1;
    empty_ra.asa1=-1.0;
    for(unsigned int l=0; l<res_m[0].size(); l++){
        res_scores.push_back(empty_ra);
    }
    for(int i=0; i<align_m.dim1(); i++){
        for(int j=1; j<align_m.dim2(); j++){
            if(align_m[i][j].size()>0){
                vector<res_alignment> tmp = align_m[i][j];
                for(unsigned int l=0; l<res_m[i].size(); l++){
                    for(unsigned int k=0; k<tmp.size(); k++){
                        if(res_m[i][l]==tmp[k].res1){
                            if(res_scores[l].min<tmp[k].min){
                                res_scores[l].min = tmp[k].min;
                            }
                            if(res_scores[l].MaxDist<tmp[k].MaxDist){
                                res_scores[l].MaxDist = tmp[k].MaxDist;
                            }
                            if(res_scores[l].NumDist<tmp[k].NumDist){
                                res_scores[l].NumDist = tmp[k].NumDist;
                            }
                            if(res_scores[l].asa1<tmp[k].asa1){
                                res_scores[l].asa1 = tmp[k].asa1;
                            }
                            if(res_scores[l].asa1<tmp[k].asa2){
                                res_scores[l].asa1 = tmp[k].asa2;
                            }
                            break;
                        }
                    }
                }
            }
        }
    }
    /*for(unsigned int i=0; i<res_m[0].size(); i++){
        cout << endl << i;
        for(unsigned int j=0; j<res_m.size(); j++){
            cout << "\t" << res_m[j][i];
        }
        cout << "\t" << res_scores[i].min;
        cout << "\t" << res_scores[i].MaxDist;
        cout << "\t" << res_scores[i].NumDist;
        cout << "\t" << res_scores[i].asa1;
    }*/
    
    //filter align_m to agree with res_m
    for(int i=0; i<align_m.dim1(); i++){
        for(int j=0; j<align_m.dim2(); j++){
            if(align_m[i][j].size()>0){
                unsigned int ctr = 0;
                for(unsigned int k=0; k<align_m[i][j].size(); k++){
                    //check that align_m[i][j][k] is in res_m[i] (res1)
                    if(ctr>=res_m[i].size()){                           //end of alignment concensus
                        align_m[i][j].erase(align_m[i][j].begin()+k);
                        k--;
                    } else if(align_m[i][j][k].res1==res_m[i][ctr]){    //residue-pair is aligned in concensus
                        ctr++;
                    } else {                                            //residue-pair is not aligned in concensus
                        //cout << endl << i << "\t" << j << "\t" << k << "\t" << align_m[i][j][k].res1 << "\t" << res_m[i][ctr];
                        align_m[i][j].erase(align_m[i][j].begin()+k);
                        k--;
                    }
                }
            }
        }
    }
    
    string file = "pairwise";
#ifdef SIDE_SCORES_GLOBAL_COORD_FRAME
    file += "_global.txt";
#else
    file += "_local.txt";
#endif
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<res_scores.size(); i++){
            outfile << res_scores[i].min;
            outfile << "\t" << res_scores[i].MaxDist;
            outfile << "\t" << res_scores[i].NumDist;
            outfile << "\t" << res_scores[i].asa1 << endl;
        }
        outfile.close();
    }
    
    return res_m;
}

void write_pymol_concensus(string file, vector<string> &files, vector<string> &chains, vector<PDBfile> &pdbs)
{
    for(unsigned int i=1; i<pdbs.size(); i++){
        if(pdbs[i].size()!=pdbs[0].size()){
            return;
        }
    }
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        outfile << "color white, all" << endl;
        for(unsigned int i=0; i<pdbs.size(); i++){
            for(int j=0; j<pdbs[i].size(); j++){
                outfile << "color red, (" << get_filename(files[i]) << "//" << chains[i] << "/" << delete_spaces(pdbs[i].get_orig_resnum(j)) << "/" << delete_spaces(pdbs[i].get_atom(j)) << ")" << endl;
            }
        }
        outfile.close();
    }
    
    return;
}

void write_pymol_concensus(string file, vector<string> &files, vector<string> &chains, vector<vector<string> > &concensus)
{
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        outfile << "color white, all" << endl;
        for(unsigned int i=0; i<concensus.size(); i++){
            for(unsigned int j=0; j<concensus[i].size(); j++){
                outfile << "color red, (" << get_filename(files[i]) << "//" << chains[i] << "/" << concensus[i][j] << "/*)" << endl;
            }
        }
        outfile.close();
    }
    
    return;
}

void write_pymol_concensus(vector<string> &files, vector<string> &chains, Array2D<PDBfile> &pdb_m)
{
    int N = pdb_m.dim1();
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j)continue;
            write_pymol_concensus(files[i],files[j],chains[i],chains[j],pdb_m[i][j],pdb_m[j][i]);
        }
    }
    
    return;
}

void write_pymol_concensus(string &file1, string &file2, string &chain1, string &chain2, PDBfile &pdb1, PDBfile &pdb2)
{
    string f1 = get_filename(file1);
    string f2 = get_filename(file2);
    string file = f1 + "_" + f2 + ".pml";
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        outfile << "color white, all" << endl;
        for(int i=0; i<pdb1.size(); i++){
            outfile << "color red, (" << f1 << "//" << chain1 << "/" << delete_spaces(pdb1.get_orig_resnum(i)) << "/" << delete_spaces(pdb1.get_atom(i)) << ")" << endl;
        }
        for(int i=0; i<pdb2.size(); i++){
            outfile << "color red, (" << f2 << "//" << chain2 << "/" << delete_spaces(pdb2.get_orig_resnum(i)) << "/" << delete_spaces(pdb2.get_atom(i)) << ")" << endl;
        }
        outfile.close();
    }
    
    return;
}

void output_pymol_residues(string file, vector<string> &filenames, vector<string> &chains, vector<vector<string> > &concensus, vector<double> &scores, double score_cutoff)
{
#ifdef SIDE_SCORES_GLOBAL_COORD_FRAME
    file = get_filename(file)+"_global.pml";
#else
    file = get_filename(file)+"_local.pml";
#endif
    
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        outfile << "color white, all" << endl;
        vector<string> obj;
        for(unsigned int i=0; i<filenames.size(); i++){
            obj.push_back(get_filename(filenames[i]));
        }
        
        double val = 0.0;
        double ival = 1.0;
        int NO_COL = 100;   //this will actually cause NO_COL+1 colours to be created...
        for(int i=0; i<=NO_COL; i++){
            val = (double)i/NO_COL;
            ival = 1.0-val;
            double color1 = prosmart_green.x*ival+prosmart_red.x*val;
            double color2 = prosmart_green.y*ival+prosmart_red.y*val;
            double color3 = prosmart_green.z*ival+prosmart_red.z*val;
            outfile << "set_color newcolor" << i << " = [" << color1 << "," << color2 << "," << color3 << "]" << endl;
        }
        
        int index = -1;
        for(unsigned int i=0; i<scores.size(); i++){
            val = (double)NO_COL*scores[i]/score_cutoff;
            if(val < 0.0){
                index = -1;
            } else {
                if(val < (double)NO_COL){
                    index = floor(val);
                } else {
                    index = NO_COL;
                }
                for(unsigned int j=0; j<obj.size(); j++){
                    outfile << "color newcolor" << index << ", (" << obj[j] << "//" << chains[j] << "/";
                    if(concensus[j][i][0]=='-'){
                        outfile << "\\";
                    }
                    outfile << delete_spaces(concensus[j][i]) << "/*)" << endl;
                }
            }
        }
        
        outfile.close();
    }
    return;
}

Array2D<PDBfile> get_aligned_pdbs(string &workingdirectory, vector<string> &filenames, vector<string> &chains, Array2D<vector<res_alignment> > &align_m)
{
    unsigned int N = filenames.size();
    vector<PDBfile> pdb_v;
    vector<vector<res_corresp> > original_res;
    Array2D<PDBfile> pdb_m(N,N);
    
    //initialise PDB objects
    for(unsigned int i=0; i<N; i++){
        string obj = get_filename(filenames[i]) + "_" + chains[i][0];
        string input = workingdirectory + ".Input_Chains/" + obj;
        PDBfile pdb;
        pdb_v.push_back(pdb);
        pdb_v[i].read_formatted_pdb(input);
        original_res.push_back(read_original_res(input+"_orig"));
        for(int j=0; j<pdb_v[i].size(); j++){
            pdb_v[i].set_resnum(j,original_res[i][pdb_v[i].get_resnum(j)].res);
        }
    }
    
    //construct matrix of pairwise aligned concesus PDB objects
    for(unsigned int i=0; i<N; i++){
        for(unsigned int j=0; j<N; j++){
            if(align_m[i][j].size()==0)continue;
            pdb_m[i][j] = pdb_v[i];
            pdb_m[j][i] = pdb_v[j];
            //cout << endl << i << " " << j << "\t" << pdb_m[i][j].size() << "\t" << pdb_m[j][i].size();
            filter_pdb(pdb_m[i][j],align_m[i][j],original_res[i],1);    //ensure residues correspond
            filter_pdb(pdb_m[j][i],align_m[i][j],original_res[j],2);
            //cout << endl << i << " " << j << "\t" << pdb_m[i][j].size() << "\t" << pdb_m[j][i].size();
            ensure_consensus(pdb_m[i][j],pdb_m[j][i]);                  //ensure atoms correspond
            //cout << endl << i << " " << j << "\t" << pdb_m[i][j].size() << "\t" << pdb_m[j][i].size() << endl;
        }
    }
    return pdb_m;
}

void filter_pdb(PDBfile &pdb, vector<res_alignment> &align_m, vector<res_corresp> &original_res, unsigned int chain)
{
    string tmp;
    unsigned int ctr = 0;
    unsigned int ctr_next = 0;
    for(int k=0; k<pdb.size(); k++){
        tmp = pdb.get_orig_resnum(k);
        if(tmp[tmp.size()-1]==' '){
            tmp.erase(tmp.end()-1);
        }
        if(ctr_next<align_m.size()){
            //cout << endl << "'" << tmp << "'\t'" << align_m[ctr].res1 << "'\t" << align_m[ctr].res2 << "\t" << align_m[ctr_next].res1 << "\t" << align_m[ctr_next].res2;
            if(chain == 1){
                if(tmp==align_m[ctr_next].res1){
                    ctr_next++;
                    ctr = ctr_next-1;
                    continue;
                }
            } else {
                if(tmp==align_m[ctr_next].res2){
                    ctr_next++;
                    ctr = ctr_next-1;
                    continue;
                }
            }
        }
        if(chain == 1){
            if(tmp!=align_m[ctr].res1){
                pdb.erase(k);
                k--;
            }
        } else {
            if(tmp!=align_m[ctr].res2){
                pdb.erase(k);
                k--;
            }
        }
    }
    return;
}

void ensure_consensus(PDBfile &pdb1, PDBfile &pdb2)
//ensure consensus between all atoms in 2 PDBfiles (assumes consensus between residues).
{
    vector<vector<pdbline> > res1;
    vector<vector<pdbline> > res2;
    vector<pdbline> empty_vp;
    bool ATOM_EXISTS=0;

    //convert PDBs to residues
    res1.push_back(empty_vp);
    res1.back().push_back(pdb1.line(0));
    for(int i=1; i<pdb1.size(); i++){
        if(pdb1.get_orig_resnum(i)!=pdb1.get_orig_resnum(i-1)){
            res1.push_back(empty_vp);
        }
        res1.back().push_back(pdb1.line(i));
    }
    res2.push_back(empty_vp);
    res2.back().push_back(pdb2.line(0));
    for(int i=1; i<pdb2.size(); i++){
        if(pdb2.get_orig_resnum(i)!=pdb2.get_orig_resnum(i-1)){
            res2.push_back(empty_vp);
        }
        res2.back().push_back(pdb2.line(i));
    }
    
    //sanity check
    if(res1.size()!=res2.size()){
        pdb1.clear();
        pdb2.clear(); 
        return;
    }
        
    //for each residue i
    for(unsigned int i=0; i<res1.size(); i++){
        //check each atom (j) in chain 1 exists (k) in chain 2
        for(unsigned int j=0; j<res1[i].size(); j++){
            ATOM_EXISTS=0;
            for(unsigned int k=0; k<res2[i].size(); k++){
                if(strcmp(res1[i][j].atom,res2[i][k].atom)==0){
                    ATOM_EXISTS=1;
                    break;
                }
            }
            if(!ATOM_EXISTS){
                res1[i].erase(res1[i].begin()+j);
                j--;
            }
        }
        
        //check each atom (j) in chain 2 exists (k) in chain 1
        for(unsigned int j=0; j<res2[i].size(); j++){
            ATOM_EXISTS=0;
            for(unsigned int k=0; k<res1[i].size(); k++){
                if(strcmp(res2[i][j].atom,res1[i][k].atom)==0){
                    ATOM_EXISTS=1;
                    break;
                }
            }
            if(!ATOM_EXISTS){
                res2[i].erase(res2[i].begin()+j);
                j--;
            }
        }
        
        //check atoms are in same order
        for(unsigned int j=1; j<res1[i].size(); j++){
            if(strcmp(res1[i][j].atom,res2[i][j].atom)!=0){
                //need to reorder atoms
                for(unsigned int k=j+1; k<res2[i].size(); k++){
                    if(strcmp(res1[i][j].atom,res2[i][k].atom)==0){
                        pdbline tmp = res2[i][j];
                        res2[i][j] = res2[i][k];
                        res2[i][k] = tmp;
                        break;
                    }
                }
            }
        }
        
        //correct for potential side chain flips and nomenclature inconsistencies
        if(res1[i].size()>0){
            string resid = res1[i][0].resid;
            if(resid == "PHE" || resid == "TYR")
                apply_side_flips(res1[i],res2[i]," CD1"," CD2"," CE1"," CE2");
            else if(resid == "ARG")
                apply_side_flips(res1[i],res2[i]," NH1"," NH2");
            else if(resid == "ASP")
                apply_side_flips(res1[i],res2[i]," OD1"," OD2");
            else if(resid == "GLU")
                apply_side_flips(res1[i],res2[i]," OE1"," OE2");
            else if(resid == "VAL")
                apply_side_flips(res1[i],res2[i]," CG1"," CG2");
            else if(resid == "LEU")
                apply_side_flips(res1[i],res2[i]," CD1"," CD2");
#ifdef APPLY_SIDE_FLIPS_MULTIPLE
            else if(resid == "HIS")
                apply_side_flips(res1[i],res2[i]," ND1"," CD2"," CE1"," NE2");
            else if(resid == "THR")
                apply_side_flips(res1[i],res2[i]," OG1"," CG2");
            else if(resid == "ASN")
                apply_side_flips(res1[i],res2[i]," OD1"," ND2");
            else if(resid == "GLN")
                apply_side_flips(res1[i],res2[i]," OE1"," NE2");
#endif
        }
        /*cout << endl << endl << "res " << i << endl;
        for(unsigned int j=0; j<res1[i].size(); j++){
            cout << "  " << res1[i][j] << "  " << res2[i][j];
        }*/
    }
    
    //reconstruct PDBfile objects
    pdb1.clear();
    pdb2.clear();
    for(unsigned int i=0; i<res1.size(); i++){    //sanity check
        if(res1[i].size()!=res2[i].size()){
            return;
        }
    }
    for(unsigned int i=0; i<res1.size(); i++){
        for(unsigned int j=0; j<res1[i].size(); j++){
            pdb1.add(res1[i][j]);
            pdb2.add(res2[i][j]);
        }
    }

    return;
}

Array2D<PDBfile> get_concensus_pdbs(string &workingdirectory, vector<string> &filenames, vector<string> &chains, vector<vector<string> > &concensus, bool ALLOW_MISSING)
//ensure consensus between all atoms in N PDBfiles (assumes consensus between residues has been identified).
//(consensus may contain NAs in order to allow missing alignments)
{
    unsigned int N = filenames.size();
    Array2D<PDBfile> pdb_m(N,N);
    vector<PDBfile> pdb_v;
    stringstream ss;
    
    for(unsigned int i=0; i<concensus[0].size(); i++){
        cout << endl;
        for(unsigned int j=0; j<N; j++){
            cout << "\t" << concensus[j][i];
        }
    }
    
    for(unsigned int i=0; i<N; i++){
        string obj = get_filename(filenames[i]) + "_" + chains[i][0];
        string input = workingdirectory + ".Input_Chains/" + obj;
        PDBfile pdb;
        pdb.read_formatted_pdb(input);
        vector<res_corresp> original_res = read_original_res(input+"_orig");
        //unsigned int ctr = 0;
        //unsigned int ctr_next = 0;
        int CURRENT = (int)concensus[i].size()-1;
        for(int j=pdb.size()-1; j>=0; j--){
            ss.str("");
            if(original_res[pdb.get_resnum(j)].ins==' ')
                ss << original_res[pdb.get_resnum(j)].res;
            else
                ss << original_res[pdb.get_resnum(j)].res << original_res[pdb.get_resnum(j)].ins;
            pdb.set_resnum(j,original_res[pdb.get_resnum(j)].res);
            
            bool FOUND = 0;
            for(int k=CURRENT; k>=0; k--){  //no need to search from the end, because we know that concensus[i] is sorted.
                if(ss.str()==concensus[i][k]){
                    CURRENT = k;
                    FOUND = 1;                    
                    break;
                }
            }
            if(!FOUND){
                pdb.erase(j);
            }
        }
        pdb_v.push_back(pdb);
    }
    
    /*cout << endl << endl << "Filtered PDB sizes:" << endl;
    for(unsigned int i=0; i<N; i++){
        cout << "\t" << pdb_v[i].size();
    }*/
    
    //convert PDBs to residues
    vector<vector<vector<pdbline> > > residues;
    vector<vector<pdbline> > empty_vvp;
    vector<pdbline> empty_vp;
    for(unsigned int i=0; i<N; i++){
        residues.push_back(empty_vvp);
        residues.back().push_back(empty_vp);
        residues.back().back().push_back(pdb_v[i].line(0));
        for(int j=1; j<pdb_v[i].size(); j++){
            if(pdb_v[i].get_orig_resnum(j)!=pdb_v[i].get_orig_resnum(j-1)){
                residues.back().push_back(empty_vp);
            }
//#define ONLY_TAKE_MAINCHAIN
#ifdef ONLY_TAKE_MAINCHAIN
            string tmp = pdb_v[i].line(j).atom;
            if(tmp!=" N  " && tmp!=" C  " && tmp!=" CA " && tmp!=" O  ")continue;
#endif
            residues.back().back().push_back(pdb_v[i].line(j));
        }
    }
    /*for(unsigned int i=0; i<N; i++){                              //chain i
     cout << endl << "PDB " << i << endl;
     for(unsigned int j=0; j<residues[i].size(); j++){           //residue j
     cout << endl;
     for(unsigned int k=0; k<residues[i][j].size(); k++){    //atom k
     cout << residues[i][j][k];
     }
     }
        break;
     }*/
    
    //insert NA residues into "residues" from "concensus"
    for(unsigned int i=0; i<N; i++){
        for(unsigned int j=0; j<concensus[i].size(); j++){
            if(concensus[i][j]=="NA"){
                residues[i].insert(residues[i].begin()+j,empty_vp);
            }
        }
    }
    
    //sanity check
    for(unsigned int i=0; i<N; i++){
        if(concensus[0].size() != residues[i].size() || concensus[0].size() != concensus[i].size()){
            return pdb_m;   //error has occurred.
        }
    }
    
    if(ALLOW_MISSING){  //allow NAs - i.e. atoms to be missing from some chains
        
        /*for(unsigned int i=0; i<residues[0].size(); i++){               //residue i
            cout << endl;
            for(unsigned int j=0; j<residues.size(); j++){              //chain j
                cout << endl << "res " << i << "\tchain " << j;
                if(residues[j][i].size()>0){
                    cout << "\t" << residues[j][i][0].res_num << "\t" << residues[j][i][0].resid;
                }
                //cout << endl;
                //for(unsigned int k=0; k<residues[j][i].size(); k++){cout << residues[j][i][k];}
            }
        }*/
        
        //expand residues into atomic correspondences
        vector<vector<vector<pdbline> > > residue_corresp;
        for(unsigned int i=0; i<N; i++){
            residue_corresp.push_back(empty_vvp);
            for(unsigned int i=0; i<residues[0].size(); i++){
                residue_corresp.back().push_back(empty_vp);
            }
        }
        pdbline tmp_line;
        strcpy(tmp_line.atom, "XXXX");
        strcpy(tmp_line.resid, "XXX");
        strcpy(tmp_line.element, "XX");
        strcpy(tmp_line.charge, "XX");
        tmp_line.alt = ' ';
        tmp_line.chain = 'X';
        tmp_line.res_num = -1;
        tmp_line.ins_code = 'X';
        tmp_line.crd.x = -1.0;
        tmp_line.crd.y = -1.0;
        tmp_line.crd.z = -1.0;
        tmp_line.occup = -1.0;
        tmp_line.bfact = -1.0;
        tmp_line.asa = -1.0;
        tmp_line.sigma = -1.0;
        tmp_line.orig_resnum = "-1 ";
        for(unsigned int i=0; i<residues[0].size(); i++){                   //for each aligned residue
            for(unsigned int j=0; j<N; j++){                  //for each chain
                for(unsigned int k=0; k<residues[j][i].size(); k++){        //for each atom
                    //find whether this atom has already been aligned
                    bool FOUND = 0;
                    for(unsigned int l=0; l<residue_corresp[j][i].size(); l++){
                        if(residue_corresp[j][i][l].bfact != -1.0){
                            if(strcmp(residue_corresp[j][i][l].atom,residues[j][i][k].atom)==0){
                                FOUND = 1;
                                break;
                            }
                        }
                    }
                    if(!FOUND){
                        //add atom to list of alignments
                        residue_corresp[j][i].push_back(residues[j][i][k]);
                        //search for correponding atoms
                        for(unsigned int l=0; l<j; l++){
                            residue_corresp[l][i].push_back(tmp_line);
                        }
                        for(unsigned int l=j+1; l<N; l++){
                            bool FOUNDIT = 0;
                            for(unsigned int m=0; m<residues[l][i].size(); m++){
                                //cout << endl << "here:\t'" << residue_corresp[j][i].back().atom << "'\t'" << residues[l][i][m].atom << "'";
                                if(strcmp(residue_corresp[j][i].back().atom,residues[l][i][m].atom)==0){
                                    //cout << "\t***";
                                    //see whether residue has already been added
                                    for(unsigned int n=0; n<residue_corresp[l][i].size(); n++){
                                        if(residue_corresp[l][i][n].bfact != -1.0){
                                            if(strcmp(residue_corresp[l][i][n].atom,residues[l][i][m].atom)==0){
                                                //cout << "\tFOUNDIT";
                                                FOUNDIT = 1;
                                                break;
                                            }
                                        }
                                    }
                                    if(!FOUNDIT){
                                        //cout << "\tNOT FOUNDIT";
                                        residue_corresp[l][i].push_back(residues[l][i][m]);
                                    }
                                    FOUNDIT = 1;
                                    break;
                                }
                            }
                            if(!FOUNDIT){
                                residue_corresp[l][i].push_back(tmp_line);
                            }
                        }
                    }
                }
            }
            
            //correct for potential side chain flips and nomenclature inconsistencies
            if(residue_corresp[0][i].size()>0){
                
                //find residue name of aligned residue from each chain
                vector<string> resid;
                for(unsigned int j=0; j<N; j++){
                    resid.push_back("");
                    for(unsigned int k=0; k<residue_corresp[j][i].size(); k++){
                        if(residue_corresp[j][i][k].bfact != -1){
                            resid.back() = residue_corresp[j][i][k].resid;
                            break;
                        }
                    }
                }

                //get indexes of reference chains with unique residue types
                vector<int> idx;
                for(unsigned int j=0; j<N; j++){
                    if(resid[j].size()>0){
                        bool FOUND = 0;
                        for(unsigned int k=0; k<idx.size(); k++){
                            if(resid[idx[k]]==resid[j]){
                                FOUND = 1;
                                break;
                            }
                        }
                        if(!FOUND){
                            idx.push_back(j);
                        }
                    }
                }
                                
                if(idx.size()>0){       //at least one valid atom
                    if(idx.size()>1){   //more than one different residue type
                        for(unsigned int j=0; j<idx.size(); j++){
                            for(unsigned int k=0; k<idx.size(); k++){
                                if(j!=k){
                                    //check for PHE-LEU residues, swapping the first LEU onto the first PHE
                                    if(resid[idx[j]]== "PHE" && resid[idx[k]]== "LEU"){
                                        apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[idx[k]][i]," CD1"," CD2");
                                    }
                                    //check for PHE-TYR residues, swapping the first TYR onto the first PHE
                                    if(resid[idx[j]]== "PHE" && resid[idx[k]]== "TYR"){
                                        apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[idx[k]][i]," CD1"," CD2"," CE1"," CE2");
                                    }
                                    //check for VAL-ILE residues, swapping the first VAL onto the first ILE
                                    if(resid[idx[j]]== "ILE" && resid[idx[k]]== "VAL"){
                                        apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[idx[k]][i]," CG1"," CG2");
                                    }
                                }
                            }
                        }
                    }
                    for(unsigned int j=0; j<idx.size(); j++){
                        for(unsigned int k=idx[j]; k<N; k++){
                            if(resid[idx[j]]==resid[k]){
                                if(resid[k]== "PHE" || resid[k]== "TYR")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," CD1"," CD2"," CE1"," CE2");
                                else if(resid[k]== "ARG")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," NH1"," NH2");
                                else if(resid[k]== "ASP")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," OD1"," OD2");
                                else if(resid[k]== "GLU")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," OE1"," OE2");
                                else if(resid[k]== "VAL")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," CG1"," CG2");
                                else if(resid[k]== "LEU")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," CD1"," CD2");
#ifdef APPLY_SIDE_FLIPS_MULTIPLE
                                else if(resid[k]== "HIS")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," ND1"," CD2"," CE1"," NE2");
                                else if(resid[k]== "THR")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," OG1"," CG2");
                                else if(resid[k]== "ASN")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," OD1"," ND2");
                                else if(resid[k]== "GLN")
                                    apply_side_flips(residue_corresp[idx[j]][i],residue_corresp[k][i]," OE1"," NE2");
#endif
                            }
                        }
                    }
                }
            }
        }
        
        unsigned int MIN_NO_ATOMS = 0.5*residue_corresp.size();     //require atom to be present in at least half of chains in order to be included
        if(MIN_NO_ATOMS<=1){
            MIN_NO_ATOMS=2;     //if only comparing two structures
        }

        //remove atoms that only have few observations
        if(MIN_NO_ATOMS>1){
            for(unsigned int i=0; i<residue_corresp[0].size(); i++){                //residue i
                for(unsigned int j=0; j<residue_corresp[0][i].size(); j++){         //atom j
                    unsigned int ctr = 0;
                    for(unsigned int k=0; k<residue_corresp.size(); k++){           //chain k
                        if(residue_corresp[k][i][j].bfact != -1.0){
                            ctr++;
                        }
                    }
                    if(ctr<MIN_NO_ATOMS){
                        for(unsigned int k=0; k<residue_corresp.size(); k++){
                            residue_corresp[k][i].erase(residue_corresp[k][i].begin()+j);
                        }
                        j--;
                    }
                }
            }
        }

        
        /*for(unsigned int i=0; i<residue_corresp[0].size(); i++){               //residue i
            cout << endl;
            for(unsigned int j=0; j<residue_corresp.size(); j++){              //chain j
                cout << endl << "res " << i << "\tchain " << j << endl;
                for(unsigned int k=0; k<residue_corresp[j][i].size(); k++){    //atom k
                    cout << residue_corresp[j][i][k];
                }
            }
        }*/
        
        //reconstruct PDBfile objects
        pdb_v.clear();
        PDBfile empty_pdb;
        empty_pdb.clear();
        for(unsigned int i=0; i<N; i++){                                //chain i
            pdb_v.push_back(empty_pdb);
            for(unsigned int j=0; j<residue_corresp[i].size(); j++){           //residue j
                for(unsigned int k=0; k<residue_corresp[i][j].size(); k++){    //atom k
                    residue_corresp[i][j][k].sigma = (double)j;                //unique number for each aligned residue (whether it exists or not) - this is used for reconstruction.
                    pdb_v[i].add(residue_corresp[i][j][k]);
                }
            }
        }
        
        //sanity check
        for(unsigned int i=1; i<N; i++){
            if(pdb_v[i].size()!=pdb_v[0].size()){
                pdb_v.clear();
            }
        }
        
        for(unsigned int i=0; i<N; i++){
            for(unsigned int j=i+1; j<N; j++){
                pdb_m[i][j] = pdb_v[i];
                pdb_m[j][i] = pdb_v[j];
            }
        }

        return pdb_m;
    }
        

    
    bool ATOM_EXISTS=0;
    //ensure atoms correspond, for each residue
    for(unsigned int i=0; i<residues[0].size(); i++){               //residue i
        //check atom j in chain 0 exists in all other chains
        if(residues[0][i].size()>0){
            for(unsigned int j=0; j<residues[0][i].size(); j++){            //atom j in chain 0
                for(unsigned int k=1; k<residues.size(); k++){
                    if(residues[k][i].size()==0){
                        continue;
                    }
                    ATOM_EXISTS=0;
                    for(unsigned int l=0; l<residues[k][i].size(); l++){    //atom l in chain k
                        if(strcmp(residues[0][i][j].atom,residues[k][i][l].atom)==0){
                            ATOM_EXISTS=1;
                            break;
                        }
                    }
                    if(!ATOM_EXISTS){
                        residues[0][i].erase(residues[0][i].begin()+j);
                        j--;
                        break;
                    }
                }
            }
        }
        //check atom k in chain j exists in chain 0
        for(unsigned int j=1; j<residues.size(); j++){
            for(unsigned int k=0; k<residues[j][i].size(); k++){
                ATOM_EXISTS=0;
                for(unsigned int l=0; l<residues[0][i].size(); l++){
                    if(strcmp(residues[0][i][l].atom,residues[j][i][k].atom)==0){
                        ATOM_EXISTS=1;
                        break;
                    }
                }
                if(!ATOM_EXISTS){
                    residues[j][i].erase(residues[j][i].begin()+k);
                    k--;
                }
            }
        }
        //check atoms are in same order
        for(unsigned int j=1; j<residues.size(); j++){
            for(unsigned int k=0; k<residues[0][i].size(); k++){
                if(strcmp(residues[0][i][k].atom,residues[j][i][k].atom)!=0){
                    //need to reorder atoms
                    for(unsigned int l=k+1; l<residues[j][i].size(); l++){
                        if(strcmp(residues[0][i][k].atom,residues[j][i][l].atom)==0){
                            pdbline tmp = residues[j][i][l];
                            residues[j][i][l] = residues[j][i][k];
                            residues[j][i][k] = tmp;
                            break;
                        }
                    }
                }
            }
        }
        //correct for potential side chain flips and nomenclature inconsistencies
        if(residues[0][i].size()>0){
            string resid = residues[0][i][0].resid;
            for(unsigned int j=1; j<residues.size(); j++){  //chain j
                if(resid == "PHE" || resid == "TYR")
                    apply_side_flips(residues[0][i],residues[j][i]," CD1"," CD2"," CE1"," CE2");
                else if(resid == "ARG")
                    apply_side_flips(residues[0][i],residues[j][i]," NH1"," NH2");
                else if(resid == "ASP")
                    apply_side_flips(residues[0][i],residues[j][i]," OD1"," OD2");
                else if(resid == "GLU")
                    apply_side_flips(residues[0][i],residues[j][i]," OE1"," OE2");
                else if(resid == "VAL")
                    apply_side_flips(residues[0][i],residues[j][i]," CG1"," CG2");
                else if(resid == "LEU")
                    apply_side_flips(residues[0][i],residues[j][i]," CD1"," CD2");
#ifdef APPLY_SIDE_FLIPS_MULTIPLE
                else if(resid == "HIS")
                    apply_side_flips(residues[0][i],residues[j][i]," ND1"," CD2"," CE1"," NE2");
                else if(resid == "THR")
                    apply_side_flips(residues[0][i],residues[j][i]," OG1"," CG2");
                else if(resid == "ASN")
                    apply_side_flips(residues[0][i],residues[j][i]," OD1"," ND2");
                else if(resid == "GLN")
                    apply_side_flips(residues[0][i],residues[j][i]," OE1"," NE2");
#endif
            }
        }
        
        /*cout << endl;
        for(unsigned int j=0; j<residues.size(); j++){              //chain j
            cout << endl << "res " << i << "\tchain " << j << endl;
            for(unsigned int k=0; k<residues[j][i].size(); k++){    //atom k
                cout << residues[j][i][k];
            }
        }*/
    }
    
    //reconstruct PDBfile objects
    pdb_v.clear();
    PDBfile empty_pdb;
    empty_pdb.clear();
    for(unsigned int i=0; i<N; i++){                                //chain i
        pdb_v.push_back(empty_pdb);
        for(unsigned int j=0; j<residues[i].size(); j++){           //residue j
            for(unsigned int k=0; k<residues[i][j].size(); k++){    //atom k
                pdb_v[i].add(residues[i][j][k]);
            }
        }
    }
    
    //sanity check
    for(unsigned int i=1; i<N; i++){
        if(pdb_v[i].size()!=pdb_v[0].size()){
            pdb_v.clear();
        }
    }

    for(unsigned int i=0; i<N; i++){
        for(unsigned int j=i+1; j<N; j++){
            pdb_m[i][j] = pdb_v[i];
            pdb_m[j][i] = pdb_v[j];
        }
    }
            
    return pdb_m;
}

void apply_side_flips(vector<pdbline> &res1, vector<pdbline> &res2, string s1, string s2)
{
    Coords tmp1;
    Coords tmp2;
    unsigned int i1 = 0;
    unsigned int i2 = 0;
    bool GOT1 = 0;
    bool GOT2 = 0;
    vector<unsigned int> idx;
    for(unsigned int i=0; i<res1.size(); i++){
        if(res1[i].bfact==-1.0 || res2[i].bfact==-1.0){
            continue;
        }
        tmp1.add(res1[i].crd);
        tmp2.add(res2[i].crd);
        idx.push_back(i);
        if(strcmp(res1[i].atom,s1.c_str())==0){
            i1 = idx.size()-1;
            GOT1 = 1;
        } else if(strcmp(res1[i].atom,s2.c_str())==0){
            i2 = idx.size()-1;
            GOT2 = 1;
        }
    }    
    if(!GOT1 || !GOT2)return; //at least one of the required atoms doesn't exist
    
    Transform transf = get_transf(tmp1,tmp2);
    Coords tmp3 = transf.transform(tmp2);
    double d_current = RMSD(tmp1,tmp3);
    
    tmp2 = tmp2.swap(i1,i2);
    transf = get_transf(tmp1,tmp2);
    tmp3 = transf.transform(tmp2);
    double d_swap = RMSD(tmp1,tmp3);
    
    //cout << endl << res1[i1] << res1[i2] << "\t" << d_current << "\t" << d_swap << endl;
        
    if(d_swap<d_current){
        pdbline tmp = res2[idx[i1]];
        res2[idx[i1]] = res2[idx[i2]];
        res2[idx[i2]] = tmp;
    }
    
    return;
}

void apply_side_flips(vector<pdbline> &res1, vector<pdbline> &res2, string s1, string s2, string s3, string s4)
{
    Coords tmp1;
    Coords tmp2;
    unsigned int i1 = 0;
    unsigned int i2 = 0;
    unsigned int i3 = 0;
    unsigned int i4 = 0;
    bool GOT1 = 0;
    bool GOT2 = 0;
    bool GOT3 = 0;
    bool GOT4 = 0;
    vector<unsigned int> idx;
    for(unsigned int i=0; i<res1.size(); i++){
        if(res1[i].bfact==-1.0 || res2[i].bfact==-1.0){
            continue;
        }
        tmp1.add(res1[i].crd);
        tmp2.add(res2[i].crd);
        idx.push_back(i);
        if(strcmp(res1[i].atom,s1.c_str())==0){
            i1 = idx.size()-1;
            GOT1 = 1;
        } else if(strcmp(res1[i].atom,s2.c_str())==0){
            i2 = idx.size()-1;
            GOT2 = 1;
        } else if(strcmp(res1[i].atom,s3.c_str())==0){
            i3 = idx.size()-1;
            GOT3 = 1;
        } else if(strcmp(res1[i].atom,s4.c_str())==0){
            i4 = idx.size()-1;
            GOT4 = 1;
        }
    }
    if(!GOT1 || !GOT2 || !GOT3 || !GOT4)return; //at least one of the atoms doesn't exist
    
    Transform transf = get_transf(tmp1,tmp2);
    Coords tmp3 = transf.transform(tmp2);
    double d_current = RMSD(tmp1,tmp3);
    
    tmp2 = tmp2.swap(i1,i2);
    tmp2 = tmp2.swap(i3,i4);
    transf = get_transf(tmp1,tmp2);
    tmp3 = transf.transform(tmp2);
    double d_swap = RMSD(tmp1,tmp3);
    //cout << endl << res1[i1] << res1[i2] << "\t" << d_current << "\t" << d_swap << endl;
    
    if(d_swap<d_current){
        pdbline tmp = res2[idx[i1]];
        res2[idx[i1]] = res2[idx[i2]];
        res2[idx[i2]] = tmp;
        tmp = res2[idx[i3]];
        res2[idx[i3]] = res2[idx[i4]];
        res2[idx[i4]] = tmp;
    }
    
    return;
}

Array2D<Coords> get_coord_m(Array2D<PDBfile> &pdb_m, bool SIEVE, bool CONSENSUS)
{
    int N = pdb_m.dim1();
    Array2D<Coords> result(N,N);

    //get coords and pairwise sieve fit if required
    for(int i=0; i<N; i++){
        for(int j=1; j<N; j++){
            if(i>=j)continue;
            pdb_m[i][j].getcoords(result[i][j]);
            pdb_m[j][i].getcoords(result[j][i]);
            if(SIEVE){
                vector<unsigned int> removed = sieve_fit(result[i][j],result[j][i]);     //filter coords
                for(unsigned int k=0; k<removed.size(); k++){
                    //cout << endl << pdb_m[i][j].size() << "\t" << pdb_m[j][i].size() << "\t" << removed[k];
                    /*if((int)pdb_m[i][j].size()<=removed[k] || (int)pdb_m[j][i].size()<=removed[k]){
                        cout << endl << "got here" << endl;
                        exit(-1);
                    }*/
                    if(CONSENSUS){
                        for(int x=0; x<N; x++){
                            for(int y=0; y<N; y++){
                                if(x==y)continue;
                                pdb_m[x][y].erase(removed[k]);
                            }
                        }
                    } else {
                        pdb_m[i][j].erase(removed[k]);
                        pdb_m[j][i].erase(removed[k]);
                    }
                }
            }
        }
    }
    
    if(SIEVE && CONSENSUS){
        for(int i=0; i<N; i++){
            for(int j=1; j<N; j++){
                if(i>=j)continue;
                pdb_m[i][j].getcoords(result[i][j]);
                pdb_m[j][i].getcoords(result[j][i]);
            }
        }
    }

    return result;
}

void output_pairwise_superpositions(Array2D<Coords> &crd_m, vector<PDBfile> &pdb_align)
{
    Array2D<vector<double> > distM(crd_m.dim1(),crd_m.dim2());
    vector<double> empty;
    Coords crds1;
    Coords crds2;
    
    //get distances
    for(int i=0; i<crd_m.dim1(); i++){
        for(int j=1; j<crd_m.dim2(); j++){
            if(i>=j)continue;
            distM[i][j] = empty;
            Transform transf = get_transf(crd_m[i][j],crd_m[j][i]);
            pdb_align[i].getcoords(crds1);
            pdb_align[j].getcoords(crds2);
            Coords crdj = transf.transform(crds2);
            for(int k=0; k<crdj.size(); k++){
                distM[i][j].push_back(dist(crds1.get(k),crdj.get(k)));
                //cout << i << "\t" << j << "\t" << k << endl << "\t" << (*crd_m[i][j].get(k)) << "\t" << (*crdj.get(k));
            }
        }
    }

    //convert to list
    Array1D<vector<double> > dist_list(distM[0][1].size());
    for(int k=0; k<dist_list.dim1(); k++){
        for(int i=0; i<distM.dim1(); i++){
            for(int j=1; j<distM.dim2(); j++){
                if(i>=j)continue;
                dist_list[k].push_back(distM[i][j][k]);
            }
        }
    }
    
    string file1 = "all_final_pairwise_data.txt";
    ofstream outfile1;
    outfile1.open(file1.c_str());
    if(outfile1.is_open()){
        for(int i=0; i<dist_list.dim1(); i++){
            outfile1 << pdb_align[0].get_resnum(i) << pdb_align[0].get_atom(i);
            for(unsigned int j=0; j<dist_list[i].size(); j++){
                outfile1 << "\t" << dist_list[i][j];
            }
            outfile1 << endl;
        }
        outfile1.close();
    }
    return;
}

vector<coord> identify_translations(Array2D<Coords> &crd_m)
{
    int N = crd_m.dim1();
    vector<coord> result;
    Array1D<Coords> crds(N);

    //pool all coords
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j)continue;
            crds[i].append(crd_m[i][j]);
        }
    }
    
    //get means
    for(int i=0; i<N; i++){
        result.push_back(crds[i].mean());
    }
    
    //normalise coords
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j)continue;
            crd_m[i][j] = crd_m[i][j] - result[i];
        }
    }
    
    return result;
}

#define CALCULATE_SS_FOR_GEN_PROC
vector<Transform> multiple_structure_superposition1(Array2D<Coords> &crd_m, vector<coord> &means)
//linearly update R_t using mean
{    
    //initialise identity transformation
    initialise_identityM();
    Transform empty_t;
    empty_t.create();
    vector<Transform> transf_v;
    int N = crd_m.dim1();
    
    //crd_m[k][l] contains X_k that is used when comparing with X_l
    
    //calculate X_k^T*X_l for k!=l.
    Array2D<double> temp(3,3);
    Array2D<Array2D<double> > XkTXl(N,N);
    for(int i=1; i<N; i++){
        for(int j=0; j<i; j++){
            //note: can perform coord filtering here (e.g. sieve fitting) - translations would require special consideration.
            temp = crd_m[i][j]*crd_m[j][i];
            XkTXl[i][j] = temp;
            XkTXl[j][i] = t(temp);
        }
    }
    
/*#define GET_ANALYTICAL_SOLUTION
#ifdef GET_ANALYTICAL_SOLUTION
    Array2D<Array2D<double> > R_kM(N,N);
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j){
                temp = get_identityM();
            } else {
                temp = rotateM(crd_m[i][j],crd_m[j][i]);
            }
            R_kM[i][j] = temp;
            cout << endl << "rotation " << i << " " << j << ":" << endl << R_kM[i][j];
        }
    }
    cout << endl << endl;
    
    Array2D<double> R_hat(3*N,3*N);
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            for(int k=0; k<3; k++){
                for(int l=0; l<3; l++){
                    R_hat[3*i+k][3*j+l]=R_kM[i][j][k][l];
                }
            }
        }
    }
    cout << "R_hat:" << endl << R_hat << endl << endl;

    Eigenvalue<double> tempEIG(R_hat);
    Array2D<double> eig_vec;
    tempEIG.getV(eig_vec);
    Array1D<double> eigen;
	tempEIG.getRealEigenvalues(eigen);
    
    cout << "eigenvectors:" << endl << eig_vec << endl << endl;
    cout << "eigenvalues:" << endl << eigen << endl << endl;
    
    Array1D<Array2D<double> > R_k_(N);
    for(int i=0; i<N; i++){
        for(int k=0; k<3; k++){
            for(int l=0; l<3; l++){
                temp[k][l] = eig_vec[3*i+k][l];
            }
        }
        R_k_[i] = temp;
        double trRRT = tr(matmult(R_k_[i],R_k_[i]));
        cout << endl << "rotation " << i << ":" << endl << R_k_[i] << endl << trRRT << endl << det3x3(R_k_[i]) << endl;
        Array2D<double> tmp = R_k_[i];
        for(int k=0; k<3; k++){
            for(int l=0; l<3; l++){
                tmp[k][l] = tmp[k][l]/sqrt(trRRT);
            }
        }
        cout << endl << tmp << endl << tr(matmult(tmp,tmp)) << endl << det3x3(tmp) << endl;
        
    }
    
#endif*/
    
    //calculate initial R_k
    Array1D<Array2D<double> > R_k(N);
    R_k[0] = get_identityM();
    for(int i=1; i<N; i++){
        temp = rotateM(crd_m[0][i],crd_m[i][0]);
        R_k[i] = temp;
    }
    
    //iteratively optimise rotations R_k
    int ctr = 0;
    double max_angle = 0.0;
    while(1){
        max_angle = 0.0;
#ifdef CALCULATE_SS_FOR_GEN_PROC
        double tr_XkTXlRlRkT = 0.0;
        double superpose_ss = 0.0;
#endif
        for(int i=0; i<N; i++){        //to include k=0 then select i=0..N instead of i=1..N
            Array2D<double> sumXkTXlRl(3,3,(double)0.0);
            for(int j=0; j<N; j++){
                if(j==i)continue;
                sumXkTXlRl += matmult(XkTXl[i][j],R_k[j]);
            }
            Array2D<double> R_old = R_k[i];
            R_k[i]=rotateM(sumXkTXlRl);
            Array2D<double> R_diff = crossprod(R_old,R_k[i]);
            double angle = get_angle(R_diff);
            if(angle>max_angle)max_angle=angle;
#ifdef CALCULATE_SS_FOR_GEN_PROC
            tr_XkTXlRlRkT += tr(crossprod(sumXkTXlRl,R_k[i]));
            for(int j=0; j<N; j++){
                if(j==i)continue;
                temp = matmult(crd_m[i][j].asmatrix(),R_k[i]) - matmult(crd_m[j][i].asmatrix(),R_k[j]);
                superpose_ss+=tr_cov(temp,temp);
            }
#endif
        }
#ifdef CALCULATE_SS_FOR_GEN_PROC
        cout.precision(6);
        cout << endl << "Iteration:" << ctr << "\tMax angle: " << max_angle;
        cout.precision(15);
        cout << "\ttr(): " << tr_XkTXlRlRkT;
        cout << "\tss: " << superpose_ss;
#else
        cout << endl << "Iteration:" << ctr << "\tMax angle: " << max_angle;
#endif
        ctr++;
        if(ctr>30)break;
        if(max_angle<0.0001)break;
    }
    
    //get transformations to superpose onto means[0] (first chain)
    Transform temp_t;
    for(int i=0; i<N; i++){
        temp_t.create(means[i],R_k[i],means[0]);
        transf_v.push_back(temp_t);
    }
    
    /*for(unsigned int i=0; i<N; i++){
     cout << endl << "rotation " << i << ":  " << transf_v[i].get_r();
     }*/
    
    return transf_v;
}

vector<Transform> multiple_structure_superposition1(vector<PDBfile> &pdb_v, vector<string> &filenames, vector<string> &chains)
//brute force approach
{
    //initialise identity transformation
    initialise_identityM();
    Transform empty_t;
    empty_t.create();
    vector<Transform> transf_v;
    vector<string> info;
    bool OUT_PDB_FULL = 0;
    string fileout2 = "1/"+get_full_filename(filenames[0]);
    double max_angle = 0.0;
    int ctr=0;
    
    //it is assumed that the atoms in the PDBs directly correspond.
    //convert each PDB to coordinates
    unsigned int N = pdb_v.size();
    Coords empty;
    vector<Coords> crd_v;
    for(unsigned int i=0; i<N; i++){
        crd_v.push_back(empty);
        pdb_v[i].getcoords(crd_v[i]);
    }
    
    //output original chain 1
    string empty_s="";
    string tmp="1";
    create_directory(empty_s,tmp);
    
    tmp="2";
    create_directory(empty_s,tmp);
    output_pdb(info,filenames[0],fileout2,chains[0][0],empty_t,0,OUT_PDB_FULL);
    
    //get initial superposition using the first chain as the target, and output each chain
    transf_v.push_back(empty_t);
    for(unsigned int i=1; i<N; i++){
        transf_v.push_back(get_transf(crd_v[0],crd_v[i]));
        crd_v[i] = transf_v[i].transform(crd_v[i]);  //crd_v[i] is updated
        fileout2 = "1/"+get_full_filename(filenames[i]);
        output_pdb(info,filenames[i],fileout2,chains[i][0],transf_v[i],1,OUT_PDB_FULL);
    }
    
    //iterative deterministic optimisation of transformations transf_v - coordinates crd_v should also be changed.
    ctr=0;
    while(1){
        max_angle = multiple_structure_iteration(crd_v,transf_v);
        cout << endl << "Iteration:" << ctr << "\tMax angle: " << max_angle;
        ctr++;
        if(ctr>30)break;
        if(max_angle<0.00001)break;
    }
    
    /*for(unsigned int i=0; i<N; i++){
     cout << endl << "rotation " << i << ":  " << transf_v[i].get_r();
     }*/
    
    //output PDB files
    for(unsigned int i=0; i<transf_v.size(); i++){
        fileout2 = "2/"+get_full_filename(filenames[i]);
        output_pdb(info,filenames[i],fileout2,chains[i][0],transf_v[i],1,OUT_PDB_FULL);
    }
    
    //output PDB file loader
    ofstream outfile;
    outfile.open("loader.pml");
    if(outfile.is_open()){
        for(unsigned int i=0; i<N; i++){
            outfile << "load 1/"+get_full_filename(filenames[i]) << endl;
            outfile << "load 2/"+get_full_filename(filenames[i]) << endl;
        }
        outfile.close();
    }
    
    return transf_v;
}

//#define CALCULATE_SS_FOR_GEN_PROC
vector<Transform> multiple_structure_superposition2(vector<PDBfile> &pdb_v, vector<string> &filenames, vector<string> &chains)
//linearly update R_t using mean
{    
    //initialise identity transformation
    initialise_identityM();
    Transform empty_t;
    empty_t.create();
    vector<Transform> transf_v;
    vector<string> info;
    bool OUT_PDB_FULL = 0;
    string fileout2 = "1/"+get_full_filename(filenames[0]);
    double max_angle = 0.0;
    int ctr=0;

    //it is assumed that the atoms in the PDBs directly correspond.
    //convert each PDB to coordinates
    unsigned int N = pdb_v.size();
    Coords empty;
    vector<Coords> crd_v;
    for(unsigned int i=0; i<N; i++){
        crd_v.push_back(empty);
        pdb_v[i].getcoords(crd_v[i]);
    }

    //get means and normalise all coords
    vector<coord> means;
    for(unsigned int i=0; i<N; i++){
        means.push_back(crd_v[i].mean());
        crd_v[i] = crd_v[i] - means[i];
    }
    
    //problem with sieve fitting - translations will change. 
    //pairwise superpositions, with fitting, should be performed externally, 
    
    //calculate X_k^T*X_l for k!=l.
    Array2D<double> temp(3,3);
    Array2D<Array2D<double> > XkTXl(N,N);
    for(unsigned int i=1; i<N; i++){
        for(unsigned int j=0; j<i; j++){
            //note - can perform any pairwise coord filtering here (e.g. sieve fitting)
            temp = crd_v[i]*crd_v[j];
            XkTXl[i][j]=temp;
            XkTXl[j][i]=t(temp);
        }
    }
    
    //calculate initial R_k
    Array1D<Array2D<double> > R_k(N);
    R_k[0] = get_identityM();
    for(unsigned int i=1; i<N; i++){
        temp = rotateM(crd_v[0],crd_v[i]);
        R_k[i]=temp;
    }
    
    //iteratively optimise rotations R_k
    ctr=0;
    while(1){
        max_angle = 0.0;
#ifdef CALCULATE_SS_FOR_GEN_PROC
        double tr_XkTXlRlRkT = 0.0;
        double superpose_ss = 0.0;
#endif
        for(unsigned int i=0; i<N; i++){        //to include k=0 then select i=0..N instead of i=1..N
            Array2D<double> sumXkTXlRl(3,3,(double)0.0);
            for(unsigned int j=0; j<N; j++){
                if(j==i)continue;
                sumXkTXlRl += matmult(XkTXl[i][j],R_k[j]);
            }
            Array2D<double> R_old = R_k[i];
            R_k[i]=rotateM(sumXkTXlRl);
            Array2D<double> R_diff = crossprod(R_old,R_k[i]);
            double angle = get_angle(R_diff);
            if(angle>max_angle)max_angle=angle;
#ifdef CALCULATE_SS_FOR_GEN_PROC
            tr_XkTXlRlRkT += tr(crossprod(sumXkTXlRl,R_k[i]));
            for(unsigned int j=0; j<N; j++){
                if(j==i)continue;
                temp = matmult(crd_v[i].asmatrix(),R_k[i]) - matmult(crd_v[j].asmatrix(),R_k[j]);
                superpose_ss+=tr_cov(temp,temp);
            }
#endif
        }
#ifdef CALCULATE_SS_FOR_GEN_PROC
        cout.precision(6);
        cout << endl << "Iteration:" << ctr << "\tMax angle: " << max_angle;
        cout.precision(15);
        cout << "\ttr(): " << tr_XkTXlRlRkT;
        cout << "\tss: " << superpose_ss;
#else
        cout << endl << "Iteration:" << ctr << "\tMax angle: " << max_angle;
#endif
        ctr++;
        if(ctr>30)break;
        if(max_angle<0.0001)break;
    }
        
    //get transformations to superpose onto crd_v[0]
    Transform temp_t;
    for(unsigned int i=0; i<N; i++){
        temp_t.create(means[i],R_k[i],means[0]);
        transf_v.push_back(temp_t);
    }
    
    /*for(unsigned int i=0; i<N; i++){
        cout << endl << "rotation " << i << ":  " << transf_v[i].get_r();
    }*/
    
    //output PDB files
    for(unsigned int i=0; i<transf_v.size(); i++){
        fileout2 = "2/"+get_full_filename(filenames[i]);
        output_pdb(info,filenames[i],fileout2,chains[i][0],transf_v[i],1,OUT_PDB_FULL);
    }
    
    //output PDB file loader
    ofstream outfile;
    outfile.open("loader.pml");
    if(outfile.is_open()){
        for(unsigned int i=0; i<N; i++){
            outfile << "load 1/"+get_full_filename(filenames[i]) << endl;
            outfile << "load 2/"+get_full_filename(filenames[i]) << endl;
        }
        outfile.close();
    }
    
    return transf_v;
}

//#define CALCULATE_SS_FOR_GEN_PROC
vector<Transform> multiple_structure_superposition3(vector<PDBfile> &pdb_v, vector<string> &filenames, vector<string> &chains)
//optimise rotations using average quaternion
{
    //initialise identity transformation
    initialise_identityM();
    Transform empty_t;
    empty_t.create();
    vector<Transform> transf_v;
    vector<string> info;
    bool OUT_PDB_FULL = 0;
    string fileout2 = "1/"+get_full_filename(filenames[0]);
    double max_angle = 0.0;
    int ctr=0;
    
    //it is assumed that the atoms in the PDBs directly correspond.
    //convert each PDB to coordinates
    unsigned int N = pdb_v.size();
    Coords empty;
    vector<Coords> crd_v;
    for(unsigned int i=0; i<N; i++){
        crd_v.push_back(empty);
        pdb_v[i].getcoords(crd_v[i]);
    }
    
    //get means and normalise all coords
    vector<coord> means;
    for(unsigned int i=0; i<N; i++){
        means.push_back(crd_v[i].mean());
        crd_v[i] = crd_v[i] - means[i];
    }
    
    //calculate X_k^T*X_l for k!=l.
    Array2D<double> temp(3,3);
    Array2D<Array2D<double> > XkTXl(N,N);
    for(unsigned int i=1; i<N; i++){
        for(unsigned int j=0; j<i; j++){
            //note - can perform any pairwise coord filtering here (e.g. sieve fitting)
            temp = crd_v[i]*crd_v[j];
            XkTXl[i][j]=temp;
            XkTXl[j][i]=t(temp);
        }
    }
    
    //calculate initial R_k
    Array1D<Array2D<double> > R_k(N);
    R_k[0] = get_identityM();
    for(unsigned int i=1; i<N; i++){
        temp = rotateM(crd_v[0],crd_v[i]);
        R_k[i]=temp;
    }
    
    //calculate rotation corresponding to svd of XkTXl
    Array2D<Array2D<double> > Rkl(N,N);
    for(unsigned int i=1; i<N; i++){
        for(unsigned int j=0; j<i; j++){
            temp = rotateM(XkTXl[i][j]);
            Rkl[i][j]=temp;
            Rkl[j][i]=t(temp);
        }
    }
    
#ifdef CALCULATE_SS_FOR_GEN_PROC
    double superpose_ss = 0.0;
    for(unsigned int i=0; i<N; i++){
        for(unsigned int j=0; j<N; j++){
            if(j==i)continue;
            temp = matmult(crd_v[i].asmatrix(),R_k[i]) - matmult(crd_v[j].asmatrix(),R_k[j]);
            superpose_ss+=tr_cov(temp,temp);
        }
    }
    cout.precision(6);
    cout << endl << "Iteration:0";
    cout.precision(15);
    cout << "\tss: " << superpose_ss;
#endif
    
    //iteratively optimise rotations R_k
    ctr=0;
    while(1){
        max_angle = 0.0;
#ifdef CALCULATE_SS_FOR_GEN_PROC
        double superpose_ss = 0.0;
#endif
        
        //calculate rotations that optimally pairwise superpose the chains
        Array1D<vector<Array2D<double> > > Rvals(N);
        for(unsigned int i=0; i<N; i++){            //to include k=0 then select i=0..N instead of i=1..N
            for(unsigned int j=0; j<N; j++){
                if(i==j)continue;
                Rvals[i].push_back(matmult(Rkl[i][j],R_k[j]));
            }
        }
        /*for(unsigned int i=0; i<N; i++){
         cout << endl << i << endl << R_k[i];
         for(unsigned int j=0; j<Rvals[i].size(); j++){
         cout << endl << Rvals[i][j];
         }
         }*/
        
        for(unsigned int i=0; i<N; i++){
            vector<quarternion> temp_q = get_quarternions(Rvals[i]);
            quarternion temp_qav = average_quarternion(temp_q);
            Array2D<double> R_old = R_k[i];
            R_k[i] = get_rot_from_quart_single(temp_qav);
            Array2D<double> R_diff = crossprod(R_old,R_k[i]);
            double angle = get_angle(R_diff);
            if(angle>max_angle)max_angle=angle;
        }
        
#ifdef CALCULATE_SS_FOR_GEN_PROC
        superpose_ss = 0.0;
        for(unsigned int i=0; i<N; i++){
            for(unsigned int j=0; j<N; j++){
                if(j==i)continue;
                temp = matmult(crd_v[i].asmatrix(),R_k[i]) - matmult(crd_v[j].asmatrix(),R_k[j]);
                superpose_ss+=tr_cov(temp,temp);
            }
        }
        cout.precision(6);
        cout << endl << "Iteration: " << ctr << "\tMax angle: " << max_angle;
        cout.precision(15);
        cout << "\tss: " << superpose_ss;
#endif
        ctr++;
        if(ctr>30)break;
        if(max_angle<0.0001)break;
    }
    
    //get transformations to superpose onto crd_v[0]
    Transform temp_t;
    for(unsigned int i=0; i<N; i++){
        temp_t.create(means[i],R_k[i],means[0]);
        transf_v.push_back(temp_t);
    }
    
    /*for(unsigned int i=0; i<N; i++){
     cout << endl << "rotation " << i << ":  " << transf_v[i].get_r();
     }*/
    
    //output PDB files
    for(unsigned int i=0; i<transf_v.size(); i++){
        fileout2 = "2/"+get_full_filename(filenames[i]);
        output_pdb(info,filenames[i],fileout2,chains[i][0],transf_v[i],1,OUT_PDB_FULL);
    }
    
    //output PDB file loader
    ofstream outfile;
    outfile.open("loader.pml");
    if(outfile.is_open()){
        for(unsigned int i=0; i<N; i++){
            outfile << "load 1/"+get_full_filename(filenames[i]) << endl;
            outfile << "load 2/"+get_full_filename(filenames[i]) << endl;
        }
        outfile.close();
    }
    
    return transf_v;
}

double multiple_structure_iteration(vector<Coords> &crd_v, vector<Transform> &transf_v)
{
    //get transformations that superpose all chain-pairs
    unsigned int N = transf_v.size();
    vector<vector<Transform> > transf_m;
    vector<Transform> empty_vt;
    for(unsigned int i=0; i<N; i++){
        transf_m.push_back(empty_vt);
    }    
    for(unsigned int i=0; i<N; i++){
        for(unsigned int j=i+1; j<N; j++){
            transf_m[i].push_back(get_transf(crd_v[j],crd_v[i]));
            transf_m[j].push_back(get_transf(crd_v[i],crd_v[j]));
        }
    }
    
    //get average transformations
    double max_angle = 0.0;
    for(unsigned int i=0; i<N; i++){
        Transform transf = get_average_transformation(transf_m[i]);
        //cout << endl << i << "\tUpdate angle:\t" << transf.get_r_angle();
        
        if(transf.get_r_angle()>max_angle){
            max_angle = transf.get_r_angle();
        }
        
        //apply transformation to coordinates
        crd_v[i] = transf.transform(crd_v[i]);
        
        //update final transformation
        transf_v[i].apply(transf);
    }
    
    return max_angle;
}

void multiple_structure_scoring(vector<string> &filenames, vector<PDBfile> &pdb_v, vector<Transform> &transf_v)
{    
    //get transformed coords
    vector<Coords> crd_transf;
    for(unsigned int i=0; i<pdb_v.size(); i++){
        Coords crd;
        pdb_v[i].getcoords(crd);                            //get coords from pdbfile
        crd_transf.push_back(transf_v[i].transform(crd));   //transform coords
    }    
        
    //get max dissimilar scores
    coord tmp_crd;
    vector<double> scores;
    vector<vector<double> > all_scores;
    vector<vector<double> > all_asa;
    vector<double> empty;
    for(int i=0; i<pdb_v[0].size(); i++){       //for each (aligned) atom
        //cout << endl << pdb_v[0].line(i);
        vector<coord> crds;
        vector<bool> is_present;
        for(unsigned int j=0; j<crd_transf.size(); j++){    //for each chain
            if(pdb_v[j].get_occup(i)!=-1.0 && pdb_v[j].get_resnum(i)!=-1){      //only take aligned atoms (some chains may not contain the aligned atom)
                is_present.push_back(1);
                crds.push_back(crd_transf[j].get_crd(i));
            } else {
                is_present.push_back(0);
                crds.push_back(tmp_crd);
            }
        }
        double max_dist = 0.0;
        all_scores.push_back(empty);
        all_asa.push_back(empty);
        for(unsigned int j=0; j<crds.size()-1; j++){
            for(unsigned int k=j+1; k<crds.size(); k++){
                if(is_present[j] && is_present[k]){
                    all_scores.back().push_back(dist(crds[j],crds[k]));                    
                } else {
                    all_scores.back().push_back(-1.0);
                }
                if(all_scores.back().back()>max_dist){
                    max_dist = all_scores.back().back();
                }
                if(pdb_v[j].get_asa(i)>pdb_v[k].get_asa(i)){
                    all_asa.back().push_back(pdb_v[j].get_asa(i));
                } else {
                    all_asa.back().push_back(pdb_v[k].get_asa(i));
                }
            }
        }
        scores.push_back(max_dist);
    }
    
    
    //output colour script
    ofstream outfile;
    vector<string> obj;
    string file = "max_dist.pml";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        outfile << "color white, all" << endl;
//#define HIDE_ALL_WHITE_ATOMS
#ifdef HIDE_ALL_WHITE_ATOMS
        outfile << "hide all" << endl;
#endif
        for(unsigned int i=0; i<pdb_v.size(); i++){
            obj.push_back(get_filename(filenames[i]));
        }
        
        double val = 0.0;
        double ival = 1.0;
        int NO_COL = 100;   //this will actually cause NO_COL+1 colours to be created...
        for(int i=0; i<=NO_COL; i++){
            val = (double)i/NO_COL;
            ival = 1.0-val;
            double color1 = prosmart_green.x*ival+prosmart_red.x*val;
            double color2 = prosmart_green.y*ival+prosmart_red.y*val;
            double color3 = prosmart_green.z*ival+prosmart_red.z*val;
            outfile << "set_color newcolor" << i << " = [" << color1 << "," << color2 << "," << color3 << "]" << endl;
        }
        
        double score_cutoff = 1.0;
        int index = -1;
        for(unsigned int i=0; i<scores.size(); i++){
            val = (double)NO_COL*scores[i]/score_cutoff;
            if(val >= 0.0){
                if(val < (double)NO_COL){
                    index = floor(val);
                } else {
                    index = NO_COL;
                }
#ifdef HIDE_ALL_WHITE_ATOMS
                for(unsigned int j=0; j<obj.size(); j++){
                    if(pdb_v[j].get_bfact(i) == -1.0){
                        continue;
                    }
                    outfile << "show (" << obj[j] << "//" << pdb_v[j].get_chain(i) << "/";
                    if(pdb_v[j].get_resnum(i)<0){
                        outfile << "\\";
                    }
                    outfile << delete_spaces(pdb_v[j].get_orig_resnum(i)) << "/" << delete_spaces(pdb_v[j].get_atom(i)) << ")" << endl;
                }
#endif
                outfile << "color newcolor" << index << ",";
                bool FOUND = 0;
                for(unsigned int j=0; j<obj.size(); j++){
                    if(pdb_v[j].get_bfact(i) == -1.0){
                        continue;
                    }
                    if(FOUND){
                        outfile << " |";
                    } else {
                        FOUND = 1;
                    }
                    outfile << " (" << obj[j] << "//" << pdb_v[j].get_chain(i) << "/";
                    if(pdb_v[j].get_resnum(i)<0){
                        outfile << "\\";
                    }
                    outfile << delete_spaces(pdb_v[j].get_orig_resnum(i)) << "/" << delete_spaces(pdb_v[j].get_atom(i)) << ")";
                }
                if(!FOUND){
                    outfile << " tmp" << endl;  //sanity check; avoids colouring everything one colour
                }
                outfile << endl;
            }
        }

        outfile.close();
    }
    
    vector<string> chainpair_names;
    file = "all_pairwise_names.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<obj.size()-1; i++){
            for(unsigned int j=i+1; j<obj.size(); j++){
                chainpair_names.push_back(obj[i]+"_"+obj[j]);
                outfile << chainpair_names.back() << endl;
            }
        }
        outfile.close();
    }
    
    file = "all_pairwise_scores.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<all_scores.size(); i++){
            outfile << pdb_v[0].get_orig_resnum(i) << "\t" << pdb_v[0].get_resid(i) << "\t" << delete_spaces(pdb_v[0].get_atom(i));
            for(unsigned int j=0; j<all_scores[i].size(); j++){
                if(all_scores[i][j]>=0.0){
                    outfile.precision(2);
                    outfile << "\t" << all_scores[i][j];
                } else {
                    outfile << "\tNA";
                }
            }
            outfile << endl;
        }
        outfile.close();
    }
    
    file = "final_multiple_global_atoms_allchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "resnum\tresid\tatom\tdist\tmaxasa" << endl;
        outfile.precision(3);
        for(unsigned int i=0; i<all_scores.size(); i++){
            for(unsigned int j=0; j<all_scores[i].size(); j++){
                if(all_scores[i][j]>=0.0){
                    outfile << pdb_v[0].get_orig_resnum(i) << "\t" << pdb_v[0].get_resid(i) << "\t" << delete_spaces(pdb_v[0].get_atom(i)) << "\t" << all_scores[i][j] << "\t" << all_asa[i][j] << endl;
                }
            }
        }
        outfile.close();
    }
    
    file = "final_multiple_global_atoms_maxchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "resnum\tresid\tatom\tdist" << endl;
        outfile.precision(3);
        for(unsigned int i=0; i<scores.size(); i++){
            outfile << pdb_v[0].get_orig_resnum(i) << "\t" << pdb_v[0].get_resid(i) << "\t" << delete_spaces(pdb_v[0].get_atom(i)) << "\t" << scores[i] << endl;
        }
        outfile.close();
    }

    ////////////////////////////////////////
    
    vector<vector<vector<double> > > final_vector;  //residue:atom:chain
    vector<vector<vector<double> > > final_vector_asa;
    vector<vector<double> > final_tmp;
    vector<string> final_resnum;
    vector<string> final_resid;
    
    final_vector.push_back(final_tmp);
    final_vector[0].push_back(all_scores[0]);
    final_vector_asa.push_back(final_tmp);
    final_vector_asa[0].push_back(all_asa[0]);
    final_resnum.push_back(pdb_v[0].get_orig_resnum(0));
    final_resid.push_back(pdb_v[0].get_resid(0));
    for(unsigned int i=1; i<all_scores.size(); i++){
        cout << endl << i << "\t" << pdb_v[0].get_orig_resnum(i) << "\t" << pdb_v[0].get_sigma(i) << "\t" << pdb_v[0].get_sigma(i-1);
        if(pdb_v[0].get_sigma(i)!=pdb_v[0].get_sigma(i-1)){   //new residue - sigma has been modified to store unique res codes
            final_vector.push_back(final_tmp);
            final_vector_asa.push_back(final_tmp);
            final_resnum.push_back(pdb_v[0].get_orig_resnum(i));
            final_resid.push_back(pdb_v[0].get_resid(i));
        }
        final_vector.back().push_back(all_scores[i]);
        final_vector_asa.back().push_back(all_asa[i]);
    }
    /*for(unsigned int i=0; i<final_vector.size(); i++){
        cout << endl << "Residue " << i;
        for(unsigned int j=0; j<final_vector[i].size(); j++){
            cout << endl << "atom " << j << "\t" << final_vector[i][j].size() << endl;
            for(unsigned int k=0; k<final_vector[i][j].size(); k++){
                cout << "\t" << final_vector[i][j][k];
            }
        }
    }*/

    
    vector<vector<vector<double> > > final_vector_res;  //residue:chain:atom
    vector<vector<vector<double> > > final_vector_res_asa;
    vector<double> final_tmp1;
    for(unsigned int i=0; i<final_vector.size(); i++){
        final_vector_res.push_back(final_tmp);
        final_vector_res_asa.push_back(final_tmp);
        for(unsigned int k=0; k<final_vector[0][0].size(); k++){
            final_vector_res[i].push_back(final_tmp1);
            final_vector_res_asa[i].push_back(final_tmp1);
        }
        for(unsigned int j=0; j<final_vector[i].size(); j++){
            for(unsigned int k=0; k<final_vector[i][j].size(); k++){
                final_vector_res[i][k].push_back(final_vector[i][j][k]);
                final_vector_res_asa[i][k].push_back(final_vector_asa[i][j][k]);
            }
        }
    }
        
    vector<vector<double> > final_max;  //residue:chain - inner vectors are of different lengths - only store chain-pairs with aligned residues
    vector<vector<double> > final_min;
    vector<vector<double> > final_mean;
    vector<vector<double> > final_median;
    //vector<vector<double> > final_sumasa;
    vector<vector<unsigned int> > final_atomsall;
    vector<vector<unsigned int> > final_atoms;
    vector<vector<unsigned int> > final_atoms2;
    vector<unsigned int> final_atomsall_chain(final_vector_res[0].size(),0);
    vector<unsigned int> final_atoms_chain(final_vector_res[0].size(),0);
    vector<unsigned int> final_atoms2_chain(final_vector_res[0].size(),0);
    vector<vector<double> > asa_score_g1(final_vector_res[0].size());
    vector<unsigned int> tmp_vui;
    vector<double> final_empty(final_vector[0][0].size(),0.0);
    
    for(unsigned int i=0; i<final_vector_res.size(); i++){
        final_max.push_back(final_tmp1);
        final_min.push_back(final_tmp1);
        final_mean.push_back(final_tmp1);
        final_median.push_back(final_tmp1);
        //final_sumasa.push_back(final_tmp1);
        final_atomsall.push_back(tmp_vui);
        final_atoms.push_back(tmp_vui);
        final_atoms2.push_back(tmp_vui);
        for(unsigned int j=0; j<final_vector_res[i].size(); j++){
            vector<double> tmpv_score;
            vector<double> tmpv_asa;
            for(unsigned int k=0; k<final_vector_res[i][j].size(); k++){
                if(final_vector_res[i][j][k]>=0.0){
                    tmpv_score.push_back(final_vector_res[i][j][k]);
                    tmpv_asa.push_back(final_vector_res_asa[i][j][k]);
                }
            }
            if(tmpv_score.size()>0){
                final_max[i].push_back(max(tmpv_score));
                final_min[i].push_back(min(tmpv_score));
                final_mean[i].push_back(average(tmpv_score));
                final_median[i].push_back(median(tmpv_score));
                //final_sumasa[i].push_back(sum(tmpv_asa));
                final_atomsall[i].push_back(tmpv_score.size());
                final_atoms[i].push_back(n_greater_than(tmpv_score,1.0));
                final_atoms2[i].push_back(n_greater_than(tmpv_score,2.0));
                final_atomsall_chain[j] += final_atomsall[i].back();
                final_atoms_chain[j] += final_atoms[i].back();
                final_atoms2_chain[j] += final_atoms2[i].back();
                
                for(unsigned int k=0; k<tmpv_score.size(); k++){
                    if(tmpv_score[k]>1.0){
                        asa_score_g1[j].push_back(tmpv_asa[k]);
                    }
                }
            }
        }
    }
    
    vector<unsigned int> asa_l10(asa_score_g1.size(),0);
    vector<unsigned int> asa_10_50(asa_score_g1.size(),0);
    vector<unsigned int> asa_g50(asa_score_g1.size(),0);
    for(unsigned int i=0; i<asa_score_g1.size(); i++){
        for(unsigned int j=0; j<asa_score_g1[i].size(); j++){
            if(asa_score_g1[i][j]<10.0){
                asa_l10[i]++;
            } else if(asa_score_g1[i][j]>50.0){
                asa_g50[i]++;
            } else {
                asa_10_50[i]++;
            }
        }
    }
    
    /*if(final_max.size()!=asa_by_res.size()){
        cout << endl << "Error!!! " << final_max.size() << " " << asa_by_res.size() << endl;
        return;
    }
    for(unsigned int i=0; i<final_max.size(); i++){
        if(final_max[i].size()!=asa_by_res[i].size()){
            cout << endl << "Error!!! " << i << " " << final_max[i].size() << " " << asa_by_res[i].size() << endl;
            return;
        }
    }*/

    file = "final_multiple_global_res_allchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "resnum\tresid\tmax\tmin\tmean\tmedian\tn_g1\tn_g2\tn_all" << endl;
        for(unsigned int i=0; i<final_max.size(); i++){
            outfile.precision(3);
            for(unsigned int j=0; j<final_max[i].size(); j++){
                outfile << final_resnum[i] << "\t" << final_resid[i] << "\t" << final_max[i][j] << "\t" << final_min[i][j] << "\t" << final_mean[i][j] << "\t" << final_median[i][j] << "\t" << final_atoms[i][j] << "\t" << final_atoms2[i][j] << "\t" << final_atomsall[i][j] << endl;
            }
        }
        outfile.close();
    }
    
    file = "final_multiple_global_res_overall.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "chain_pair              \tn_align\tn_g1\tp_g1\tn_g2\tp_g2\tasa_l10\tasa1050\tasa_g50" << endl;
        for(unsigned int i=0; i<chainpair_names.size(); i++){
            outfile << chainpair_names[i] << "\t" << final_atomsall_chain[i] << "\t" << final_atoms_chain[i] << "\t" << 100*final_atoms_chain[i]/final_atomsall_chain[i] << "\t" << final_atoms2_chain[i] << "\t" << 100*final_atoms2_chain[i]/final_atomsall_chain[i] << "\t" << asa_l10[i] << "\t" << asa_10_50[i] << "\t" << asa_g50[i] << endl;
        }
        outfile.close();
    }
    
    vector<double> final_max_all;
    vector<double> final_mean_all;
    vector<double> final_median_all;
    //vector<double> final_maxsumasa;
    for(unsigned int i=0; i<final_vector_res.size(); i++){
        final_max_all.push_back(max(final_max[i]));
        //final_maxsumasa.push_back(max(asa_by_res[i]));
        vector<double> tmpv_score;
        for(unsigned int j=0; j<final_vector_res[i].size(); j++){
            for(unsigned int k=0; k<final_vector_res[i][j].size(); k++){
                if(final_vector_res[i][j][k]>=0.0){
                    tmpv_score.push_back(final_vector_res[i][j][k]);
                }
            }
        }
        final_mean_all.push_back(average(tmpv_score));
        final_median_all.push_back(median(tmpv_score));
    }
    
    file = "final_multiple_global_res_maxchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "resnum\tresid\tmax\tmean\tmedian" << endl;
        outfile.precision(3);
        for(unsigned int i=0; i<final_resnum.size(); i++){
            outfile << final_resnum[i] << "\t" << final_resid[i] << "\t" << final_max_all[i] << "\t" << final_mean_all[i] << "\t" << final_median_all[i] << endl;
        }
        outfile.close();
    }

/*    vector<string> res_id;
    vector<string> res_resid;
    vector<unsigned int> res_large;
    vector<double> res_max;
    res_id.push_back(pdb_v[0].get_orig_resnum(0));
    res_resid.push_back(pdb_v[0].get_resid(0));
    res_large.push_back(0);
    res_max.push_back(0.0);
    for(unsigned int i=0; i<scores.size(); i++){
        if(res_id.back()!=pdb_v[0].get_orig_resnum(i)){     //assume that atoms corresponding to same residue are always consecutive
            res_id.push_back(pdb_v[0].get_orig_resnum(i));
            res_resid.push_back(pdb_v[0].get_resid(i));
            res_large.push_back(0);
            res_max.push_back(0.0);
        }
        if(scores[i]>1.0){
            res_large.back()++;
        }
        if(scores[i]>res_max.back()){
            res_max.back()=scores[i];
        }
    }
    
    ////////////////////////////////////////
    

    vector<unsigned int> tmp;
    final_resnum.push_back(pdb_v[0].get_orig_resnum(0));
    final_resid.push_back(pdb_v[0].get_resid(0));
    final_max.push_back(all_scores[0]);
    final_min.push_back(all_scores[0]);
    final_sumasa.push_back(all_asa[0]);
    final_vector.push_back(final_tmp);
    final_vector[0].push_back(all_scores[0]);
    final_atoms.push_back(tmp);
    final_atoms2.push_back(tmp);
    final_atomsall.push_back(tmp);
    for(unsigned int i=0; i<all_scores[0].size(); i++){
        if(all_scores[0][i]>1){
            final_atoms[0].push_back(1);
        } else {
            final_atoms[0].push_back(0);
        }
        if(all_scores[0][i]>2){
            final_atoms2[0].push_back(1);
        } else {
            final_atoms2[0].push_back(0);
        }
        final_atomsall[0].push_back(1);
    }
    for(unsigned int i=1; i<all_scores.size(); i++){
        if(pdb_v[0].get_orig_resnum(i)!=final_resnum.back()){   //new residue
            final_resnum.push_back(pdb_v[0].get_orig_resnum(i));
            final_resid.push_back(pdb_v[0].get_resid(i));
            final_max.push_back(all_scores[i]);
            final_min.push_back(all_scores[i]);
            final_sumasa.push_back(all_asa[i]);
            final_vector.push_back(final_tmp);
            //cout << endl << "Here " << final_vector.back().size() << " " << all_scores[i].size();
            final_atoms.push_back(tmp);
            final_atoms2.push_back(tmp);
            final_atomsall.push_back(tmp);
            for(unsigned int j=0; j<all_scores[i].size(); j++){
                if(all_scores[i][j]>1){
                    final_atoms.back().push_back(1);
                } else {
                    final_atoms.back().push_back(0);
                }
                if(all_scores[i][j]>2){
                    final_atoms2.back().push_back(1);
                } else {
                    final_atoms2.back().push_back(0);
                }
                final_atomsall.back().push_back(1);
            }
        } else {
            for(unsigned int j=0; j<all_scores[i].size(); j++){
                //final_vector.back()[j].push_back(all_scores[i][j]);
                if(all_scores[i][j]>1){
                    final_atoms.back()[j]++;
                }
                if(all_scores[i][j]>2){
                    final_atoms2.back()[j]++;
                }
                final_atomsall.back()[j]++;
                if(all_scores[i][j]>final_max.back()[j]){
                    final_max.back()[j] = all_scores[i][j];
                }
                if(all_scores[i][j]<final_min.back()[j]){
                    final_min.back()[j] = all_scores[i][j];
                }
                final_sumasa.back()[j] += all_asa[i][j];
            }
        }
        final_vector.back().push_back(all_scores[i]);
    }
    //int tmpcin;
    //cin >> tmpcin;
  
    vector<vector<double> > final_vector_allchains;
    vector<double> final_mean_allchains;
    vector<double> final_median_allchains; 
    vector<double> final_maxsumasa_allchains; 
    for(unsigned int i=0; i<final_vector.size(); i++){  //each residue
        final_mean.push_back(final_tmp1);
        final_median.push_back(final_tmp1);
        final_vector_allchains.push_back(final_tmp1);
        final_maxsumasa_allchains.push_back(final_sumasa[i][0]);
        for(unsigned int j=0; j<final_vector[i].size(); j++){  //each atom
            final_mean[i].push_back(average(final_vector[i][j]));
            final_median[i].push_back(median(final_vector[i][j]));
            for(unsigned int k=0; k<final_vector[i][j].size(); k++){  //each chain-pair
                final_vector_allchains[i].push_back(final_vector[i][j][k]);
            }
            if(final_maxsumasa_allchains.back()<final_sumasa[i][j]){
                final_maxsumasa_allchains.back()=final_sumasa[i][j];
            }
        }
        final_mean_allchains.push_back(average(final_vector_allchains[i]));
        final_median_allchains.push_back(median(final_vector_allchains[i]));
    }
    
    vector<unsigned int> final_nonzero;
    for(unsigned int i=0; i<final_atoms.size(); i++){
        final_nonzero.push_back(0);
        for(unsigned int j=0; j<final_atoms[i].size(); j++){
            if(final_atoms[i][j]>0){
                final_nonzero[i]++;
            }
        }
    }
    file = "all_pairwise_res_max.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<final_resnum.size(); i++){
            outfile << final_resnum[i] << "\t" << final_resid[i];
            for(unsigned int j=0; j<final_max[i].size(); j++){
                outfile.precision(2);
                outfile << "\t" << final_max[i][j];
            }
            outfile << endl;
        }
        outfile.close();
    }
    file = "final_multiple_global_res_allchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "resnum\tresid\tmax\tmin\tmean\tmedian\tsumasa\tn_g1\tn_g2\tn_all" << endl;
        for(unsigned int i=0; i<final_resnum.size(); i++){
            outfile.precision(3);
            for(unsigned int j=0; j<final_max[i].size(); j++){
                outfile << final_resnum[i] << "\t" << final_resid[i] << "\t" << final_max[i][j] << "\t" << final_min[i][j] << "\t" << final_mean[i][j] << "\t" << final_median[i][j] << "\t" << final_sumasa[i][j] << "\t" << final_atoms[i][j] << "\t" << final_atoms2[i][j] << "\t" << final_atomsall[i][j] << endl;
            }
        }
        outfile.close();
    }
    file = "all_pairwise_res_min.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<final_resnum.size(); i++){
            outfile << final_resnum[i] << "\t" << final_resid[i];
            for(unsigned int j=0; j<final_min[i].size(); j++){
                outfile.precision(2);
                outfile << "\t" << final_min[i][j];
            }
            outfile << endl;
        }
        outfile.close();
    }
    file = "all_pairwise_res_atoms.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "\t\t";
        for(unsigned int i=0; i<obj.size()-1; i++){
            outfile << "\t|";
            for(unsigned int j=i+1; j<obj.size(); j++){
                outfile << "\t" << obj[i] << "_" << obj[j];
            }
        }
        outfile << endl;
        int tmp = obj.size()-1;
        int tmp1 = obj.size()-1;
        outfile << "\t\t\t|";
        for(unsigned int i=0; i<final_atoms[0].size(); i++){
            unsigned int tmp0 = 0;
            for(unsigned int j=0; j<final_atoms.size(); j++){
                tmp0 += final_atoms[j][i];
            }
            if(i==tmp1){
                outfile << "\t|";
                tmp--;
                tmp1+=tmp;
            }
            outfile << "\t" << tmp0;
        }
        outfile << endl;
        for(unsigned int i=0; i<final_resnum.size(); i++){
            outfile << final_resnum[i] << "\t" << final_resid[i] << "\t" << final_nonzero[i] << "\t|";
            int tmp = obj.size()-1;
            int tmp1 = obj.size()-1;
            for(unsigned int j=0; j<final_atoms[i].size(); j++){
                if(j==tmp1){
                    outfile << "\t|";
                    tmp--;
                    tmp1+=tmp;
                }
                outfile << "\t" << final_atoms[i][j];
            }
            outfile << endl;
        }
        outfile.close();
    }
    
    //////////
    
    file = "all_pairwise_maxasa.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<all_scores.size(); i++){
            outfile << pdb_v[0].get_orig_resnum(i) << "\t" << pdb_v[0].get_resid(i) << "\t" << delete_spaces(pdb_v[0].get_atom(i));
            for(unsigned int j=0; j<all_asa[i].size(); j++){
                outfile << "\t" << all_asa[i][j];
            }
            outfile << endl;
        }
        outfile.close();
    }
    file = "final_multiple_global_atoms_maxchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "resnum\tresid\tatom\tdist" << endl;
        outfile.precision(3);
        for(unsigned int i=0; i<scores.size(); i++){
            outfile << pdb_v[0].get_orig_resnum(i) << "\t" << pdb_v[0].get_resid(i) << "\t" << delete_spaces(pdb_v[0].get_atom(i)) << "\t" << scores[i] << endl;
        }
        outfile.close();
    }
    
    res_id.push_back(pdb_v[0].get_orig_resnum(0));
    res_resid.push_back(pdb_v[0].get_resid(0));
    res_large.push_back(0);
    res_max.push_back(0.0);
    for(unsigned int i=0; i<scores.size(); i++){
        if(res_id.back()!=pdb_v[0].get_orig_resnum(i)){     //assume that atoms corresponding to same residue are always consecutive
            res_id.push_back(pdb_v[0].get_orig_resnum(i));
            res_resid.push_back(pdb_v[0].get_resid(i));
            res_large.push_back(0);
            res_max.push_back(0.0);
        }
        if(scores[i]>1.0){
            res_large.back()++;
        }
        if(scores[i]>res_max.back()){
            res_max.back()=scores[i];
        }
    }
    file = "max_dist_residues.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(unsigned int i=0; i<res_id.size(); i++){
            outfile << res_id[i] << "\t" << res_resid[i] << "\t" << res_large[i] << "\t" << res_max[i] << endl;
        }
        outfile.close();
    }
    file = "final_multiple_global_res_maxchains.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "resnum\tresid\tmax\tmean\tmedian\tmaxsumasa" << endl;
        outfile.precision(3);
        for(unsigned int i=0; i<res_id.size(); i++){
            outfile << res_id[i] << "\t" << res_resid[i] << "\t" << res_max[i] << "\t" << final_mean_allchains[i] << "\t" << final_median_allchains[i] << "\t" << final_maxsumasa_allchains[i] << endl;
        }
        outfile.close();
    }*/
    
    return;
}    

void output_pdb(vector<string> &info, string &filein, string &fileout, char chain, Transform &transform, bool DO_TRANSFORM, bool OUT_PDB_FULL) 
{ 
    string line;
    ifstream infile(filein.c_str(), ios::in);
    ofstream outfile;
    if(infile.is_open()){
        outfile.open(fileout.c_str());
		if(outfile.is_open()){
			outfile << "# ProSMART PDB File" << endl;
			for(unsigned int i=0; i<info.size(); i++){
				outfile << info[i] << endl;
			}
            while(!infile.eof()){
                line.clear();
                getline(infile,line);
                
                if(line.size()<66)continue;                     //minimum length to populate a 'pdbline' entry
                
                if(!OUT_PDB_FULL && chain!=' ' && line[21]!=chain)continue;		//only take this chain, unless chain is not specified
                
                if(line.substr(0,6)!="ATOM  "){
                    if(line.substr(0,6)!="HETATM"){
                        continue;
                    } //else if(line.substr(17,3)!="MSE")continue;
                }
                
                /*stringstream ss;      //this renames all residue numbers by -600 (useful if need to combine two chains into one)
                 ss.str("");
                 ss << (atoi(line.substr(22,4).c_str())-600);
                 while(ss.str().size()!=3)ss << " ";
                 line.replace(23,3,ss.str());*/
                /*stringstream ss;      //this renumbers all atom names (useful if need to reorder chains)
                 ss.str("");
                 idx++;
                 ss << idx;
                 while(ss.str().size()!=4)ss << " ";
                 line.replace(7,4,ss.str());*/
                
                if(DO_TRANSFORM){
                    coord crd_orig, crd_new;
                    crd_orig.x = atof(line.substr(30,8).c_str());
                    crd_orig.y = atof(line.substr(38,8).c_str());
                    crd_orig.z = atof(line.substr(46,8).c_str());
                    crd_new = transform.transform(crd_orig);            //transform coords
                    
                    string tempx = standardise_double(d_to_str(crd_new.x),8,3);              //format coords correctly
                    string tempy = standardise_double(d_to_str(crd_new.y),8,3);
                    string tempz = standardise_double(d_to_str(crd_new.z),8,3);
                    
                    line.replace(30,24,"                        ");
                    line.replace(38-tempx.size(),tempx.size(),tempx);   //replace only coords in the line
                    line.replace(46-tempy.size(),tempy.size(),tempy);
                    line.replace(54-tempz.size(),tempz.size(),tempz);
                }
                
                outfile << line << endl;
            }
			outfile.close();
			cout << endl << "PDB written to: " << fileout.c_str();
        }
		else cout << "Unable to open output file " << fileout.c_str() << " for writing" << endl;
        infile.close();
    } else {
        cout << "Unable to open " << filein << " for reading" << endl;
    }
	
    return;
}


void output_multi_res(string &error_file, vector<vector<string> > &scores_in, string &fileout)
{
  ofstream outfile;
  outfile.open(fileout.c_str());
  if(outfile.is_open()){
    for(unsigned int k=0; k<scores_in[0].size()-1; k+=2){
			outfile << scores_in[0][k];
			for(unsigned int j=0; j<scores_in.size(); j++){
				outfile << "\t" << scores_in[j][k+1];
			}
			outfile << endl;
    }
    outfile.close();
    cout << endl << "Multiple residue scores file written to: " << fileout;
  } else {
    xml_entry(error_file,0,5,fileout);
    cout << "Unable to open output file " << fileout << " for writing" << endl;
  }
  return;
}


vector<int> convert_alignment(vector<alignment> &align_v, vector<string> &res_idx)
//returns a vector that indexes the aligned residues (in align_v) by their position in the res_idx vector.
//length(result)==length(res_idx)
{
  vector<int> result;
  
  int align_idx=0;
  for(unsigned int i=0; i<res_idx.size(); i++){
    if((int)align_v.size()>align_idx){
			if(res_idx[i]==align_v[align_idx].res1){
				result.push_back(align_idx);
				align_idx++;
			} else {
				result.push_back(-1);
			} 
		} else {
			result.push_back(-1);
		}
  }
	
  /*for(int i=0; i<res_idx.size(); i++){
	 cout << endl << i << " " << res_idx[i] << "\t" << result[i] << " ";
	 if(result[i] >= 0){
	 cout << align_v[result[i]].res1 << "\t" << align_v[result[i]].rmsd;
	 }
	 }*/
	
  return result;
}

void multiple_sidechain(vector<vector<int> > &indexed_alignments, vector<vector<alignment> > &align_v, vector<string> &res_idx, string &fileout)
{
  for(unsigned int i=0; i<res_idx.size(); i++){
		cout << endl << i;
		for(unsigned int j=0; j<indexed_alignments.size(); j++){
			cout << " " << indexed_alignments[j][i];
		}
		cout << "\t\t" << res_idx[i];
		for(unsigned int j=0; j<indexed_alignments.size(); j++){
			if(indexed_alignments[j][i]>=0){
				cout << " " << align_v[j][indexed_alignments[j][i]].res1;
			}
		}
  }
  
  cout << endl << "fileout: " << fileout << endl;
	
  return;
}

vector<alignment> filter_res_alignment(vector<alignment> &initial, double main_cutoff, double side_cutoff)
{
  vector<alignment> result;
  
  for(unsigned int i=0; i<initial.size(); i++){
    if(initial[i].res1==initial[i].res2){	//ncs restraint conditions
			if(initial[i].minimum <= main_cutoff){
				if(initial[i].rmsd <= side_cutoff){
					//cout << endl << initial[i].res1 << " " << initial[i].res2 << " " << initial[i].central<< " " << initial[i].minimum << " " << initial[i].rmsd << " " << initial[i].av; 
					result.push_back(initial[i]);
				}}}
  }
  return result;
}


void write_scores_triangular(string &error_file, vector<string> &filenames1, vector<string> &chains1, string &SINGLE_dir, string &MULTI_dir)
{
  string fileoutname="";
  vector<string> header;
  vector<vector<string> > s1;
  vector<vector<string> > s2;
  vector<vector<string> > s3;
  vector<vector<string> > s4;
  vector<vector<string> > s5;
  vector<vector<string> > s6;
    vector<vector<string> > s7;
  vector<string> t1;
  vector<string> t2;
  vector<string> t3;
  vector<string> t4;
  vector<string> t5;
  vector<string> t6;
    vector<string> t7;
  vector<string> temp;
  string filein_temp;
  string filein;
	
  for(unsigned int i=0; i<chains1.size()-1; i++){	//read score info from file
		t1.clear();
		t2.clear();
		t3.clear();
		t4.clear();
		t5.clear();
		t6.clear();
      t7.clear();
		filein_temp = SINGLE_dir + get_filename(filenames1[i]) + "_" + chains1[i] + "_";
		for(unsigned int j=i+1; j<chains1.size(); j++){
			filein = filein_temp + get_filename(filenames1[j]) + "_" + chains1[j] + ".txt";
			temp = read_file(filein);
			if(temp.size()<14){
				xml_entry(error_file,1,1);
				cout << endl << "Error encountered with file: " << filein << endl;
                t1.push_back("-1");
                t2.push_back("-1");
                t3.push_back("-1");
                t4.push_back("-1");
                t5.push_back("-1");
                t6.push_back("-1");
                t7.push_back("-1");
				continue;
			}
			t1.push_back(temp[1]);
			t2.push_back(temp[3]);
			t3.push_back(temp[5]);
			t4.push_back(temp[7]);
			t5.push_back(temp[9]);
			t6.push_back(temp[11]);
			t7.push_back(temp[13]);
		}
		s1.push_back(t1);
		s2.push_back(t2);
		s3.push_back(t3);
		s4.push_back(t4);
		s5.push_back(t5);
		s6.push_back(t6);
      s7.push_back(t7);
  }
  
  /*for(int i=0; i<chains1.size(); i++){
	 fileoutname += get_filename(filenames1[i]) + chains1[i] + "_";
	 }
	 string Aligned_path = sub_dir + fileoutname + "Aligned.txt";
	 string Percent_path = sub_dir + fileoutname + "Percent.txt";
	 string SeqIden_path = sub_dir + fileoutname + "SeqIden.txt";
	 string Central_path = sub_dir + fileoutname + "Central.txt";
	 string Minimum_path = sub_dir + fileoutname + "Minimum.txt";
	 string RMSD_path = sub_dir + fileoutname + "RMSD.txt";*/
  
  string Aligned_path = MULTI_dir + "NumberAligned";
  string Percent_path = MULTI_dir + "PercentAligned";
  string SeqIden_path = MULTI_dir + "SequenceIdentity";
  string Central_path = MULTI_dir + "AverageProcrustes";
  string Minimum_path = MULTI_dir + "AverageFlexible";
  string RMSD_path = MULTI_dir + "GlobalRMSD";
    string LowFrag_path = MULTI_dir + "BestFragmentScore";
  
  for(unsigned int i=0; i<chains1.size(); i++){
    header.push_back(get_filename(filenames1[i]) + chains1[i]);		//generate column titles
  }
  /*for(unsigned int i=0; i<chains1.size()-1; i++){
	 row_title += " " + get_filename(filenames1[i]) + chains1[i];		//generate row titles
	 }
	 header.push_back(col_title);
	 header.push_back(row_title);*/
	
  write_matrix_triangular(error_file,Aligned_path,s1,header,-1,-1,0,0);
  write_matrix_triangular(error_file,Percent_path,s2,header,0,100,0,100);
  write_matrix_triangular(error_file,SeqIden_path,s3,header,0,40,0,100);
  write_matrix_triangular_SI(error_file,Central_path,s4,header,0.8,2.0,1,0,s3);
	//write_matrix_triangular(error_file,Central_path,s4,header,0.8,2.0,1,0);
  write_matrix_triangular(error_file,Minimum_path,s5,header,0.5,1.5,1,0);
  write_matrix_triangular(error_file,RMSD_path,s6,header,0,5,1,0);
    write_matrix_triangular(error_file,LowFrag_path,s7,header,0,1,1,0);
	/*if(temp.size()==16){
	 write_matrix_triangular(error_file,NrmProc_path,s7,header,0.001,0.05,1,0);
	 write_matrix_triangular(error_file,KolSmir_path,s8,header,0.5,1.0,0,1);
	 }*/
  return;
}

vector<vector<double> > write_scores(string &error_file, vector<string> &filenames1, vector<string> &filenames2, vector<string> &chains1, vector<string> &chains2, string &SINGLE_dir, string &MULTI_dir, bool HBOND_TYPE_ALIGNMENT)
{
  string fileoutname="";
  vector<string> header1;
	vector<string> header2;
  vector<vector<string> > s1;
  vector<vector<string> > s2;
  vector<vector<string> > s3;
  vector<vector<string> > s4;
  vector<vector<string> > s5;
  vector<vector<string> > s6;
    vector<vector<string> > s7;
  vector<string> t1;
  vector<string> t2;
  vector<string> t3;
  vector<string> t4;
  vector<string> t5;
  vector<string> t6;
    vector<string> t7;
  vector<string> temp;
  string filein_temp;
  string filein;
  
  vector<vector<double> > d4;
  vector<double> d4i;
	
//#define TMP_OUTPUT_HITS
#ifdef TMP_OUTPUT_HITS
    vector<string> tmp_hits;
    vector<string> tmp_hits_chain;
    vector<double> tmp_hits_val;
#endif
  for(unsigned int i=0; i<chains1.size(); i++){	//read score info from file
		t1.clear();
		t2.clear();
		t3.clear();
		t4.clear();
		t5.clear();
		t6.clear();
      t7.clear();
		filein_temp = SINGLE_dir + get_filename(filenames1[i]) + "_" + chains1[i] + "_";
		for(unsigned int j=0; j<chains2.size(); j++){
            if(HBOND_TYPE_ALIGNMENT && i!=j)continue;
			filein = filein_temp + get_filename(filenames2[j]) + "_" + chains2[j] + ".txt";
			temp = read_file(filein);
			if(temp.size()<14){
				xml_entry(error_file,1,1);
				cout << endl << "Error encountered with file: " << filein << endl;
                t1.push_back("-1");
                t2.push_back("-1");
                t3.push_back("-1");
                t4.push_back("-1");
                t5.push_back("-1");
                t6.push_back("-1");
                t7.push_back("-1");
				continue;
			}
			t1.push_back(temp[1]);
			t2.push_back(temp[3]);
			t3.push_back(temp[5]);
			t4.push_back(temp[7]);
			t5.push_back(temp[9]);
			t6.push_back(temp[11]);
			t7.push_back(temp[13]);
#ifdef TMP_OUTPUT_HITS
            if(str_to_double(temp[9])<=1.0){
                tmp_hits.push_back(get_filename(filenames2[j]));
                tmp_hits_chain.push_back(chains2[j]);
                tmp_hits_val.push_back(str_to_double(temp[9]));
            }
#endif
		}
		s1.push_back(t1);
		s2.push_back(t2);
		s3.push_back(t3);
		s4.push_back(t4);
		s5.push_back(t5);
		s6.push_back(t6);
      s7.push_back(t7);
  }
    
#ifdef TMP_OUTPUT_HITS
    vector<string> tmp_hits_order;
    vector<string> tmp_hits_chain_order;
    vector<double> tmp_hits_val_order;
    while(tmp_hits.size()>0){
        unsigned int tmp_i = 0;
        for(unsigned int j=1; j<tmp_hits.size(); j++){
            if(tmp_hits_val[tmp_i] > tmp_hits_val[j]){
                tmp_i = j;
            }
        }
        tmp_hits_order.push_back(tmp_hits[tmp_i]);
        tmp_hits_chain_order.push_back(tmp_hits_chain[tmp_i]);
        tmp_hits_val_order.push_back(tmp_hits_val[tmp_i]);
        tmp_hits.erase(tmp_hits.begin()+tmp_i);
        tmp_hits_chain.erase(tmp_hits_chain.begin()+tmp_i);
        tmp_hits_val.erase(tmp_hits_val.begin()+tmp_i);
    }
    cout << endl;
    for(unsigned int i=0; i<tmp_hits_order.size(); i++){
        cout << tmp_hits_order[i] << " ";
    }
    cout << endl;
    for(unsigned int i=0; i<tmp_hits_chain_order.size(); i++){
        cout << tmp_hits_chain_order[i] << " ";
    }
    cout << endl;
    for(unsigned int i=0; i<tmp_hits_val_order.size(); i++){
        cout << tmp_hits_val_order[i] << " ";
    }
    cout << endl;
#endif
    
   string Aligned_path = MULTI_dir + "NumberAligned";
   string Percent_path = MULTI_dir + "PercentAligned";
   string SeqIden_path = MULTI_dir + "SequenceIdentity";
   string Central_path = MULTI_dir + "AverageProcrustes";
   string Minimum_path = MULTI_dir + "AverageFlexible";
   string RMSD_path = MULTI_dir + "GlobalRMSD";
   string LowFrag_path = MULTI_dir + "BestFragmentScore";

	for(unsigned int i=0; i<chains1.size(); i++){
    header1.push_back(get_filename(filenames1[i]) + chains1[i]);		//generate column titles
  }
	for(unsigned int i=0; i<chains2.size(); i++){
    header2.push_back(get_filename(filenames2[i]) + chains2[i]);		//generate column titles
  }
  
	write_matrix(error_file,Aligned_path,s1,header1,header2,-1,-1,0);
  write_matrix(error_file,Percent_path,s2,header1,header2,0,100,0);
  write_matrix(error_file,SeqIden_path,s3,header1,header2,0,40,0);
  write_matrix(error_file,Central_path,s4,header1,header2,0.5,2.0,1);
  write_matrix(error_file,Minimum_path,s5,header1,header2,0.5,1.5,1);
  write_matrix(error_file,RMSD_path,s6,header1,header2,0,5,1);
    write_matrix(error_file,LowFrag_path,s7,header1,header2,0,1,1);

  for(unsigned int i=0; i<s4.size(); i++){
    d4i.clear();
    for(unsigned int j=0; j<s4[i].size(); j++){
      d4i.push_back(atof(s4[i][j].c_str()));
    }
		d4.push_back(d4i);
  }
	
  return d4;
}

void write_scores_allonall(string &error_file, vector<string> &filenames1, vector<string> &chains1, string &SINGLE_dir, string &MULTI_dir)
{
  string fileoutname="";
  vector<string> header;
  vector<vector<string> > s1;
  vector<vector<string> > s2;
  vector<vector<string> > s3;
  vector<vector<string> > s4;
  vector<vector<string> > s5;
  vector<vector<string> > s6;
    vector<vector<string> > s7;
  vector<string> t1;
  vector<string> t2;
  vector<string> t3;
  vector<string> t4;
  vector<string> t5;
  vector<string> t6;
    vector<string> t7;
  vector<string> temp;
  string filein_temp;
  string filein;
	
  for(unsigned int i=0; i<chains1.size(); i++){	//read score info from file
		t1.clear();
		t2.clear();
		t3.clear();
		t4.clear();
		t5.clear();
		t6.clear();
      t7.clear();
		filein_temp = SINGLE_dir + get_filename(filenames1[i]) + "_" + chains1[i] + "_";
		for(unsigned int j=0; j<chains1.size(); j++){
			if(i==j){
				t1.push_back("");
				t2.push_back("");
				t3.push_back("");
				t4.push_back("");
				t5.push_back("");
				t6.push_back("");
				t7.push_back("");
				continue;
			}
			filein = filein_temp + get_filename(filenames1[j]) + "_" + chains1[j] + ".txt";
			temp = read_file(filein);
			if(temp.size()<14){
				xml_entry(error_file,1,1);
				cout << endl << "Error encountered with file: " << filein << endl;
                t1.push_back("-1");
                t2.push_back("-1");
                t3.push_back("-1");
                t4.push_back("-1");
                t5.push_back("-1");
                t6.push_back("-1");
                t7.push_back("-1");
				continue;
			}
			t1.push_back(temp[1]);
			t2.push_back(temp[3]);
			t3.push_back(temp[5]);
			t4.push_back(temp[7]);
			t5.push_back(temp[9]);
			t6.push_back(temp[11]);
			t7.push_back(temp[13]);
		}
		s1.push_back(t1);
		s2.push_back(t2);
		s3.push_back(t3);
		s4.push_back(t4);
		s5.push_back(t5);
		s6.push_back(t6);
      s7.push_back(t7);
  }
  
   string Aligned_path = MULTI_dir + "NumberAligned";
   string Percent_path = MULTI_dir + "PercentAligned";
   string SeqIden_path = MULTI_dir + "SequenceIdentity";
   string Central_path = MULTI_dir + "AverageProcrustes";
   string Minimum_path = MULTI_dir + "AverageFlexible";
   string RMSD_path = MULTI_dir + "GlobalRMSD";
   string LowFrag_path = MULTI_dir + "BestFragmentScore";
  for(unsigned int i=0; i<chains1.size(); i++){
    header.push_back(get_filename(filenames1[i]) + chains1[i]);		//generate column titles
  }
	
  write_matrix_all_on_all(error_file,Aligned_path,s1,header,-1,-1,0,0);
  write_matrix_all_on_all(error_file,Percent_path,s2,header,0,100,0,100);
  write_matrix_all_on_all(error_file,SeqIden_path,s3,header,0,40,0,100);
  write_matrix_all_on_all(error_file,Central_path,s4,header,0.5,2.0,1,0);
  write_matrix_all_on_all(error_file,Minimum_path,s5,header,0.5,1.5,1,0);
  write_matrix_all_on_all(error_file,RMSD_path,s6,header,0,5,1,0);
    write_matrix_all_on_all(error_file,LowFrag_path,s7,header,0,1,1,0);
  return;
}

void write_matrix(string &error_file, string fileout_, vector<vector<string> > &matr, vector<string> &header1, vector<string> &header2, double MIN, double MAX, bool DIRECTION)
//DIRECTION == 0 -> low scores are bad.
//DIRECTION == 1 -> low scores are good.
//MAX < 0 -> dont use colours
{
	double MULTIPLIER = 0.0;
	if(MAX > MIN){
		MULTIPLIER = 1.0/(MAX-MIN);
	}
	string fileout = fileout_ + ".html";
	string fileout_txt = fileout_ + ".txt";
	
	vector<vector<double> > scores;
	vector<double> temp;
	double temp1 = 0.0;
	
	if(MAX >= 0.0){
		for(unsigned int i=0; i<matr.size(); i++){
			temp.clear();
			for(unsigned int j=0; j<matr[i].size(); j++){
				temp1 = (atof(matr[i][j].c_str())-MIN)*MULTIPLIER;
				if(temp1 < 0.0){
					temp1 = 0.0;
				}
				if(temp1 > 1.0){
					temp1 = 1.0;
				}
				if(DIRECTION == 0){
					temp.push_back(temp1);
				} else {
					temp.push_back(1.0-temp1);
				}
			}
			scores.push_back(temp);
		}
	}
	
	ofstream outfile;
	ofstream outfile_txt;
  outfile.open(fileout.c_str());
	outfile_txt.open(fileout_txt.c_str());
  if(outfile.is_open()){
		if(outfile_txt.is_open()){
			outfile << "<!--ProSMART Multiple Scores File-->" << endl;
			
			if(MAX >= 0){	//use colours - create function to load stylesheet
				outfile << "<head><script language=\"JavaScript\">" << endl
				<< "function setActiveStyleSheet(title) {var i, a, main;" << endl
				<< "for(i=0; (a = document.getElementsByTagName(\"link\")[i]); i++) {if(a.getAttribute(\"rel\").indexOf(\"style\") != -1 && a.getAttribute(\"title\")) {a.disabled = true; if(a.getAttribute(\"title\") == title) a.disabled = false;}}}"
				<< "</script></head>" << endl;
			}
			
			outfile << "<html><style type='text/css'>" << endl
			<< "tr{font-size:70%;}" << endl 
			<< "a{font-family:geneva, 'trebuchet MS', garuda, helvetica, arial, sans-serif, sans, verdana; font-size:70%;}" << endl
			<< "</style>" << endl
			<< "<LINK rel=\"alternate stylesheet\" href=\".black.css\" type=\"text/css\" title=\"black\">" << endl
			<< "<body onload=\"setActiveStyleSheet('black');\"><table border='0'>" << endl;
			outfile << "<tr><td></td>";
			for(unsigned int i=0; i<header2.size(); i++){
				outfile << "<td>" << header2[i] << "</td>";
			}
			outfile << "</tr>";
			for(unsigned int i=0; i<matr.size(); i++){
				outfile << endl << "<tr><td>" << header1[i] << "</td>";	
				for(unsigned int j=0; j<matr[i].size(); j++){
					if(MAX < 0){	//dont use colours
						outfile << "<td>" << matr[i][j] << "</td>";
					} else {
						outfile << "<td style='color:rgb(" 
						<< (int)(prosmart_green.x*scores[i][j] + prosmart_red.x*(1-scores[i][j])) << ","
						<< (int)(prosmart_green.y*scores[i][j] + prosmart_red.y*(1-scores[i][j])) << "," 
						<< (int)(prosmart_green.z*scores[i][j] + prosmart_red.z*(1-scores[i][j])) << ")'>" << matr[i][j] << "</td>";
					}
				}
				outfile << "</tr>";
			}
			outfile << endl << "</table>" << endl;
			
			if(MAX >= 0){	//use colours - allow colour changing
				outfile << endl << "<br /><p>" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('black'); return false;\">black</a><br />" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('default'); return false;\">colour</a>" << endl
				<< "</p>" << endl;
			}
			
			outfile << endl << "</body></html>" << endl;
			//cout << endl << "File written to: " << fileout;
			
			outfile_txt << "# ProSMART Multiple Scores File" << endl;
            //outfile_txt << "-";
            for(unsigned int i=0; i<header2.size(); i++){
				outfile_txt << "\t" << header2[i];
			}
            outfile_txt << endl;
			for(unsigned int i=0; i<matr.size(); i++){
                outfile_txt << header1[i];	
				for(unsigned int j=0; j<matr[i].size(); j++){
					outfile_txt << "\t" << matr[i][j];
				}
				outfile_txt << endl;
			}
			
			outfile_txt.close();
		} else {
			xml_entry(error_file,0,5,fileout_txt);
			cout << "Unable to open output file " << fileout_txt << " for writing" << endl;
		}
		outfile.close();
  } else {
    xml_entry(error_file,0,5,fileout);
    cout << "Unable to open output file " << fileout << " for writing" << endl;
  }
	
  return;
}

void write_matrix_triangular_SI(string &error_file, string fileout_, vector<vector<string> > &matr, vector<string> &header, double MIN, double MAX, bool DIRECTION, double val, vector<vector<string> > &SI)
//DIRECTION == 0 -> low scores are bad.
//DIRECTION == 1 -> low scores are good.
//MAX < 0 -> dont use colours
{
	double MULTIPLIER = 0.0;
	if(MAX > MIN){
		MULTIPLIER = 1.0/(MAX-MIN);
	}
	string fileout = fileout_ + ".html";
	string fileout_txt = fileout_ + ".txt";
	
	unsigned int N = header.size();
	vector<vector<double> > scores;
	vector<double> temp;
	double temp1;
	
	vector<vector<double> > all;
	vector<double> temp2;
	for(unsigned int i=0; i<N; i++){
		temp2.push_back(val);
	}
	for(unsigned int i=0; i<N; i++){
		all.push_back(temp2);
	}	
	
	int j_ = 0;
	if(MAX < 0){	//no colours are to be used
		for(unsigned int i=0; i<matr.size(); i++){
			for(unsigned int j=0; j<matr[i].size(); j++){
				j_ = i+j+1;
				all[i][j_] = atof(matr[i][j].c_str());
				all[j_][i] = all[i][j_];
			}
		}
	} else {
		for(unsigned int i=0; i<matr.size(); i++){
			temp.clear();
			for(unsigned int j=0; j<matr[i].size(); j++){
				j_ = i+j+1;
				all[i][j_] = atof(matr[i][j].c_str());
				all[j_][i] = all[i][j_];
				temp1 = (all[i][j_]-MIN)*MULTIPLIER;
				if(temp1 < 0.0){
					temp1 = 0.0;
				}
				if(temp1 > 1.0){
					temp1 = 1.0;
				}
				if(DIRECTION == 0){
					temp.push_back(temp1);
				} else {
					temp.push_back(1.0-temp1);
				}
			}
			scores.push_back(temp);
		}
	}
	
	ofstream outfile;
	ofstream outfile_txt;
  outfile.open(fileout.c_str());
	outfile_txt.open(fileout_txt.c_str());
  if(outfile.is_open()){
		if(outfile_txt.is_open()){
			outfile << "<!--ProSMART Multiple Scores File-->" << endl;
			
			if(MAX >= 0){	//use colours - create function to load stylesheet
				outfile << "<head><script language=\"JavaScript\">" << endl
				<< "function setActiveStyleSheet(title) {var i, a, main;" << endl
				<< "for(i=0; (a = document.getElementsByTagName(\"link\")[i]); i++) {if(a.getAttribute(\"rel\").indexOf(\"style\") != -1 && a.getAttribute(\"title\")) {a.disabled = true; if(a.getAttribute(\"title\") == title) a.disabled = false;}}}"
				<< "</script></head>" << endl;
			}
			
			outfile << "<html><style type='text/css'>" << endl
			<< "tr{font-size:70%;}" << endl 
			<< "a{font-family:geneva, 'trebuchet MS', garuda, helvetica, arial, sans-serif, sans, verdana; font-size:70%;}" << endl
			<< "</style>" << endl
			<< "<LINK rel=\"alternate stylesheet\" href=\".black.css\" type=\"text/css\" title=\"black\">" << endl
			<< "<body><table border='0'>" << endl;
			outfile << "<tr><td></td>";
			for(unsigned int i=0; i<header.size(); i++){
				outfile << "<td>" << header[i] << "</td>";
			}
			outfile << "</tr>";
			for(unsigned int i=0; i<matr.size(); i++){
				outfile << endl << "<tr><td>" << header[i] << "</td>";
				for(unsigned int j=0; j<i; j++){
					outfile << "<td style='color:rgb(200,200,200)'>" << SI[j][i-1-j] << "</td>";
				}
				outfile << "<td></td>";
				for(unsigned int j=0; j<matr[i].size(); j++){
					if(MAX < 0){	//dont use colours
						outfile << "<td>" << matr[i][j] << "</td>";
					} else {
						outfile << "<td style='color:rgb(" 
						<< (int)(prosmart_green.x*scores[i][j] + prosmart_red.x*(1-scores[i][j])) << ","
						<< (int)(prosmart_green.y*scores[i][j] + prosmart_red.y*(1-scores[i][j])) << "," 
						<< (int)(prosmart_green.z*scores[i][j] + prosmart_red.z*(1-scores[i][j])) << ")'>" << matr[i][j] << "</td>";
					}
				}
				outfile << "</tr>";
			}
			outfile << endl << "<tr><td>" << header[matr.size()] << "</td>";
			for(unsigned int j=0; j<SI.size(); j++){
				outfile << "<td style='color:rgb(200,200,200)'>" << SI[j][SI.size()-1-j] << "</td>";
			}
			outfile << "<td></td>";
			outfile << endl << "</table>" << endl;
			
			if(MAX >= 0){	//use colours - allow colour changing
				outfile << endl << "<br /><p>" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('black'); return false;\">black</a><br />" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('default'); return false;\">colour</a>" << endl
				<< "</p>" << endl;
			}
			
			outfile << endl << "</body></html>" << endl;			//cout << endl << "File written to: " << fileout;
			
			outfile_txt << "# ProSMART Multiple Scores File" << endl;
            //outfile_txt << "-";
            for(unsigned int i=0; i<header.size(); i++){
				outfile_txt << "\t" << header[i];
			}
            outfile_txt << endl;
			for(unsigned int i=0; i<N; i++){
                outfile_txt << header[i];
				for(unsigned int j=0; j<N; j++){
					outfile_txt << "\t" << all[i][j];
				}
				outfile_txt << endl;
			}
			
			outfile_txt.close();
		} else {
			xml_entry(error_file,0,5,fileout_txt);
			cout << "Unable to open output file " << fileout_txt << " for writing" << endl;
		}
		outfile.close();
  } else {
    xml_entry(error_file,0,5,fileout);
    cout << "Unable to open output file " << fileout << " for writing" << endl;
  }
	
  return;
}

void write_matrix_triangular(string &error_file, string fileout_, vector<vector<string> > &matr, vector<string> &header, double MIN, double MAX, bool DIRECTION, double val)
//DIRECTION == 0 -> low scores are bad.
//DIRECTION == 1 -> low scores are good.
//MAX < 0 -> dont use colours
{
	double MULTIPLIER = 0.0;
	if(MAX > MIN){
		MULTIPLIER = 1.0/(MAX-MIN);
	}
	string fileout = fileout_ + ".html";
	string fileout_txt = fileout_ + ".txt";
	
	unsigned int N = header.size();
	vector<vector<double> > scores;
	vector<double> temp;
	double temp1;
	
	vector<vector<double> > all;
	vector<double> temp2;
	for(unsigned int i=0; i<N; i++){
		temp2.push_back(val);
	}
	for(unsigned int i=0; i<N; i++){
		all.push_back(temp2);
	}	
	
	int j_ = 0;
	if(MAX < 0){	//no colours are to be used
		for(unsigned int i=0; i<matr.size(); i++){
			for(unsigned int j=0; j<matr[i].size(); j++){
				j_ = i+j+1;
				all[i][j_] = atof(matr[i][j].c_str());
				all[j_][i] = all[i][j_];
			}
		}
	} else {
		for(unsigned int i=0; i<matr.size(); i++){
			temp.clear();
			for(unsigned int j=0; j<matr[i].size(); j++){
				j_ = i+j+1;
				all[i][j_] = atof(matr[i][j].c_str());
				all[j_][i] = all[i][j_];
				temp1 = (all[i][j_]-MIN)*MULTIPLIER;
				if(temp1 < 0.0){
					temp1 = 0.0;
				}
				if(temp1 > 1.0){
					temp1 = 1.0;
				}
				if(DIRECTION == 0){
					temp.push_back(temp1);
				} else {
					temp.push_back(1.0-temp1);
				}
			}
			scores.push_back(temp);
		}
	}
	
	ofstream outfile;
	ofstream outfile_txt;
  outfile.open(fileout.c_str());
	outfile_txt.open(fileout_txt.c_str());
  if(outfile.is_open()){
		if(outfile_txt.is_open()){
			outfile << "<!--ProSMART Multiple Scores File-->" << endl;
			if(MAX >= 0){	//use colours - create function to load stylesheet
				outfile << "<head><script language=\"JavaScript\">" << endl
				<< "function setActiveStyleSheet(title) {var i, a, main;" << endl
				<< "for(i=0; (a = document.getElementsByTagName(\"link\")[i]); i++) {if(a.getAttribute(\"rel\").indexOf(\"style\") != -1 && a.getAttribute(\"title\")) {a.disabled = true; if(a.getAttribute(\"title\") == title) a.disabled = false;}}}"
				<< "</script></head>" << endl;
			}
			
			outfile << "<html><style type='text/css'>" << endl
			<< "tr{font-size:70%;}" << endl 
			<< "a{font-family:geneva, 'trebuchet MS', garuda, helvetica, arial, sans-serif, sans, verdana; font-size:70%;}" << endl
			<< "</style>" << endl
			<< "<LINK rel=\"alternate stylesheet\" href=\".black.css\" type=\"text/css\" title=\"black\">" << endl
			<< "<body><table border='0'>" << endl;
			outfile << "<tr><td></td>";
			for(unsigned int i=0; i<header.size(); i++){
				outfile << "<td>" << header[i] << "</td>";
			}
			outfile << "</tr>";
			for(unsigned int i=0; i<matr.size(); i++){
				outfile << endl << "<tr><td>" << header[i] << "</td>";
				for(unsigned int j=0; j<=i; j++){
					outfile << "<td></td>";
				}			
				for(unsigned int j=0; j<matr[i].size(); j++){
					if(MAX < 0){	//dont use colours
						outfile << "<td>" << matr[i][j] << "</td>";
					} else {
						outfile << "<td style='color:rgb(" 
						<< (int)(prosmart_green.x*scores[i][j] + prosmart_red.x*(1-scores[i][j])) << ","
						<< (int)(prosmart_green.y*scores[i][j] + prosmart_red.y*(1-scores[i][j])) << "," 
						<< (int)(prosmart_green.z*scores[i][j] + prosmart_red.z*(1-scores[i][j])) << ")'>" << matr[i][j] << "</td>";
					}
				}
				outfile << "</tr>";
			}
			outfile << endl << "<tr><td>" << header[matr.size()] << "</td>";
			for(unsigned int j=0; j<=matr.size(); j++){
				outfile << "<td></td>";
			}
			outfile << endl << "</table>" << endl;
			
			if(MAX >= 0){	//use colours - allow colour changing
				outfile << endl << "<br /><p>" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('black'); return false;\">black</a><br />" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('default'); return false;\">colour</a>" << endl
				<< "</p>" << endl;
			}
			
			outfile << endl << "</body></html>" << endl;			//cout << endl << "File written to: " << fileout;
			
			//cout << endl << "File written to: " << fileout;
			
            outfile_txt << "# ProSMART Multiple Scores File" << endl;
            //outfile_txt << "-";
            for(unsigned int i=0; i<header.size(); i++){
				outfile_txt << "\t" << header[i];
			}
            outfile_txt << endl;
			for(unsigned int i=0; i<N; i++){
                outfile_txt << header[i];
				for(unsigned int j=0; j<N; j++){
					outfile_txt << "\t" << all[i][j];
				}
				outfile_txt << endl;
			}
			
			outfile_txt.close();
		} else {
			xml_entry(error_file,0,5,fileout_txt);
			cout << "Unable to open output file " << fileout_txt << " for writing" << endl;
		}
		outfile.close();
  } else {
    xml_entry(error_file,0,5,fileout);
    cout << "Unable to open output file " << fileout << " for writing" << endl;
  }
	
  return;
}

void write_matrix_all_on_all(string &error_file, string fileout_, vector<vector<string> > &matr, vector<string> &header, double MIN, double MAX, bool DIRECTION, double val)
//DIRECTION == 0 -> low scores are bad.
//DIRECTION == 1 -> low scores are good.
//MAX < 0 -> dont use colours
{
	double MULTIPLIER = 0.0;
	if(MAX > MIN){
		MULTIPLIER = 1.0/(MAX-MIN);
	}
	string fileout = fileout_ + ".html";
	string fileout_txt = fileout_ + ".txt";
	
	unsigned int N = header.size();
	vector<vector<double> > scores;
	vector<double> temp;
	double temp1;
	
	vector<vector<double> > all;
	vector<double> temp2;
	for(unsigned int i=0; i<N; i++){
		temp2.push_back(val);
	}
	for(unsigned int i=0; i<N; i++){
		all.push_back(temp2);
	}
	
	/*cout << endl;
	 cout << fileout << endl;
	 for(unsigned int i=0; i<matr.size(); i++){
	 for(unsigned int j=0; j<matr[i].size(); j++){
	 cout << matr[i][j] << "\t";
	 }
	 cout << endl;
	 }
	 cout << endl;*/
	
	if(MAX < 0){	//no colours are to be used
		for(unsigned int i=0; i<matr.size(); i++){
			for(unsigned int j=0; j<matr[i].size(); j++){
				if(i!=j){
					all[i][j] = atof(matr[i][j].c_str());
				}
			}
		}
	} else {
		for(unsigned int i=0; i<N; i++){
			temp.clear();
			for(unsigned int j=0; j<N; j++){
				if(i==j){
					temp.push_back(0);
					continue;
				}
				all[i][j] = atof(matr[i][j].c_str());
				temp1 = (all[i][j]-MIN)*MULTIPLIER;
				if(temp1 < 0.0){
					temp1 = 0.0;
				}
				if(temp1 > 1.0){
					temp1 = 1.0;
				}
				if(DIRECTION == 0){
					temp.push_back(temp1);
				} else {
					temp.push_back(1.0-temp1);
				}
			}
			scores.push_back(temp);
		}
	}
	
	ofstream outfile;
	ofstream outfile_txt;
  outfile.open(fileout.c_str());
	outfile_txt.open(fileout_txt.c_str());
  if(outfile.is_open()){
		if(outfile_txt.is_open()){
			outfile << "<!--ProSMART Multiple Scores File-->" << endl;
			if(MAX >= 0){	//use colours - create function to load stylesheet
				outfile << "<head><script language=\"JavaScript\">" << endl
				<< "function setActiveStyleSheet(title) {var i, a, main;" << endl
				<< "for(i=0; (a = document.getElementsByTagName(\"link\")[i]); i++) {if(a.getAttribute(\"rel\").indexOf(\"style\") != -1 && a.getAttribute(\"title\")) {a.disabled = true; if(a.getAttribute(\"title\") == title) a.disabled = false;}}}"
				<< "</script></head>" << endl;
			}
			
			outfile << "<html><style type='text/css'>" << endl
			<< "tr{font-size:70%;}" << endl 
			<< "a{font-family:geneva, 'trebuchet MS', garuda, helvetica, arial, sans-serif, sans, verdana; font-size:70%;}" << endl
			<< "</style>" << endl
			<< "<LINK rel=\"alternate stylesheet\" href=\".black.css\" type=\"text/css\" title=\"black\">" << endl
			<< "<body><table border='0'>" << endl;
			outfile << "<tr><td></td>";
			for(unsigned int i=0; i<header.size(); i++){
				outfile << "<td>" << header[i] << "</td>";
			}
			outfile << "</tr>";
			for(unsigned int i=0; i<matr.size(); i++){
				outfile << endl << "<tr><td>" << header[i] << "</td>";
				for(unsigned int j=0; j<matr[i].size(); j++){
					if(i==j){
						outfile << "<td></td>";
						continue;
					}
					if(MAX < 0){	//dont use colours
						outfile << "<td>" << matr[i][j] << "</td>";
					} else {
						outfile << "<td style='color:rgb(" 
						<< (int)(prosmart_green.x*scores[i][j] + prosmart_red.x*(1-scores[i][j])) << ","
						<< (int)(prosmart_green.y*scores[i][j] + prosmart_red.y*(1-scores[i][j])) << "," 
						<< (int)(prosmart_green.z*scores[i][j] + prosmart_red.z*(1-scores[i][j])) << ")'>" << matr[i][j] << "</td>";
					}
				}
				outfile << "</tr>";
			}
			outfile << endl << "</table>" << endl;
			
			if(MAX >= 0){	//use colours - allow colour changing
				outfile << endl << "<br /><p>" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('black'); return false;\">black</a><br />" << endl
				<< "<a href=\"#\" onclick=\"setActiveStyleSheet('default'); return false;\">colour</a>" << endl
				<< "</p>" << endl;
			}
			
			outfile << endl << "</body></html>" << endl;			//cout << endl << "File written to: " << fileout;
			
			//cout << endl << "File written to: " << fileout;
			
            outfile_txt << "# ProSMART Multiple Scores File" << endl;
            //outfile_txt << "-";
            for(unsigned int i=0; i<header.size(); i++){
				outfile_txt << "\t" << header[i];
			}
            outfile_txt << endl;
			for(unsigned int i=0; i<N; i++){
                outfile_txt << header[i];
				for(unsigned int j=0; j<N; j++){
					outfile_txt << "\t" << all[i][j];
				}
				outfile_txt << endl;
			}
			
			outfile_txt.close();
		} else {
			xml_entry(error_file,0,5,fileout_txt);
			cout << "Unable to open output file " << fileout_txt << " for writing" << endl;
		}
		outfile.close();
  } else {
    xml_entry(error_file,0,5,fileout);
    cout << "Unable to open output file " << fileout << " for writing" << endl;
  }
	
  return;
}

void write_residue_type(string &error_file, string &fileout, vector<res_alignment> &align_types)
{
	ofstream outfile;
  outfile.open(fileout.c_str());
  if(outfile.is_open()){
		outfile << "# ProSMART Fragment Types File" << endl;
		outfile << "# Res\tType" << endl;
		for(unsigned int i=0; i<align_types.size(); i++){
			outfile << align_types[i].res1 << "\t" << align_types[i].res2 << endl;
		}
		outfile.close();
    //cout << endl << "File written to: " << fileout;
  } else {
    xml_entry(error_file,0,5,fileout);
    cout << "Unable to open output file " << fileout << " for writing" << endl;
  }	
	
  return;	
}

vector<string> write_restraints(vector<string> &filenames1, vector<string> &filenames2, vector<string> &chains1, vector<string> &chains2, vector<vector<int> > &restrain_chains, string &error_file, string HELIX, string &RES_dir, string &workingdirectory, bool TROUBLESHOOT_RESTRAINT_FILES, vector<string> &multi_filenames, bool COPY_FINAL_RESTRAINTS)
{
	string temp_multi_filename;
	vector<vector<int> > idx;
	vector<int> tmp_v;
	vector<vector<string> > file_list;
	vector<string> tmp_vs;
	vector<string> final_restrain_files;
	bool already_used;
	
	for(unsigned int i=0; i<chains1.size(); i++){	//create final pdb-based restraint files
		//get indexes of chains that go in different final restraints files
		already_used = 0;
		for(unsigned int j=0; j<i; j++){
			if(filenames1[i] == filenames1[j]){
				already_used = 1;
				break;
			}
		}
		if(already_used==0){
			tmp_v.clear();
			for(unsigned int j=i; j<chains1.size(); j++){
				if(filenames1[i] == filenames1[j]){
					tmp_v.push_back(j);
				}
			}
			idx.push_back(tmp_v);
		}
	}
	for(unsigned int i=0; i<idx.size(); i++){
		tmp_vs.clear();
		final_restrain_files.push_back(get_filename(filenames1[idx[i][0]]));
		for(unsigned int j=0; j<idx[i].size(); j++){
			for(unsigned int k=0; k<restrain_chains[idx[i][j]].size(); k++){
				tmp_vs.push_back(final_restrain_files[i] + "_" + chains1[idx[i][j]] + "_" + get_filename(filenames2[restrain_chains[idx[i][j]][k]]) + "_" + chains2[restrain_chains[idx[i][j]][k]]);
			}
		}
		file_list.push_back(tmp_vs);
	}	
	
	for(unsigned int i=0; i<file_list.size(); i++){
		temp_multi_filename = workingdirectory + "Restraints/";
		if(HELIX == "1"){
			temp_multi_filename += "LIB_";
		}
		temp_multi_filename += final_restrain_files[i];
		multi_filenames.push_back(temp_multi_filename);
	}
    
	write_restraints_generic(file_list, multi_filenames, error_file, RES_dir, workingdirectory, 1, TROUBLESHOOT_RESTRAINT_FILES);

    if(COPY_FINAL_RESTRAINTS){
        //output final restraints files to output directory
        vector<string> copied_multi_filenames;
        for(unsigned int i=0; i<multi_filenames.size(); i++){
            if(file_list[i].size()==0){
                file_list.erase(file_list.begin()+i);
                multi_filenames.erase(multi_filenames.begin()+i);
                i--;
            } else {
                multi_filenames[i] += ".txt";
                temp_multi_filename = "\"" + multi_filenames[i] + "\" \"" + workingdirectory + "../" + get_full_filename(multi_filenames[i]) + "\"";
#ifdef WINDOWS
                system(("copy /Y "+replace_chars(temp_multi_filename,"/","\\")+" > nul").c_str());
#else
                system(("cp " + temp_multi_filename).c_str());
#endif           
            }
        }
    }

	return final_restrain_files;
}

vector<vector<string> > write_restraints_chain2(vector<string> &filenames1, vector<string> &filenames2, vector<string> &chains1, vector<string> &chains2, vector<vector<int> > &restrain_chains, string &error_file, string HELIX, string &RES_dir, string &workingdirectory, bool TROUBLESHOOT_RESTRAINT_FILES)
{
	string temp_multi_filename;
	vector<string> multi_filenames;
	vector<vector<int> > idx;
	vector<int> tmp_v;
	vector<vector<string> > file_list;
	vector<string> file_list_chain2;
	vector<string> tmp_vs;
	vector<string> final_restrain_files;
	bool already_used;
	string tmp_s;
	vector<string> result;
	vector<string> result1;
	vector<vector<string> > result2;
	
	for(unsigned int i=0; i<chains1.size(); i++){	//create final pdb-based restraint files
		//get indexes of chains that go in different final restraints files
		already_used = 0;
		for(unsigned int j=0; j<i; j++){
			if(filenames1[i] == filenames1[j]){
				already_used = 1;
				break;
			}
		}
		if(already_used==0){
			tmp_v.clear();
			for(unsigned int j=i; j<chains1.size(); j++){
				if(filenames1[i] == filenames1[j]){
					tmp_v.push_back(j);
				}
			}
			idx.push_back(tmp_v);
		}
	}
	for(unsigned int l=0; l<chains2.size(); l++){
		tmp_s = get_filename(filenames2[l]) + "_" + chains2[l];
		for(unsigned int i=0; i<idx.size(); i++){
			tmp_vs.clear();
			final_restrain_files.push_back(get_filename(filenames1[idx[i][0]]));
			for(unsigned int j=0; j<idx[i].size(); j++){
				for(unsigned int k=0; k<restrain_chains[idx[i][j]].size(); k++){
					if(restrain_chains[idx[i][j]][k]!=(int)l){
						continue;
					}
					tmp_vs.push_back(final_restrain_files[i] + "_" + chains1[idx[i][j]] + "_" + tmp_s);
				}
			}
			file_list.push_back(tmp_vs);
			file_list_chain2.push_back(tmp_s);
		}	
	}
	
	for(unsigned int i=0; i<file_list.size(); i++){
		if(HELIX == "1"){
			tmp_s = "LIB_";
			temp_multi_filename += "LIB_";
		} else {
			tmp_s = "";
		}
		tmp_s += final_restrain_files[i] + "_" + file_list_chain2[i];
		temp_multi_filename = workingdirectory + "Restraints/PDB_Chain/" + tmp_s;
		multi_filenames.push_back(temp_multi_filename);
		result.push_back(tmp_s);
		result1.push_back(final_restrain_files[i] + " &nbsp; " + file_list_chain2[i]);
	}
	
	write_restraints_generic(file_list, multi_filenames, error_file, RES_dir, workingdirectory, 0, TROUBLESHOOT_RESTRAINT_FILES);
	
	result2.push_back(result);
	result2.push_back(result1);
	return result2;
}

void write_restraints_generic(vector<vector<string> > &file_list, vector<string> multi_filenames, string &error_file, string &RES_dir, string &workingdirectory, bool WRITE_XML, bool TROUBLESHOOT_RESTRAINT_FILES)
{
	string fileout = workingdirectory + "restrain_script";
	string firstline = "# ProSMART Restraints File";
	int return_status;
	ofstream outfile;
    string temp;
    bool file_exists;
	
	//create header for final restraints file(s)
	for(unsigned int i=0; i<file_list.size(); i++){
        if(file_list[i].size()==0)continue;
        temp = multi_filenames[i];
#ifdef WINDOWS
        temp += ".txt";    
#endif
		outfile.open(temp.c_str());
		if(!outfile.is_open()){
			xml_entry(error_file,0,5,temp);
			cout << "Unable to open output file " << temp << " for writing. Program terminated." << endl;
			exit(-1);
		}
		outfile << firstline << endl
		<< "#" << endl
		<< "# Comprises Chain-Chain Restraints Files: " << endl;
		for(unsigned int j=0; j<file_list[i].size(); j++){
			outfile << "#   " << file_list[i][j] << endl;
		}
		outfile << "#" << endl << "#" 
		<< endl << "# --------------------------------------------------" 
		<< endl << "#" << endl << "#" << endl;
		outfile.close();	
	}
    
    if(TROUBLESHOOT_RESTRAINT_FILES){
        cout << endl << endl << "Performing extra validation during creation of restraints files:";
        for(unsigned int i=0; i<file_list.size(); i++){
            cout << endl << "  " << multi_filenames[i] << ".txt";
        }
        cout << endl << endl;
        
        for(unsigned int i=0; i<file_list.size(); i++){
            file_exists = check_file_exists(multi_filenames[i]);
            if(file_exists){
                cout << "Required temporary file does exist: " << multi_filenames[i] << endl;
            } else {
                cout << endl << "Required temporary file does not exist: " << multi_filenames[i];
                cout << endl << "Error! Program terminated." << endl << endl;
                exit(-1);
            }
        }
    }
	
    //concatenate restraint files
    for(unsigned int i=0; i<file_list.size(); i++){
        if(file_list[i].size()==0)continue;
        if(TROUBLESHOOT_RESTRAINT_FILES){
            cout << endl << "concatenating files: ";
            for(unsigned int j=0; j<file_list[i].size(); j++){
                cout << endl << "  " << RES_dir << file_list[i][j] << ".txt";
            }
            cout << endl << "into file: " << multi_filenames[i] << ".txt" << endl;
        }
#ifdef WINDOWS
        temp = "copy \"" + multi_filenames[i] + ".txt\"";
        for(unsigned int j=0; j<file_list[i].size(); j++)
			temp += "+\"" + RES_dir + file_list[i][j] + ".txt\"";
        temp += " \"" + multi_filenames[i] + ".txt\" > NUL";
        system(replace_chars(temp,"/","\\").c_str());
#else
        temp = "cat \"" + multi_filenames[i] + "\"";
        for(unsigned int j=0; j<file_list[i].size(); j++)
			temp += " \"" + RES_dir + file_list[i][j] + ".txt\"";
        temp += " > \"" + multi_filenames[i] + ".txt\"";
        system(temp.c_str());                       //concatenate restraint files
        unlink(multi_filenames[i].c_str());         //remove header
#endif
    }
    
	//make sure final restraints files were created.
	for(unsigned int i=0; i<multi_filenames.size(); i++){
        if(file_list[i].size()==0)continue;
		multi_filenames[i] += ".txt";
		return_status = validate_file(multi_filenames[i],firstline);
		if(return_status == 1){
			xml_entry(error_file,0,9,multi_filenames[i]);
			cout << "Error - restraints file: " << multi_filenames[i] << " was not created successfully. Program terminated." << endl;
			exit(-1);
		} else if(WRITE_XML==1){
			xml_entry(error_file,4,0,multi_filenames[i]);
		}
	}
    
    if(TROUBLESHOOT_RESTRAINT_FILES){
        for(unsigned int i=0; i<file_list.size(); i++){
            if(file_list[i].size()==0)continue;
            file_exists = check_file_exists(multi_filenames[i]);
            if(file_exists){
                cout << endl << "Final created restraints file does exist: " << multi_filenames[i] << endl;
            } else {
                cout << endl << "Final created restraints file does not exist: " << multi_filenames[i];
                cout << endl << "Error! Program terminated." << endl << endl;
                exit(-1);
            }        
        }
    }
	
	return;
}

void output_stylesheet(string &error_file, string &dirout)
{
	ofstream outfile;
	string fileout = dirout + ".black.css";
	outfile.open(fileout.c_str());
	if(!outfile.is_open()){
		xml_entry(error_file,0,5,fileout);
		cout << "Unable to open output file " << fileout << " for writing. Program terminated." << endl;
		exit(-1);
	}
	outfile << "td{color:black ! important;}" << endl;
	outfile.close();
	return;
}

void analyse_pairwise_results(Array2D<vector<vector<string> > > &align_m, vector<string> &files, vector<string> &chains)
{
    //generate equivalent results, with flipped res numbers. Remove invalid lines.
    for(int i=1; i<align_m.dim1(); i++){
        for(int j=0; j<i; j++){
            align_m[i][j] = align_m[j][i];
            for(unsigned int k=0; k<align_m[i][j].size(); k++){
                if(align_m[i][j][k].size()==4){
                    align_m[i][j][k][0] = align_m[j][i][k][1];
                    align_m[i][j][k][1] = align_m[j][i][k][0];
                } else {
                    align_m[i][j].erase(align_m[i][j].begin()+k);
                    align_m[j][i].erase(align_m[j][i].begin()+k);
                    k--;
                }
            }
        }
    }

    /*for(int i=0; i<align_m.dim1(); i++){
        for(int j=0; j<align_m.dim2(); j++){
            cout << endl << endl << i << " " << j << endl;
            if(align_m[i][j].size()>0){
            cout << align_m[i][j][0];
            }
        }
    }*/
    
    /////////////
    
    //convert scores to double, and get global stats.
    Array2D<vector<double> > scores(align_m.dim1(),align_m.dim2());
    Array2D<vector<string> > atomid(align_m.dim1(),align_m.dim2());
    Array2D<unsigned int> n_tot(align_m.dim1(),align_m.dim2());
    Array2D<unsigned int> n_le1(align_m.dim1(),align_m.dim2());
    Array2D<double> p_le1(align_m.dim1(),align_m.dim2());
    for(int i=0; i<align_m.dim1(); i++){
        for(int j=0; j<align_m.dim2(); j++){
            n_tot[i][j] = align_m[i][j].size();
            n_le1[i][j] = 0;
            for(unsigned int k=0; k<n_tot[i][j]; k++){
                scores[i][j].push_back(str_to_double(align_m[i][j][k][3]));
                atomid[i][j].push_back(align_m[i][j][k][0]+"_"+align_m[i][j][k][2]);
                if(scores[i][j][k]<=1.0){
                    n_le1[i][j]++;
                }
            }
            if(n_tot[i][j]==0){
                p_le1[i][j] = 1.0;
            } else {
                p_le1[i][j] = (double)n_le1[i][j]/n_tot[i][j];
            }
        }
    }
    
    /////////////
    
    //get max number of aligned atoms - single number used for all structures, ensuring that relative numbers are on the same scale.
    unsigned int max_aligned_atoms = 0;
    for(int i=0; i<scores.dim1(); i++){
        for(int j=0; j<scores.dim2(); j++){
            if(scores[i][j].size()>max_aligned_atoms){
                max_aligned_atoms = scores[i][j].size();
            }
        }
    }

    //filter each structure-pair so that only core is left.
    for(int i=0; i<scores.dim1(); i++){
        for(int j=0; j<scores.dim2(); j++){
            for(unsigned int k=0; k<scores[i][j].size(); k++){
                if(scores[i][j][k]>1.0){
                    scores[i][j].erase(scores[i][j].begin()+k);
                    atomid[i][j].erase(atomid[i][j].begin()+k);
                    k--;
                }
            }
        }
    }
    
    //for each structure, iteratively filter core by considering next-most similar structure. Always done relative to the target structure, not all-on-all.
    vector<vector<unsigned int> > core_member;
    vector<vector<unsigned int> > core_size;
    vector<unsigned int> tmp_vui;    
    for(unsigned int i=0; i<files.size(); i++){
        core_member.push_back(tmp_vui);
        core_size.push_back(tmp_vui);
        core_member[i].push_back(i);
        core_size[i].push_back(max_aligned_atoms);
        
        vector<bool> clustered(files.size(),0);
        clustered[i] = 1;
        filter_core(atomid,scores,i,clustered,core_size[i],core_member[i]);
    }
    
    /////////////
    
    vector<string> file_id;
    for(unsigned int i=0; i<files.size(); i++){
        file_id.push_back(get_filename(files[i]).substr(0,4)+chains[i]);
    }
    
    string file = "final_pairwise_prop_less_eq1.txt";
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(int i=0; i<align_m.dim1(); i++){
            outfile << "\t" << file_id[i];
        }
        outfile << endl;
        outfile.precision(2);
        for(int i=0; i<align_m.dim1(); i++){
            outfile << file_id[i];
            for(int j=0; j<align_m.dim2(); j++){
                outfile << "\t" << p_le1[i][j];
            }
            outfile << endl;
        }
        outfile.close();
    }
    
    file = "final_pairwise_core_iterative_n.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << file_id[0];
        for(int i=1; i<scores.dim1(); i++){
            outfile << "\t" << file_id[i];
        }
        outfile << endl;
        for(int i=0; i<scores.dim1(); i++){
            outfile << core_size[0][i];
            for(int j=1; j<scores.dim2(); j++){
                outfile << "\t" << core_size[j][i];
            }
            outfile << endl;
        }
        outfile.close();
    }
    
    file = "final_pairwise_core_iterative_names.txt";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        for(int i=0; i<scores.dim1(); i++){
            outfile << file_id[core_member[0][i]];
            for(int j=1; j<scores.dim2(); j++){
                outfile << "\t" << file_id[core_member[j][i]];
            }
            outfile << endl;
        }
        outfile.close();
    }
    
    return;
}

void filter_core(Array2D<vector<string> > &atomid, Array2D<vector<double> > &scores, unsigned int &idx, vector<bool> &clustered, vector<unsigned int> &core_size, vector<unsigned int> &core_member)
{
    bool valid = 0;
    for(unsigned int i=0; i<clustered.size(); i++){
        if(clustered[i]==0){
            valid = 1;
            break;
        }
    }
    if(!valid){
        return;
    }
    
    //filter all scores[idx][j] to agree with scores[idx][idx] if after first iteration
    if(scores[idx][idx].size()>0){
        for(int j=0; j<atomid.dim2(); j++){
            if((int)idx!=j){
                unsigned int tmpidx = 0;
                for(unsigned int k=0; k<atomid[idx][j].size(); k++){
                    bool found = 0;
                    for(unsigned int l=tmpidx; l<atomid[idx][idx].size(); l++){
                        if(atomid[idx][j][k]==atomid[idx][idx][l]){
                            found = 1;
                            tmpidx = l;
                            break;
                        }
                    }
                    if(!found){
                        scores[idx][j].erase(scores[idx][j].begin()+k);
                        atomid[idx][j].erase(atomid[idx][j].begin()+k);
                        k--;
                    }
                }
            }
        }
    }
    
    //find structure with most similar core to current core
    unsigned int jdx = 0;
    unsigned int current_size = 0;
    for(int j=0; j<scores.dim2(); j++){
        if(clustered[j]==0){
            if(scores[idx][j].size()>current_size){
                jdx = j;
                current_size = scores[idx][j].size();
            }
        }
    }
    
    //store current state of core in scores[idx][idx]
    scores[idx][idx] = scores[idx][jdx];
    atomid[idx][idx] = atomid[idx][jdx];
    
    clustered[jdx] = 1;
    core_member.push_back(jdx);
    core_size.push_back(scores[idx][idx].size());
    
    filter_core(atomid,scores,idx,clustered,core_size,core_member);
    return;
}



