//
// This file is part of ProSMART.
//

#include "prosmart_params.h"

void set_params(params &A)
{
	//Program Options
	A.ALIGN = 0;
	A.RESTRAIN = 0;
	A.CLUSTER = "1";
	A.ALL_ON_ALL = 0;
	A.TRIANGULAR = 1;
	A.MAX_THREADS = 4;
	A.NMR = 0;
	A.NMR_TARGET = 1;
	A.MAX_FRAG_LEN = 17;
	A.error_file = "";
    A.output_dir = "ProSMART_Output"SLASH;
	A.filenames1.clear();
	A.filenames2.clear();
	A.chains1.clear();
	A.chains2.clear();
	A.protein_or_dnarna1.clear();
	A.protein_or_dnarna2.clear();
	A.new_chain = " ";
	A.BIN_DIR = "";
	A.LIB_DIR = "";
	A.align_location = "";
	A.restrain_location = "";
	A.lib_fraglen.clear();
	A.lib_score.clear();
	A.config.clear();
	A.library_config = "";
	A.OVERRIDE_LIB_SCORE = "";
	A.OVERRIDE_LIB_FRAGLEN = "";
    A.INPUT_DIR = "";
    A.WORKING_DIRECTORY = "";
    A.ALIGN_dir = "";
    A.TRANS_dir = "";
    A.PDB_dir = "";
    A.PYMOL_dir = "";
    A.INPUTCHAINS_dir = "";
    A.MERGE_CHAINS = 0;
   A.OCCUPANCY_THRESHOLD = 0.0;
    
	//Colours:
	A.G1 = "1.0";
	A.G2 = "1.0";
	A.G3 = "0.0";
	A.R1 = "0.8";
	A.R2 = "0.0";
	A.R3 = "0.0";
	
	//Align Parameters
	A.FRAG_LEN = "9";
    A.SCORE_THRESHOLD = "1000.0";
    A.ALIGN_SCORE = "1.0";
    A.SIDE_SCORE = "1.0";
    A.ALIGN_MODE = "0";						//default mode = transform/score using all mainchain atoms.
    A.EXTRA_REFINEMENT = "1";
    A.HELIX = "0";
	A.HELIX_ONLY = "0";
	A.STRAND_ONLY = "0";
    A.IDENTICAL = "0";
	A.OUT_PDB = "1";
    A.OUT_PDB_FULL = "0";
	A.OUT_COLOR = "1";
	A.OUTPUT_SCREW = "1";
    A.HELIX_CUTOFF = "1.0";
    A.HELIX_PENALTY = "0.2";
	A.CLUSTER_SCORE = "1.0";
	A.CLUSTER_DEGREES = "15";
	A.CLUSTER_MIN = "10";
	A.CLUSTER_LINKAGE = "0.999";			// ~ 3.6 degrees
   A.CLUSTER_TRANS_LINKAGE = "0.3";
   A.CLUSTER_TRANS_MAX = "10.0";   // warning - default is different if -dna_rna is specified.
	A.CLUSTER_RIGIDITY = "0.999";
	A.CLUSTER_COLOR = "1.0";
	A.SUPERPOSE = "1.0";
	A.SUPERPOSE_SIDE = "0";
	A.OUTPUT_CLUSTER_DM = "0";
   A.OUTPUT_RMSD_FILES = "0.0";
	A.PDB1A.clear();
	A.PDB1B.clear();
	A.CHAIN1A.clear();
	A.CHAIN1B.clear();
	A.RANGE1A.clear();
	A.RANGE1B.clear();
    A.ALIGN_PDB.clear();
    A.ALIGN_CHAIN.clear();
    A.ALIGN_RANGE1.clear();
    A.ALIGN_RANGE2.clear();
    A.ALIGN_RM_PDB.clear();
    A.ALIGN_RM_CHAIN.clear();
    A.ALIGN_RM_RESIDUE.clear();
    A.DISPLAY_AS_COSINE_DISTANCE = "0";
    A.USE_MAINCHAIN_FOR_SIDERMSD = "0";
    A.NUM_DIST_THRESHOLD = "1.0";
    A.FIX_SIDE_CHAIN_ERRORS = "1";
    A.NO_REWARD_SEQ = "0";
    A.REWARD_SEQ = "-2.0";
    A.SCORE_STRICT = "0";

    //Sequence Alignment Parameters
    A.SEQUENCE_ALIGNMENT = 0;
    A.SEQ_MIN = 20;
    A.SEQ_RATIO = 2.0;
    A.SEQ_ID = 0.6;
    A.SEQ_ID_MAX = 1.0;
	
	//Restrain Parameters
    A.GENERIC = 0;
	A.RESTRAIN_OPTION = 0;
	A.RESTRAIN_TO_SELF = 0;
   A.RESTRAIN_SEQID = 75.0;
   A.RESTRAIN_SCORE = 10.0;
	A.sphere = "6.0";
	A.sigma = "0.1";							//default sigma
	A.MIN_DISTANCE = "2.5";					//removes restraints with distances below this value.
	A.score_cutoff = "10.0";
	A.sigma_method = "0";
	A.RESTRAIN_SIDE_CUTOFF = "10.0";
	A.RESTRAIN_MULTIPLIER_CUTOFF = "1000.0";
	A.SIGMA_WEIGHT = "1.0";
	A.MINIMUM_SIGMA = "0.01";
	A.GET_BONDS = "0";
   A.GET_ATOMIC_UNCERTAINTY = "0";
	A.REFMAC_NAME = "refmac5";
	A.BFAC1 = "0";
	A.BFAC2 = "0";
	A.MAINCHAIN_ONLY = "0";
    A.RESTRAIN_PDB.clear();
    A.RESTRAIN_CHAIN.clear();
    A.RESTRAIN_RANGE1.clear();
    A.RESTRAIN_RANGE2.clear();
    A.RESTRAIN_RM_PDB.clear();
    A.RESTRAIN_RM_CHAIN.clear();
    A.RESTRAIN_RM_RESIDUE.clear();
    A.OUTPUT_PDB_CHAIN_RESTRAINTS = 0;
    A.COPY_FINAL_RESTRAINTS_FILES = 1;
    A.TYPE = "-1";
    
    //Generic bond restraints (e.g. for hydrogen bonds)
    A.HBOND_RESTRAINTS = 0;
    A.HBOND_DIST = 2.8;
    A.HBOND_DIST_MAX = 3.5;
    A.HBOND_DIST_MIN = 2;
    A.MIN_SEP = 3;
    A.MAX_SEP = -1;
    A.ALLOW_SEP.clear();
    A.RM_SEP.clear();
    A.HBOND_OPTIONS = 2;
    A.OVERRIDE_MAX_BONDS = 0;
    
    //Internal bond options
    A.H_HELIX = 0;
    A.H_HELIX_3 = 0;
    A.H_HELIX_4 = 0;
    A.H_HELIX_5 = 0;
    A.H_SHEET = 0;
    A.FORCE_OPTIONS = 0;
    A.FORCE_MIN_SEP = 0;
    A.H_FILENAMES.clear();
    A.STRICT_FRAG = 0;
    A.FORCE_ALLOW_SEP = 0;
    
    //Troubleshooting Flags
    A.TROUBLESHOOT_RESTRAINT_FILES = 0;
    A.TROUBLESHOOT_HBOND_RESTRAINTS = 0;
    
    return;	
}

void print_header()
{
	cout << endl << "#################################################################"
	<< endl << "#################################################################"
	<< endl << "############ ProSMART version 0.845 ##### 6 Oct 2014 ############"
	<< endl << "#################################################################"
    << endl << "##"
    << endl << "## Copyright (C) 2011-2014 Robert Nicholls and Garib Murshudov"
    << endl << "##"
    << endl << "## ProSMART is distributed in the hope that it will be useful,"
    << endl << "## but WITHOUT ANY WARRANTY; without even the implied warranty" 
    << endl << "## of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE." 
    << endl << "## See the GNU Lesser General Public License (v3) for details."
    << endl << "## The authors accept no liability whatsoever for any damage"
    << endl << "## caused by the use or misuse of this software."
    << endl << "##"
   << endl << "## REFERENCE 1 - PRIMARY CITATION, STRUCTURAL COMPARISON:"
   << endl << "## Conformation-Independent Structural Comparison of"
   << endl << "## Macromolecules with ProSMART"
   << endl << "## RA Nicholls, M Fischer, S McNicholas and GN Murshudov (2014)"
   << endl << "## Acta Crystallographica D70, 2487-2499"
   << endl << "##"
   << endl << "## REFERENCE 2 - LOW-RESOLUTION REFINEMENT:"
   << endl << "## Recent Advances in Low Resolution Refinement Tools in REFMAC5"
   << endl << "## RA Nicholls, F Long and GN Murshudov (2013)"
   << endl << "## Advancing Methods for Biomolecular Crystallography, 231-258"
   << endl << "##"
    << endl << "## REFERENCE 3 - LOW-RESOLUTION REFINEMENT:"
    << endl << "## Low Resolution Refinement Tools in REFMAC5"
    << endl << "## RA Nicholls, F Long and GN Murshudov (2012)"
    << endl << "## Acta Crystallographica D68, 404-417"
    << endl << "##"
    << endl << "## REFERENCE 4 - DETAILED DESCRIPTION:"
    << endl << "## Conformation-Independent Comparison of Protein Structures"
    << endl << "## RA Nicholls (2011)"
    << endl << "## University of York (thesis) (etheses.whiterose.ac.uk/2120/)"
    << endl << "##"
    << endl << "## More info: www2.mrc-lmb.cam.ac.uk/groups/murshudov/"
    << endl << "##"
    << endl << "#################################################################"
	<< endl << endl;
    return;
}

vector<string> get_args(char *argv[], int argc, params &PARAMS)
{
	vector<string> args;
	
	bool HELP = 0;
    if(argc < 2){
        HELP = 1;
    } else if(argc == 2){
		args.push_back(argv[1]);
        if(args[0] == "help" || args[0] == "-help" || args[0] == "--help"){
			HELP = 1;
		} 
    }
	args.clear();
    if(HELP == 1){
        cout << endl << "ProSMART arguments:" << endl << endl
        << "Program Options:" << endl
        << "-a\t\t\t" << "only ProSMART ALIGN, use main chain atoms (default)." << endl
        << "-a1\t\t\t" << "only ProSMART ALIGN, use CA atoms (slightly faster)." << endl
        << "-a2\t\t\t" << "only ProSMART ALIGN, superpose main, score CA (slower)." << endl
        << "-r\t\t\t" << "only ProSMART RESTRAIN (assumes alignment exists)." << endl
        << "-o [dir path]\t\t" <<	"output directory (default 'ProSMART_Output/')." << endl
        << "-input_dir [dir path]\t" <<	"input directory." << endl
        << "-f [file path]\t\t" << "external file providing arguments to the program." << endl
#ifndef WINDOWS
        << "-threads [int]\t\t" << "max number of simultaneous threads (default '" << PARAMS.MAX_THREADS << "')." << endl
#endif
        << "-xml\t\t\t" << "output XML file (file path: './xmlout.xml')." << endl
        << "-xml [file path]\t" << "output XML file (specify file path)." << endl
        << "-refmac [exec. name]\t" << "name of REFMAC executable (default '" << PARAMS.REFMAC_NAME << "')." << endl
        << "-nmr\t\t\t" << "specify input file is NMR/MD ensemble." << endl
        << "-merge\t\t\t" << "merge chains within PDB file (to compare oligomers)." << endl
        << "-dna_rna\t\t" << "use nucleotide chains instead of protein chains." << endl
        << endl
        << "Target and Secondary Chain Selection:" << endl
        << "-p1 [file path]\t\t" << "location of target PDB file(s) (required)." << endl
        << "-p2 [file path]\t\t" << "location of secondary PDB file(s)." << endl
        << "-c1 [char]\t\t" << "chains of interest in PDB file(s) 1." << endl
        << "-c2 [char]\t\t" << "chains of interest in PDB file(s) 2." << endl
        << "-id\t\t\t" << "assumes all structures are sequence-identical" << endl
        << "\t\t\t" << "(requires direct correspondence of residue numbers)." << endl
        << "-allonall\t\t" << "do all-on-all comparison (not just half-matrix)." << endl
        << "-align [pdb] [chain(s)] [resnum (int)] [resnum (int)]" << endl
        << "\t\t\t" << "specify residue ranges (can use 'ALL' for pdb/chain)." << endl
        << "-align_rm [pdb] [chain(s)] [resnum (ins_code optional)]" << endl
        << "\t\t\t" << "remove single residues (can use 'ALL' for pdb/chain)." << endl
        << "-occup [double]\t\t" << "remove atoms with occupancy lower than (default '" << PARAMS.OCCUPANCY_THRESHOLD << "')." << endl
        << endl
        << "Fragment Library Options:" << endl
        << "-lib\t\t\t" << "use fragment library (-p2 will be ignored)." << endl
        << "-helix\t\t\t" << "run helix alignment (-p2 will be ignored)." << endl
        << "-strand\t\t\t" << "run strand alignment (-p2 will be ignored)." << endl
        << "-library_config [file]\t" << "alternative library config file." << endl 
        << "-library [dir path]\t" << "location of an alternative fragment library." << endl 
        << "-lib_score [double]\t" << "override all score thresholds in the fragment library." << endl 
        << "-lib_fraglen [int]\t" << "override all fragment lengths in the fragment library." << endl 
        << endl
        << "ProSMART ALIGN Alignment Options:" << endl
        << "-len [int]\t\t" << "length of fragment, in residues (default '" << PARAMS.FRAG_LEN << "')." << endl
        << "-score [double]\t\t" << "Procrustes cutoff for alignment (default '" << PARAMS.SCORE_THRESHOLD << "')." << endl
        << "-superpose [double]\t" << "Procrustes cutoff for superposition (default '" << PARAMS.SUPERPOSE << "')." << endl
       << "-superpose_side\t\t" << "Use side chain atoms when superposing structures." << endl
       << "-sieve\t\t\t" << "Use sieve-fitting to post-process superposition." << endl
       << "-sieve [double]\t\t" << "Use sieve-fitting specifying final RMSD (default '" << SIEVE_PARAMETER << "')." << endl
        << "-helixcutoff [double]\t" << "helix dissimilarity cutoff (default '" << PARAMS.HELIX_CUTOFF << "')." << endl
        << "-helixpenalty [double]\t" << "dynamic alignment penalty on helices (default '" << PARAMS.HELIX_PENALTY << "')." << endl
        << "-no_reward_seq\t\t" << "do not assign special score to seq-identical fragments." << endl
        << "-reward_seq [double]\t" << "score for seq-identical fragments (default '" << PARAMS.REWARD_SEQ << "')." << endl
        << "-skip_refine\t\t" << "Do not refine alignment." << endl
        << "-main_dist\t\t" << "include main N,C,O for side chain scores." << endl
        << "-numdist [double]\t" << "interatomic distance threshold for NumDist score." << endl
        << "-no_fix_errors\t\t" << "do not account for side chain flips during scoring." << endl
        << endl
        << "ProSMART ALIGN Rigid Substructure Identification Options:" << endl
        << "-cluster_skip\t\t" << "Skip searching for rigid substructures." << endl
        << "-cluster_score [double]\t" << "Procrustes score threshold (default '" << PARAMS.CLUSTER_SCORE << "')." << endl
        << "-cluster_angle [double]\t" << "Intrafragment rotation score threshold (default '" << PARAMS.CLUSTER_DEGREES << "')." << endl
        << "-cluster_min [double]\t" << "Min number of fragments in a cluster (default '" << PARAMS.CLUSTER_MIN << "')." << endl
        << "-cluster_link [double]\t" << "Single linkage threshold (cosine, default '" << PARAMS.CLUSTER_LINKAGE << "')." << endl
        << "-cluster_tr_link [d.]\t" << "Single linkage translation threshold (default '" << PARAMS.CLUSTER_TRANS_LINKAGE << "')." << endl
        << "-cluster_tr_max [d.]\t" << "Clustering max separation threshold (default '" << PARAMS.CLUSTER_TRANS_MAX << "')." << endl
        << "-cluster_rigid [double]\t" << "Final cluster rigidity (cosine, default '" << PARAMS.CLUSTER_RIGIDITY << "')." << endl
        << "-cluster_color [double]\t" << "Scales cluster color resolution (default '" << PARAMS.CLUSTER_COLOR << "')." << endl
        << "-output_dm\t\t" << "Output cluster distance matrices." << endl
        << endl
        << "ProSMART ALIGN Output Options:" << endl
        << "-cosine\t\t\t" << "Display intrafragment rotation score as cosine distance." << endl
        << "-score_strict\t\t" << "Require >0 occup, no alts, side chain completeness." << endl
        << "-output_rmsd\t\t" << "Output global and cluster-based atom-pair RMSD files." << endl
        << "-skip_screw\t\t" << "Skip calculation of screw axis-angle representation." << endl
        << "-quick\t\t\t" << "Don't output PDB files and colour scripts." << endl
        << "-out_pdb\t\t" << "Output superposed PDB files." << endl
        << "-out_pdb_full\t\t" << "Output PDB files comprise all chains, not just target." << endl
        << "-out_colour\t\t" << "Output PyMOL/Chimera colour scripts." << endl
        << "-colour_score [double]\t" << "colour score cutoff (default '" << PARAMS.ALIGN_SCORE << "')." << endl
        << "-side_score [double]\t" << "adjusts colour resolution of sidechains (default '" << PARAMS.SIDE_SCORE << "')." << endl
        << "-col1 [d. d. d.]\t" << "RGB colour code for similar (default '" << PARAMS.G1 << " " << PARAMS.G2 << " " << PARAMS.G3 << "')." << endl
        << "-col2 [d. d. d.]\t" << "RGB colour code for dissimilar (default '" << PARAMS.R1 << " " << PARAMS.R2 << " " << PARAMS.R3 << "')." << endl
        << endl
        //sequence alignment is hidden option, as it has minimal sanity checks, and requires specific formatting of input arguments.
       /*<< "Special Alternative Program Mode - Sequence Identity Calculation:" << endl
        << "-seq_align\t\t" << "All-on-all chain comparison, get sequence identity." << endl
        << "-seq_min [int]\t\t" << "Remove sequences with few residues (default '" << PARAMS.SEQ_MIN << "')." << endl
        << "-seq_ratio [double]\t" << "Require similar chain lengths (default ratio '" << PARAMS.SEQ_RATIO << "')." << endl
        << "-seq_id_min [double]\t" << "Remove low sequence identity pairs (default '" << PARAMS.SEQ_ID << "')." << endl
        << "-seq_id_max [double]\t" << "Remove high sequence identity pairs (default '" << PARAMS.SEQ_ID_MAX << "')." << endl
        << endl*/
        << "ProSMART RESTRAIN Options:" << endl
        << "-self_restrain\t\t" << "generic self-restraints, e.g. for DNA/RNA (ignore -p2)." << endl
        << "-restrain_all\t\t" << "generate restraints for all-on-all chains." << endl
        << "-restrain_best\t\t" << "only use best chain(s) for restraints (default)." << endl
        << "-restrain_to_self\t" << "allow target to be external chain (default disabled)." << endl
       << "-restrain_seqid [d.]\t" << "reference chain min seq.ident. criterion (default '" << PARAMS.RESTRAIN_SEQID << "')." << endl
       << "-restrain_score [d.]\t" << "reference chain max score criterion (default '" << PARAMS.RESTRAIN_SCORE << "')." << endl
       << "-rmax [double]\t\t" << "max restraint distance (default '" << PARAMS.sphere << "')." << endl
        << "-rmin [double]\t\t" << "min restraint distance (default '" << PARAMS.MIN_DISTANCE << "')." << endl
        << "-sigma [double]\t\t" << "default sigma (default '" << PARAMS.sigma << "')." << endl
        << "-min_sigma [double]\t" << "minimum possible value of sigma (default '" << PARAMS.MINIMUM_SIGMA << "')." << endl
        << "-sigmatype [int]\t" << "0: use default constant sigma." << endl << "\t\t\t1: estimate constant sigma." << endl << "\t\t\t2: generate distance-dependent sigmas." << endl
        << "-cutoff [double]\t" << "alignment score cutoff for restraints (default '" << PARAMS.score_cutoff << "')." << endl
        << "-side_cutoff [double]\t" << "sidechain score cutoff for restraints (default '" << PARAMS.RESTRAIN_SIDE_CUTOFF << "')." << endl
        << "-multiplier [double]\t" << "remove extreme outliers (default '" << PARAMS.RESTRAIN_MULTIPLIER_CUTOFF << "')." << endl
        << "-weight [double]\t" << "scales sigmas (default '" << PARAMS.SIGMA_WEIGHT<< "', i.e. no scaling)." << endl
        << "-rm_bonds\t\t" << "specifies for bond/angle restraints to be removed." << endl
        << "-main\t\t\t" << "only generate restraints for mainchain atoms." << endl
        << "-side\t\t\t" << "generate restraints for side and main chain atoms." << endl
        << "-restrain [pdb] [chain(s)] [resnum (int)] [resnum (int)]" << endl
        << "\t\t\t" << "specify residue ranges (can use 'ALL' for pdb/chain)." << endl
        << "-restrain_rm [pdb] [chain(s)] [resnum (ins_code optional)]" << endl
        << "\t\t\t" << "remove single residues (can use 'ALL' for pdb/chain)." << endl
        << "-output_pdb_chain_restraints\t" << "disabled by default, to save disk space." << endl
        << "-no_copy\t\t" << "don't copy final restraints files to main output dir." << endl
        << "-type [int]\t\t" << "0: bond type restraints (replace existing)." 
        << endl << "\t\t\t1: bond type restraints (add to existing)."
        << endl << "\t\t\t2: external restraints (default, recommended)." << endl
        << endl
        << "ProSMART RESTRAIN Generic Bonds (Secondary-Structure H-Bond Restraints):" << endl
            << "\tNOTE: if -h is specified then general main-chain h-bond restraints," << endl
            << "\tincluding those for helices and sheets, will be generated together." << endl
            << "\tIn contrast, if -h_helix and -h_sheet are both specified then" << endl
            << "\trestraints for helices and sheets will be generated separately." << endl
            << "\tNOTE: cannot also use keywords: -p2, -c2, -lib, -helix, -strand." << endl
            << "\tNOTE: fragment lib alignment info will only be used if -strict is used." << endl
        << "-h    (or -bond)\t" << "Generate generic bond restraints (default h-bond)." << endl
        << "\t\t\t" << "for whole main chain, including helices and sheets." << endl
        << "-h_helix\t\t" << "hydrogen bond restraints for all helices." << endl
        << "-h_sheet\t\t" << "hydrogen bond restraints for beta-sheets." << endl
        << "-strict\t\t\t" << "require sufficient similarity to library fragments." << endl
        << "-3_10\t\t\t" << "only generate helix restraints for 3_10-helices." << endl
        << "-alpha\t\t\t" << "only generate helix restraints for alpha-helices." << endl
        << "-pi\t\t\t" << "only generate helix restraints for pi-helices." << endl
        /*<< "-parallel\t\t\t" << "." << endl
        << "-antiparallel\t\t\t" << "." << endl*/
        << "-bond_dist [double]\t" << "target value of the bond restraints (default '" << PARAMS.HBOND_DIST << "')." << endl
        << "-bond_min [double]\t" << "min interatomic distance for detection (default '" << PARAMS.HBOND_DIST_MIN << "')." << endl
        << "-bond_max [double]\t" << "max interatomic distance for detection (default '" << PARAMS.HBOND_DIST_MAX << "')." << endl
        << "-min_sep [int]\t\t" << "min residue separation between atom-pairs (default '" << PARAMS.MIN_SEP << "')." << endl
        << "-max_sep [int]\t\t" << "max residue separation between atom-pairs (default off)." << endl
        << "-allow_sep [int] ...\t" << "specify to allow only specific residue separations." << endl
        << "-rm_sep [int] ...\t" << "specify to disallow specific residue separations." << endl
        << "-bond_opt [int]\t\t" << "Controls how atom-pairs are selected (auto recommended):" << endl
        << "\t\t\t" << "1: only O-N restraints (used for helices)." << endl
        << "\t\t\t" << "2: O-N and N-O restraints (used for main-chain/sheets)." << endl
        << "\t\t\t" << "3: do not filter - allow all main-chain atom-pairs." << endl
        << "-bond_override [int]\t" << "Override default number of allowed bonds per atom." << endl
        << endl        
        << "Troubleshooting Options (display extra info in logfiles):" << endl
        << "-troubleshoot_restraint_files" << endl
        << "-troubleshoot_hbond_restraints" << endl
        << endl
        << "Other Options:" << endl
        << "-renamechain [char]\t" << "regenerates the PDB file with new chain_ID." << endl 
        << "-output_seq\t\t" << "Output sequence files then terminate." << endl
        << endl
        << "See online documentation for more info on functionality and usage." << endl
        << "http://www2.mrc-lmb.cam.ac.uk/groups/murshudov/  (select: Software > ProSMART)"
        << endl << endl;
    } else {
        for(int i=1; i<argc; i++){
            args.push_back(argv[i]);
        }	
    }
	
	return args;
}

void read_args(vector<string> &args, params &PARAMS)
{
	string arg_file;
    vector<string> ext_args;
    bool file_exists;

    while(1){
        arg_file = "";
        for(unsigned int i=0; i<args.size()-1; i++){
            if(args[i] == "-f"){			//specifies external arguments text file
                if(args[i+1][0] == '-'){
                    //xml_entry(PARAMS.error_file,0,0,args[i]);
                    cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
                    exit(-1);
                }
                if(args[i+1].size()>2 && args[i+1][0] == '\"' && args[i+1][args[i+1].size()-1] == '\"'){
                    args[i+1].erase(args[i+1].begin()+args[i+1].size()-1);
                    args[i+1].erase(args[i+1].begin());
                }
                arg_file = args[i+1];
                args.erase(args.begin()+i+1);
                args.erase(args.begin()+i);
                break;
            }
        }
        if(arg_file!=""){					//this allows a external (space and/or newline separated) text file to be used instead of, or as well as, command line arguments.
            cout << endl << "Reading keywords file: " << arg_file << endl;
            file_exists = check_file_exists(arg_file);
            if(file_exists == 0){
                cout << endl << "Error: " << arg_file << " could not be read." << endl << "Program terminated." << endl << endl; 
                exit(-1);
            }
            ext_args = read_file(arg_file);
            if(ext_args.size()==0){
                cout << endl << "Warning: no keywords found in " << arg_file << "." << endl << "Continuing anyway." << endl;
            }
            for(unsigned int i=0; i<ext_args.size(); i++){
                args.push_back(ext_args[i]);
                if(ext_args[i][0]=='\"'){
                    if(ext_args[i][ext_args[i].size()-1]!='\"'){
                        for(unsigned int j=i+1; j<ext_args.size(); j++){
                            args[args.size()-1] += " " + ext_args[j];
                            i++;
                            if(ext_args[j][ext_args[j].size()-1]=='\"'){
                                break;
                            }
                        }
                    }
                    args[args.size()-1].erase(args[args.size()-1].begin()+args[args.size()-1].size()-1);
                    args[args.size()-1].erase(args[args.size()-1].begin());
                }
            }
            if(args.size()==0){
                cout << endl << "Error: could not find keywords in external file: " << arg_file << ". Program terminated." << endl; 
                exit(-1);
            }
        } else {
            break;
        }
    }
    
    
	//start by getting output directory, and creating directory.
	string tmp_str = "";
	for(unsigned int i=0; i<args.size()-1; i++){
		if(args[i] == "-o"){
			if(args[i+1][0] == '-'){
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);
            }
            remove_quotes(args[i+1]);   //only if not windows
			PARAMS.output_dir = make_valid_directory(tmp_str,args[i+1]);
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			break;
		}
	}
    
    for(unsigned int i=0; i<args.size()-1; i++){
        if(args[i] == "-input_dir"){
            if(args[i+1][0] == '-'){
                cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
                exit(-1);
            }
            remove_quotes(args[i+1]);   //only if not windows
            PARAMS.INPUT_DIR = make_valid_directory(tmp_str,args[i+1]);
            args.erase(args.begin()+i+1);
            args.erase(args.begin()+i);
            break;
        }
    }


	create_directory(tmp_str,PARAMS.output_dir);
	
	for(unsigned int i=0; i<args.size(); i++){		//specify whether or not to output .xml file
		if(args[i] == "-xml"){
			if(i<args.size()-1){
				if(args[i+1][0] == '-'){
					PARAMS.error_file = PARAMS.output_dir + "xmlout.xml";
				} else {
					PARAMS.error_file = args[i+1];
					args.erase(args.begin()+i+1);
				}
			} else {
				PARAMS.error_file = PARAMS.output_dir + "xmlout.xml";
			}
			initialise_xml(PARAMS.error_file);
			args.erase(args.begin()+i);
			break;
		}
    }
   
   for(unsigned int i=0; i<args.size()-1; i++){	//for priority arguments that have a string input in the next argument
      if(args.size()<2)break;
      
		//PROGRAM OPTIONS
		if(args[i] == "-sieve"){
			if(args[i+1][0] == '-'){
				continue;
         }
         USE_SIEVE = 1;
			SIEVE_PARAMETER = atof(args[i+1].c_str());
         if(SIEVE_PARAMETER <= 0.0){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Argument: " << args[i] << " must be positive. Program terminated." << endl; 
				exit(-1);}
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
   }
    
    for(unsigned int i=0; i<args.size(); i++){		//for arguments that don't have a seperate input
       
		//PROGRAM OPTIONS
        if(read_args(args,i,"-a",PARAMS.ALIGN,1,PARAMS.ALIGN_MODE,"0"))continue;
        if(read_args(args,i,"-a1",PARAMS.ALIGN,1,PARAMS.ALIGN_MODE,"1"))continue;
        if(read_args(args,i,"-a2",PARAMS.ALIGN,1,PARAMS.ALIGN_MODE,"2"))continue;
        if(read_args(args,i,"-r",PARAMS.RESTRAIN,1))continue;
       if(read_args(args,i,"-dna_rna",USE_DNARNA,1))continue;
       if(read_args(args,i,"-sieve",USE_SIEVE,1))continue;
        if(read_args(args,i,"-merge",PARAMS.MERGE_CHAINS,1))continue;
        if(read_args(args,i,"-allonall",PARAMS.TRIANGULAR,0))continue;
       if(read_args(args,i,"-output_seq",ONLY_OUTPUT_SEQUENCE_FILES,1))continue;
#ifndef WINDOWS
       //sequence alignment doesn't work on windows, as it uses waitpid()
        if(read_args(args,i,"-seq_align",PARAMS.SEQUENCE_ALIGNMENT,1)){
            PARAMS.MAX_THREADS = 8;
            continue;
        }
#endif 
		if(args[i] == "-nmr"){
			PARAMS.NMR = 1;
			PARAMS.IDENTICAL = "1";
			if(i<args.size()-1){
				if(args[i+1][0] != '-'){
					PARAMS.NMR_TARGET = atoi(args[i+1].c_str());
					args.erase(args.begin()+i+1);
				}
			}
			args.erase(args.begin()+i);
			i--;
			if(args.size()<1) break; else continue;
		}
		
		//ALIGN
        if(read_args(args,i,"-skip_refine",PARAMS.EXTRA_REFINEMENT,"0"))continue;
        if(read_args(args,i,"-lib",PARAMS.HELIX,"1",PARAMS.HELIX_ONLY,"0",PARAMS.STRAND_ONLY,"0"))continue;
        if(read_args(args,i,"-helix",PARAMS.HELIX,"1",PARAMS.HELIX_ONLY,"1"))continue;
        if(read_args(args,i,"-strand",PARAMS.HELIX,"1",PARAMS.STRAND_ONLY,"1"))continue;
        if(read_args(args,i,"-id",PARAMS.IDENTICAL,"1"))continue;
        if(read_args(args,i,"-cluster_skip",PARAMS.CLUSTER,"0"))continue;
        if(read_args(args,i,"-out_pdb",PARAMS.OUT_PDB,"1"))continue;
        if(read_args(args,i,"-out_pdb_full",PARAMS.OUT_PDB_FULL,"1"))continue;
        if(read_args(args,i,"-out_color",PARAMS.OUT_COLOR,"1"))continue;
        if(read_args(args,i,"-skip_screw",PARAMS.OUTPUT_SCREW,"0"))continue;
        if(read_args(args,i,"-id",PARAMS.IDENTICAL,"1"))continue;
        if(read_args(args,i,"-quick",PARAMS.OUT_PDB,"0",PARAMS.OUT_COLOR,"0"))continue;
        if(read_args(args,i,"-superpose_side",PARAMS.SUPERPOSE_SIDE,"1"))continue;
        if(read_args(args,i,"-output_dm",PARAMS.OUTPUT_CLUSTER_DM,"1"))continue;
        if(read_args(args,i,"-output_rmsd",PARAMS.OUTPUT_RMSD_FILES,"1"))continue;
        if(read_args(args,i,"-cosine",PARAMS.DISPLAY_AS_COSINE_DISTANCE,"1"))continue;
        if(read_args(args,i,"-main_dist",PARAMS.USE_MAINCHAIN_FOR_SIDERMSD,"1"))continue;
        if(read_args(args,i,"-no_fix_errors",PARAMS.FIX_SIDE_CHAIN_ERRORS,"0"))continue;
        if(read_args(args,i,"-no_reward_seq",PARAMS.NO_REWARD_SEQ,"1"))continue;
       if(read_args(args,i,"-score_strict",PARAMS.SCORE_STRICT,"1"))continue;
		
		//RESTRAIN
        if(read_args(args,i,"-self_restrain",PARAMS.GENERIC,"1"))continue;
        if(read_args(args,i,"-rm_bonds",PARAMS.GET_BONDS,"1"))continue;
        if(read_args(args,i,"-main",PARAMS.MAINCHAIN_ONLY,"1"))continue;
        if(read_args(args,i,"-side",PARAMS.MAINCHAIN_ONLY,"0"))continue;
        if(read_args(args,i,"-restrain_best",PARAMS.RESTRAIN_OPTION,0))continue;
        if(read_args(args,i,"-restrain_all",PARAMS.RESTRAIN_OPTION,1))continue;
        if(read_args(args,i,"-restrain_to_self",PARAMS.RESTRAIN_TO_SELF,1))continue;
        if(read_args(args,i,"-bfac1",PARAMS.BFAC1,"1"))continue;
        if(read_args(args,i,"-bfac2",PARAMS.BFAC2,"1"))continue;
        if(read_args(args,i,"-output_pdb_chain_restraints",PARAMS.OUTPUT_PDB_CHAIN_RESTRAINTS,1))continue;
        if(read_args(args,i,"-no_copy",PARAMS.COPY_FINAL_RESTRAINTS_FILES,0))continue;
        
        //RESTRAIN HBOND
        if(read_args(args,i,"-h",PARAMS.HBOND_RESTRAINTS,1))continue;
        if(read_args(args,i,"-bond",PARAMS.HBOND_RESTRAINTS,1))continue;
        if(read_args(args,i,"-h_helix",PARAMS.HBOND_RESTRAINTS,1,PARAMS.H_HELIX,1))continue;
        if(read_args(args,i,"-3_10",PARAMS.HBOND_RESTRAINTS,1,PARAMS.H_HELIX,1,PARAMS.H_HELIX_3,1))continue;
        if(read_args(args,i,"-alpha",PARAMS.HBOND_RESTRAINTS,1,PARAMS.H_HELIX,1,PARAMS.H_HELIX_4,1))continue;
        if(read_args(args,i,"-pi",PARAMS.HBOND_RESTRAINTS,1,PARAMS.H_HELIX,1,PARAMS.H_HELIX_5,1))continue;
        if(read_args(args,i,"-h_sheet",PARAMS.HBOND_RESTRAINTS,1,PARAMS.H_SHEET,1))continue;
        if(read_args(args,i,"-strict",PARAMS.STRICT_FRAG,1))continue;

        //TROUBLESHOOTING FLAGS
        if(read_args(args,i,"-troubleshoot_restraint_files",PARAMS.TROUBLESHOOT_RESTRAINT_FILES,1))continue;
        if(read_args(args,i,"-troubleshoot_hbond_restraints",PARAMS.TROUBLESHOOT_HBOND_RESTRAINTS,1))continue;
    }
    
	if(args.size()==0){
		cout << endl << "Invalid Syntax: invalid/insufficient arguments specified." << endl << "Program terminated." << endl;
		exit(-1);
	}
   
   if(USE_DNARNA){
      PARAMS.CLUSTER_TRANS_MAX = "20.0";
   }
    
    for(unsigned int i=0; i<args.size()-1; i++){	//for arguments that have a string input in the next argument
        if(args.size()<2)break;
        
		//PROGRAM OPTIONS
#ifndef WINDOWS
		if(args[i] == "-threads"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			PARAMS.MAX_THREADS = atoi(args[i+1].c_str());
			if(PARAMS.MAX_THREADS < 1){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Argument: " << args[i] << " must be positive. Program terminated." << endl; 
				exit(-1);}
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
#endif
        if(read_args(PARAMS.error_file,args,i,"-refmac",PARAMS.REFMAC_NAME))continue;
		
		//TARGET SELECTION
		if(args[i] == "-p1"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);
            }
            remove_quotes(args[i+1]);
			args.erase(args.begin()+i);
			while(args[i][0] != '-'){
                remove_quotes(args[i]);
                if(PARAMS.INPUT_DIR.size()>0)
                    PARAMS.filenames1.push_back(PARAMS.INPUT_DIR+args[i]);
                else
                    PARAMS.filenames1.push_back(args[i]);
				args.erase(args.begin()+i);
				if(args.size()<=i){break;}
			}
			i--;
			continue;
		}
		if(args[i] == "-p2"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);
            }
            remove_quotes(args[i+1]);
			args.erase(args.begin()+i);
			while(args[i][0] != '-'){
                remove_quotes(args[i]);
                if(PARAMS.INPUT_DIR.size()>0)
                    PARAMS.filenames2.push_back(PARAMS.INPUT_DIR+args[i]);
                else
                    PARAMS.filenames2.push_back(args[i]);
				args.erase(args.begin()+i);
				if(args.size()<=i){break;}
			}
			i--;
			continue;
		}
		if(args[i] == "-c1"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			if(args[i+1].size()>1){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax - length of " << args[i] << " must be 1 character. Program terminated." << endl; 
				exit(-1);}
			args.erase(args.begin()+i);
			while(args[i][0] != '-'){
				if(args[i].size()>1){
					xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax - length of -c1 must be 1 character. Program terminated." << endl;
					exit(-1);}
				PARAMS.chains1.push_back(args[i]);
				args.erase(args.begin()+i);
				if(args.size()<=i){break;}
			}
			i--;
			continue;
		}
		if(args[i] == "-c2"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			if(args[i+1].size()>1){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: length of " << args[i] << " must be 1 character. Program terminated." << endl;
				exit(-1);}
			args.erase(args.begin()+i);
			while(args[i][0] != '-'){
				if(args[i].size()>1){
					xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax - length of -c2 must be 1 character. Program terminated." << endl;
					exit(-1);}
				PARAMS.chains2.push_back(args[i]);
				args.erase(args.begin()+i);
				if(args.size()<=i){break;}
			}
			i--;
			continue;
		}

		//ALIGN OPTIONS
        
        if(read_args(PARAMS.error_file,args,i,"-len",PARAMS.FRAG_LEN))continue;
        if(read_args(PARAMS.error_file,args,i,"-score",PARAMS.SCORE_THRESHOLD))continue;
        if(read_args(PARAMS.error_file,args,i,"-colour_score",PARAMS.ALIGN_SCORE))continue;
        if(read_args(PARAMS.error_file,args,i,"-superpose",PARAMS.SUPERPOSE))continue;

        if(read_args(PARAMS.error_file,args,i,"-refmac",PARAMS.REFMAC_NAME))continue;
		if(args[i] == "-side_score"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			PARAMS.SIDE_SCORE = args[i+1].c_str();
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
        if(read_args(PARAMS.error_file,args,i,"-helixcutoff",PARAMS.HELIX_CUTOFF))continue;
        if(read_args(PARAMS.error_file,args,i,"-helixpenalty",PARAMS.HELIX_PENALTY))continue;
        if(read_args_signed(args,i,"-reward_seq",PARAMS.REWARD_SEQ))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_score",PARAMS.CLUSTER_SCORE))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_angle",PARAMS.CLUSTER_DEGREES))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_min",PARAMS.CLUSTER_MIN))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_link",PARAMS.CLUSTER_LINKAGE))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_tr_link",PARAMS.CLUSTER_TRANS_LINKAGE))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_tr_max",PARAMS.CLUSTER_TRANS_MAX))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_rigid",PARAMS.CLUSTER_RIGIDITY))continue;
        if(read_args(PARAMS.error_file,args,i,"-cluster_color",PARAMS.CLUSTER_COLOR))continue;
        if(read_args(PARAMS.error_file,args,i,"-numdist",PARAMS.NUM_DIST_THRESHOLD))continue;
        
        if(read_args_int(PARAMS.error_file,args,i,"-seq_min",PARAMS.SEQ_MIN))continue;
        if(read_args_double(PARAMS.error_file,args,i,"-seq_ratio",PARAMS.SEQ_RATIO))continue;
        if(read_args_double(PARAMS.error_file,args,i,"-seq_id_min",PARAMS.SEQ_ID))continue;
        if(read_args_double(PARAMS.error_file,args,i,"-seq_id_max",PARAMS.SEQ_ID_MAX))continue;

       if(read_args_double(PARAMS.error_file,args,i,"-occup",PARAMS.OCCUPANCY_THRESHOLD))continue;
        
		//RESTRAIN OPTIONS
       if(read_args_double(PARAMS.error_file,args,i,"-restrain_seqid",PARAMS.RESTRAIN_SEQID))continue;
       if(read_args_double(PARAMS.error_file,args,i,"-restrain_score",PARAMS.RESTRAIN_SCORE))continue;
        if(read_args(PARAMS.error_file,args,i,"-rmax",PARAMS.sphere))continue;
        if(read_args(PARAMS.error_file,args,i,"-rmin",PARAMS.MIN_DISTANCE))continue;
        if(read_args(PARAMS.error_file,args,i,"-sigma",PARAMS.sigma))continue;
		if(args[i] == "-sigmatype"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			PARAMS.sigma_method = args[i+1];
			switch(atoi(PARAMS.sigma_method.c_str())){	//must be 0, 1 or 2.
				case 0: break;
				case 1: 
				case 2: 
					PARAMS.GET_BONDS="1";	//estimated sigmas requires refmac
					break;
                case 3: 
					PARAMS.GET_ATOMIC_UNCERTAINTY="1";
					break;
				default: xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax: " << args[i] << " incorrectly specified. Program terminated." << endl; 
					exit(-1);
			}
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
        if(read_args(PARAMS.error_file,args,i,"-cutoff",PARAMS.score_cutoff))continue;
        if(read_args(PARAMS.error_file,args,i,"-side_cutoff",PARAMS.RESTRAIN_SIDE_CUTOFF))continue;
        if(read_args(PARAMS.error_file,args,i,"-multiplier",PARAMS.RESTRAIN_MULTIPLIER_CUTOFF))continue;
        if(read_args(PARAMS.error_file,args,i,"-weight",PARAMS.SIGMA_WEIGHT))continue;
        if(read_args(PARAMS.error_file,args,i,"-min_sigma",PARAMS.MINIMUM_SIGMA))continue;
        if(args[i] == "-type"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			PARAMS.TYPE = args[i+1];
			switch(atoi(PARAMS.TYPE.c_str())){	//must be 0, 1 or 2.
				case 0: break;
				case 1: break;
				case 2:	break;
				default: xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax: " << args[i] << " incorrectly specified. Program terminated." << endl; 
					exit(-1);
			}
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
        
        //RESTRAIN HBOND
        if(read_args_double(PARAMS.error_file,args,i,"-bond_dist",PARAMS.HBOND_DIST))continue;
        if(read_args_double(PARAMS.error_file,args,i,"-bond_min",PARAMS.HBOND_DIST_MIN))continue;
        if(read_args_double(PARAMS.error_file,args,i,"-bond_max",PARAMS.HBOND_DIST_MAX))continue;
        if(read_args_int(PARAMS.error_file,args,i,"-max_sep",PARAMS.MAX_SEP))continue;
        if(args[i] == "-min_sep"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			PARAMS.MIN_SEP = str_to_int(args[i+1]);
            PARAMS.FORCE_MIN_SEP = 1;
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
        if(read_args_v_int(PARAMS.error_file,args,i,"-allow_sep",PARAMS.ALLOW_SEP)){
            PARAMS.FORCE_ALLOW_SEP = 1;
            continue;
        }
        if(read_args_v_int(PARAMS.error_file,args,i,"-rm_sep",PARAMS.RM_SEP))continue;
        if(args[i] == "-bond_opt"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			PARAMS.HBOND_OPTIONS = str_to_int(args[i+1]);
            PARAMS.FORCE_OPTIONS = 1;
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
        if(read_args_int(PARAMS.error_file,args,i,"-bond_override",PARAMS.OVERRIDE_MAX_BONDS))continue;
		
		//OTHER
		if(args[i] == "-renamechain"){
			if(args[i+1][0] == '-'){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: " << args[i] << " not specified. Program terminated." << endl; 
				exit(-1);}
			if(args[i+1].size()>1){
				xml_entry(PARAMS.error_file,0,0,args[i]);
				cout << endl << "Invalid Syntax: length of " << args[i] << " must be 1 character. Program terminated." << endl;
				exit(-1);}
			PARAMS.new_chain = args[i+1];
			args.erase(args.begin()+i+1);
			args.erase(args.begin()+i);
			i--;
			if(args.size()<2) break; else continue;
		}
        if(read_args(PARAMS.error_file,args,i,"-library_config",PARAMS.library_config))continue;
        if(read_args(PARAMS.error_file,args,i,"-library",PARAMS.LIB_DIR))continue;
        if(read_args(PARAMS.error_file,args,i,"-lib_score",PARAMS.OVERRIDE_LIB_SCORE))continue;
        if(read_args(PARAMS.error_file,args,i,"-lib_fraglen",PARAMS.OVERRIDE_LIB_FRAGLEN))continue;
    }
    
	if(args.size()>3){
		for(unsigned int i=0; i<args.size()-3; i++){	//for arguments that have 3 string inputs in the next argument
			if(args.size()<4) break;
			
            //COLORS
            if(read_args(PARAMS.error_file,args,i,"-col1",PARAMS.G1,PARAMS.G2,PARAMS.G3))continue;
            if(read_args(PARAMS.error_file,args,i,"-col2",PARAMS.R1,PARAMS.R2,PARAMS.R3))continue;
		}
	}
	
	string tmp_PDB1="";
	string tmp_CHAIN1="";
	string tmp_PDB2="";
	string tmp_CHAIN2="";
	if(args.size()>6){
		for(unsigned int i=0; i<args.size()-6; i++){
			
			//RANGES
			if(args[i] == "-range"){
				if(args[i+1][0] == '-' || args[i+2][0] == '-' || args[i+3][0] == '-' || args[i+4][0] == '-'){
					xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax: " << args << ". Range incorrectly specified. Program terminated." << endl; 
					exit(-1);
				}
				tmp_PDB1 = args[i+1];
				tmp_CHAIN1 = args[i+2];
				tmp_PDB2 = args[i+3];
				tmp_CHAIN2 = args[i+4];
				if(tmp_PDB1=="ALL"){tmp_PDB1 = "";}
				if(tmp_CHAIN1=="ALL"){tmp_CHAIN1 = "";}
				if(tmp_PDB2=="ALL"){tmp_PDB2 = "";}
				if(tmp_CHAIN2=="ALL"){tmp_CHAIN2 = "";}
				for(unsigned int j=0; j<4; j++){
					args.erase(args.begin()+i+1);
				}
				while(args.size()>i+1){
					if(args[i+1][0] == '-' || args[i+2][0] == '-'){
						xml_entry(PARAMS.error_file,0,0,args[i]);
						cout << endl << "Invalid Syntax: " << args[i+1] << " " << args[i+2] << ". Range incorrectly specified. Program terminated." << endl; 
						exit(-1);
					}
					PARAMS.PDB1A.push_back(tmp_PDB1);
					PARAMS.CHAIN1A.push_back(tmp_CHAIN1);
					PARAMS.PDB1B.push_back(tmp_PDB2);
					PARAMS.CHAIN1B.push_back(tmp_CHAIN2);
					PARAMS.RANGE1A.push_back(args[i+1]);
					PARAMS.RANGE1B.push_back(args[i+2]);
					
					args.erase(args.begin()+i+2);
					args.erase(args.begin()+i+1);
					if(args.size()>i+1){
						if(args[i+1][0] == '-'){
							break;
						}
					}
				}
				args.erase(args.begin()+i);
				i--;
				if(args.size()<3) break; else continue;
			}
		}
	}
    
    if(args.size()>4){
		for(unsigned int i=0; i<args.size()-4; i++){
			
			//RANGES
			if(args[i] == "-align"){
                
                if(args[i+1][0] == '-' || args[i+2][0] == '-' || args[i+3]!=int_to_str(str_to_int(args[i+3])) || args[i+4]!=int_to_str(str_to_int(args[i+4])) ){
					xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax: " << args << ". Align range incorrectly specified. Program terminated." << endl; 
					exit(-1);
				}
                PARAMS.ALIGN_PDB.push_back(args[i+1]);
                PARAMS.ALIGN_CHAIN.push_back(args[i+2]);
                PARAMS.ALIGN_RANGE1.push_back(str_to_int(args[i+3]));
                PARAMS.ALIGN_RANGE2.push_back(str_to_int(args[i+4]));
                
                for(unsigned int j=0; j<5; j++){
					args.erase(args.begin()+i);
				}
				i--;
				if(args.size()<3) break; else continue;
			}
		}
	}
    
    if(args.size()>4){
		for(unsigned int i=0; i<args.size()-4; i++){
			
			//RANGES
			if(args[i] == "-restrain"){
                
                if(args[i+1][0] == '-' || args[i+2][0] == '-' || args[i+3]!=int_to_str(str_to_int(args[i+3])) || args[i+4]!=int_to_str(str_to_int(args[i+4])) ){
					xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax: " << args << ". Restrain range incorrectly specified. Program terminated." << endl; 
					exit(-1);
				}
                PARAMS.RESTRAIN_PDB.push_back(args[i+1]);
                PARAMS.RESTRAIN_CHAIN.push_back(args[i+2]);
                PARAMS.RESTRAIN_RANGE1.push_back(str_to_int(args[i+3]));
                PARAMS.RESTRAIN_RANGE2.push_back(str_to_int(args[i+4]));
                
                for(unsigned int j=0; j<5; j++){
					args.erase(args.begin()+i);
				}
				i--;
				if(args.size()<3) break; else continue;
			}
		}
	}
    
    if(args.size()>3){
		for(unsigned int i=0; i<args.size()-3; i++){
			
			//RANGES
			if(args[i] == "-align_rm"){
                
                if(args[i+1][0] == '-' || args[i+2][0] == '-'){
					xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax: " << args << ". -align_rm incorrectly specified. Program terminated." << endl; 
					exit(-1);
				}
                if(args.size()>4){
                    if(args[i+4][0] != '-'){
                        xml_entry(PARAMS.error_file,0,0,args[i+4]);
                        cout << endl << "Invalid Syntax: " << args << ". -restrain_rm incorrectly specified.";
                        cout << endl << "Argument: '" << args[i+4] << "' incorrectly specified."; 
                        cout << endl << "-algin_rm specifies individual residue, not residue range. Program terminated." << endl; 
                        exit(-1);
                    }
                }
                PARAMS.ALIGN_RM_PDB.push_back(args[i+1]);
                PARAMS.ALIGN_RM_CHAIN.push_back(args[i+2]);
                PARAMS.ALIGN_RM_RESIDUE.push_back(args[i+3]);
                
                for(unsigned int j=0; j<4; j++){
					args.erase(args.begin()+i);
				}
				i--;
				if(args.size()<3) break; else continue;
			}
		}
	}
    
    if(args.size()>3){
		for(unsigned int i=0; i<args.size()-3; i++){
			
			//RANGES
			if(args[i] == "-restrain_rm"){
                
                if(args[i+1][0] == '-' || args[i+2][0] == '-'){
					xml_entry(PARAMS.error_file,0,0,args[i]);
					cout << endl << "Invalid Syntax: " << args << ". -restrain_rm incorrectly specified. Program terminated." << endl; 
					exit(-1);
				}
                if(args.size()>4){
                    if(args[i+4][0] != '-'){
                        xml_entry(PARAMS.error_file,0,0,args[i+4]);
                        cout << endl << "Invalid Syntax: " << args << ". -restrain_rm incorrectly specified.";
                        cout << endl << "Argument: '" << args[i+4] << "' incorrectly specified."; 
                        cout << endl << "-restrain_rm specifies individual residue, not residue range. Program terminated." << endl; 
                        exit(-1);
                    }
                }
                PARAMS.RESTRAIN_RM_PDB.push_back(args[i+1]);
                PARAMS.RESTRAIN_RM_CHAIN.push_back(args[i+2]);
                PARAMS.RESTRAIN_RM_RESIDUE.push_back(args[i+3]);
                
                for(unsigned int j=0; j<4; j++){
					args.erase(args.begin()+i);
				}
				i--;
				if(args.size()<3) break; else continue;
			}
		}
	}
    
    if(PARAMS.HBOND_RESTRAINTS){
        if(!PARAMS.H_HELIX && !PARAMS.H_SHEET){     
            PARAMS.HELIX = "0";                     //combined (full main-chain restraints)
            PARAMS.filenames2.clear();
            PARAMS.chains2.clear();
        } else {
            PARAMS.HELIX = "1";
            if(PARAMS.H_HELIX)PARAMS.HELIX_ONLY = "1";
            if(PARAMS.H_SHEET)PARAMS.STRAND_ONLY = "1";
        }
    }
   
   //if generating generic restraints, using fragment library or generating h-bond restraints, skip rigid substructure identification.
   if(PARAMS.GENERIC == 1 || PARAMS.HELIX=="1" || PARAMS.HBOND_RESTRAINTS){
      PARAMS.CLUSTER = "0";
   }
    
    return;	
}

void set_hbond_parameters(params &PARAMS)
{
    if(!PARAMS.HBOND_RESTRAINTS)return;
    
    string filename;
    if(!PARAMS.FORCE_OPTIONS)PARAMS.HBOND_OPTIONS = 2;  //suitable for main-chain and beta-sheet
    
    for(unsigned int i=1; i<PARAMS.ALLOW_SEP.size(); i++)
        for(unsigned int j=0; j<i; j++)
            if(PARAMS.ALLOW_SEP[j]==PARAMS.ALLOW_SEP[i]){
                PARAMS.ALLOW_SEP.erase(PARAMS.ALLOW_SEP.begin()+i);
                i--;
                break;
            }
    for(unsigned int i=1; i<PARAMS.RM_SEP.size(); i++)
        for(unsigned int j=0; j<i; j++)
            if(PARAMS.RM_SEP[j]==PARAMS.RM_SEP[i]){
                PARAMS.RM_SEP.erase(PARAMS.RM_SEP.begin()+i);
                i--;
                break;
            }
    
    
    //whole main-chain restraints
    
    if(!PARAMS.H_HELIX && !PARAMS.H_SHEET){     
        filename = PARAMS.INPUTCHAINS_dir+"hbond_main.txt";
        PARAMS.H_FILENAMES.push_back(filename);
        write_hbond_parameters_file(PARAMS,filename);
    }
    
    //separate helix restraints
    
    if(PARAMS.H_HELIX){                         
        if(PARAMS.H_HELIX_3)PARAMS.ALLOW_SEP.push_back(3);
        if(PARAMS.H_HELIX_4)PARAMS.ALLOW_SEP.push_back(4);
        if(PARAMS.H_HELIX_5)PARAMS.ALLOW_SEP.push_back(5);
        if(!PARAMS.H_HELIX_3 && !PARAMS.H_HELIX_4 && !PARAMS.H_HELIX_5){
            PARAMS.ALLOW_SEP.push_back(3);
            PARAMS.ALLOW_SEP.push_back(4);
            PARAMS.ALLOW_SEP.push_back(5);
        }
        for(unsigned int i=1; i<PARAMS.ALLOW_SEP.size(); i++)
            for(unsigned int j=0; j<i; j++)
                if(PARAMS.ALLOW_SEP[j]==PARAMS.ALLOW_SEP[i]){
                    PARAMS.ALLOW_SEP.erase(PARAMS.ALLOW_SEP.begin()+i);
                    i--;
                    break;
                }
        if(!PARAMS.FORCE_OPTIONS)PARAMS.HBOND_OPTIONS = 1;
        
        filename = PARAMS.INPUTCHAINS_dir+"hbond_helix.txt";
        PARAMS.H_FILENAMES.push_back(filename);
        write_hbond_parameters_file(PARAMS,filename);
    }
    
    //separate sheet restraints
    
    if(PARAMS.H_SHEET){
        if(!PARAMS.FORCE_MIN_SEP)PARAMS.MIN_SEP = 6;
        if(!PARAMS.FORCE_OPTIONS)PARAMS.HBOND_OPTIONS = 2;

        filename = PARAMS.INPUTCHAINS_dir+"hbond_sheet.txt";
        PARAMS.H_FILENAMES.push_back(filename);
        write_hbond_parameters_file(PARAMS,filename);
    }
    
    return;
}

void write_hbond_parameters_file(params &PARAMS,string &filename)
{
    ofstream outfile;
    outfile.open(filename.c_str());
    if(!outfile.is_open()){
        xml_entry(PARAMS.error_file,0,5,filename);
        cout << "Unable to open output file " << filename << " for writing. Program terminated." << endl;
        exit(-1);
    }
    outfile << PARAMS.HBOND_DIST << endl;
    outfile << PARAMS.HBOND_DIST_MIN << endl;
    outfile << PARAMS.HBOND_DIST_MAX << endl;
    outfile << PARAMS.MIN_SEP << endl;
    outfile << PARAMS.MAX_SEP << endl;
    outfile << "#" << endl;
    for(unsigned int i=0; i<PARAMS.ALLOW_SEP.size(); i++){
        outfile << PARAMS.ALLOW_SEP[i] << endl;
    }
    outfile << "#" << endl;
    for(unsigned int i=0; i<PARAMS.RM_SEP.size(); i++){
        outfile << PARAMS.RM_SEP[i] << endl;
    }
    outfile << "#" << endl;
    outfile << PARAMS.HBOND_OPTIONS << endl;
    outfile << PARAMS.OVERRIDE_MAX_BONDS << endl;
    outfile << PARAMS.TROUBLESHOOT_HBOND_RESTRAINTS << endl;
    outfile << PARAMS.STRICT_FRAG << endl;
    outfile << "#END" << endl;
    outfile.close();
    return;
}

void param_validation(vector<string> &args, params &PARAMS)
{
	//error if there were unknown arguments specified.
    if(args.size()>0){
        xml_entry(PARAMS.error_file,0,1);
        cout << endl << "Invalid Syntax: invalid specification of arguments: " << args << endl << "Program terminated." << endl;
		exit(-1);
    }
    
    if(PARAMS.SEQUENCE_ALIGNMENT)return;
	
	//check files have extensions
    bool tmp = 1;
	for(unsigned int i=0; i<PARAMS.filenames1.size(); i++){
		if(check_ext(PARAMS.filenames1[i])==0){
            tmp = 1;
            for(unsigned int j=0; j<i; j++){
                if(PARAMS.filenames1[i]+".pdb"==PARAMS.filenames1[j]){
                    tmp = 0;
                    break;
                }
            }
			xml_entry(PARAMS.error_file,0,2);
            if(tmp)cout << "Warning: Input PDB file " << PARAMS.filenames1[i] << " has no extension - .pdb will be assumed." << endl;
			PARAMS.filenames1[i] += ".pdb";
		}
	}
	for(unsigned int i=0; i<PARAMS.filenames2.size(); i++){
		if(check_ext(PARAMS.filenames2[i])==0){
            tmp = 1;
            for(unsigned int j=0; j<i; j++){
                if(PARAMS.filenames2[i]+".pdb"==PARAMS.filenames2[j]){
                    tmp = 0;
                    break;
                }
            }
            if(tmp)for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                if(PARAMS.filenames2[i]+".pdb"==PARAMS.filenames1[j]){
                    tmp = 0;
                    break;
                }
            }
			xml_entry(PARAMS.error_file,0,2);
			if(tmp)cout << "Warning: Input PDB file " << PARAMS.filenames2[i] << " has no extension - .pdb will be assumed." << endl;
			PARAMS.filenames2[i] += ".pdb";
		}
	}
    
    //enforce limitations on fragment length
    int FRAG_LEN_INT = atoi(PARAMS.FRAG_LEN.c_str());
    if(FRAG_LEN_INT > PARAMS.MAX_FRAG_LEN || FRAG_LEN_INT < 3){
		xml_entry(PARAMS.error_file,0,0,"-len");
        cout << endl << "Error - fragment length must be between 3 and " << PARAMS.MAX_FRAG_LEN << "." << endl << "Program terminated." << endl;
		exit(-1);
    }
    
    //enforce limitations on sphere size
    if(atof(PARAMS.sphere.c_str()) < 2.5){
		xml_entry(PARAMS.error_file,0,0,"-rmax");
        cout << endl << "Error - rmax must be greater than or equal to 2.5." << endl << "Program terminated." << endl;
		exit(-1);
    }
    
    //default is to do both align and restrain.
    if(PARAMS.ALIGN == 0 && PARAMS.RESTRAIN == 0){
        PARAMS.ALIGN = 1;
		PARAMS.RESTRAIN = 1;
    }
    
    //do not align if generating generic self-restraints
    if(PARAMS.GENERIC == 1){
        PARAMS.ALIGN = 0;
        PARAMS.RESTRAIN = 1;
        //PARAMS.GET_BONDS = "1";
        //PARAMS.MAINCHAIN_ONLY = "0";
    }
    
    //must specify -p1
    if(PARAMS.filenames1.size()==0){
        xml_entry(PARAMS.error_file,0,2);
        cout << endl << "Error - pdb1 not specified. Program terminated." << endl;
		exit(-1);
    }	
	
    return;	
}

bool get_binaries(params &PARAMS)
{
	//Get locations of binaries and library
	vector<string> bin_dir_idx;    
    string PKG_ROOT = "";
    string PKG_BIN = "";
    string PKG_LIB = "";
#ifdef PACKAGE_ROOT
    PKG_ROOT = PACKAGE_ROOT;
#endif
#ifdef PACKAGE_ROOT_BIN
    PKG_BIN = PACKAGE_ROOT_BIN;
#endif
#ifdef PACKAGE_ROOT_LIB
    PKG_LIB = PACKAGE_ROOT_LIB;
#endif
    get_bindir_libdir(PARAMS.error_file,PARAMS.BIN_DIR,PARAMS.LIB_DIR,PKG_ROOT,PKG_BIN,PKG_LIB);
    
    //check that BIN_DIR and LIB_DIR are defined.
    if(PARAMS.BIN_DIR.size()==0 || PARAMS.LIB_DIR.size()==0){
        xml_entry(PARAMS.error_file,0,6);
		cout << endl << "Error - required environment variable does not exist, or incorrect setup." << endl
        << "Potential solutions:" << endl << endl
        << "(1) If CCP4 is installed - ensure that CCP4 is setup, and ensure the presence of the $CCP4 environment variable." << endl << endl
        << "(2) If CCP4 is not installed - manually create and export the environment variable $PROSMART_DIR, which specifies the path of the ProSMART directory." << endl << endl;
		return 0;
    }
    
    PARAMS.BIN_DIR = make_valid_directory(PARAMS.error_file,PARAMS.BIN_DIR);
	PARAMS.LIB_DIR = make_valid_directory(PARAMS.error_file,PARAMS.LIB_DIR);
    
    //get contents of directories
	bin_dir_idx = read_dir(PARAMS.error_file,PARAMS.BIN_DIR);    //only needed on unix version...
    
	//Confirm binaries exist, etc.  
    //check contents of directories
    string align_exe = ALIGN_EXEC;
    string restrain_exe = RESTRAIN_EXEC;
	cout << endl;    
    if(PARAMS.ALIGN==1){
        check_dir(PARAMS.error_file,bin_dir_idx,align_exe,PARAMS.BIN_DIR);	//check that prosmart_align is present
        PARAMS.align_location = PARAMS.BIN_DIR + align_exe;
        cout << "ProSMART ALIGN location:   \t" << PARAMS.BIN_DIR << endl;
    }
    if(PARAMS.RESTRAIN==1){
        check_dir(PARAMS.error_file,bin_dir_idx,restrain_exe,PARAMS.BIN_DIR);	//check that prosmart_restrain is present
        PARAMS.restrain_location = PARAMS.BIN_DIR + restrain_exe;
		cout << "ProSMART RESTRAIN location:\t" << PARAMS.BIN_DIR << endl;
    }
	cout << "ProSMART Library location: \t" << PARAMS.LIB_DIR << endl;
 
	return 1;
}

bool get_filenames(params &PARAMS)
{
	vector<string> filenames3;
	vector<string> chains3;
    bool VALID;
	
    //allow specification of multiple chains from 1 file without having to repeat the filename.
    if(PARAMS.filenames1.size()==1 && PARAMS.chains1.size()>1){		    
        for(unsigned int i=1; i<PARAMS.chains1.size(); i++){
			PARAMS.filenames1.push_back(PARAMS.filenames1[0]);
		}
    }
    if(PARAMS.filenames2.size()==1 && PARAMS.chains2.size()>1){		
		for(unsigned int i=1; i<PARAMS.chains2.size(); i++){
			PARAMS.filenames2.push_back(PARAMS.filenames2[0]);
		}
    }
    
    //number of files must match number of chains.
    if(PARAMS.filenames1.size()!=PARAMS.chains1.size() && PARAMS.chains1.size()!=0){
        xml_entry(PARAMS.error_file,0,3);
		cout << endl << "Error - number of pdbs doesn't match number of chains. Program terminated." << endl;
		return 0;
    }
    if(PARAMS.filenames2.size()!=PARAMS.chains2.size() && PARAMS.chains2.size()!=0){
        xml_entry(PARAMS.error_file,0,3);
		cout << endl << "Error - number of pdbs doesn't match number of chains. Program terminated." << endl;
		return 0;
    }

	//check files exist
    //check that there are no spaces in the file names - if there are, then copy files with underscores.
    validate_pdb_file(PARAMS.error_file,PARAMS.filenames1,PARAMS.INPUTCHAINS_dir);
    if(PARAMS.filenames2.size()>0){
        validate_pdb_file(PARAMS.error_file,PARAMS.filenames2,PARAMS.INPUTCHAINS_dir);
    }

    // get chain IDs, if chains have not been specified.
    vector<string> tmp_chains;
    vector<string> tmp_filenames;
    if(PARAMS.chains1.size() == 0){
        tmp_filenames.clear();
        for(unsigned int i=0; i<PARAMS.filenames1.size(); i++){
            tmp_chains = get_chains(PARAMS.error_file,PARAMS.filenames1[i],PARAMS.GENERIC);
			for(unsigned int j=0; j<tmp_chains.size(); j++){
				PARAMS.chains1.push_back(tmp_chains[j]);
				tmp_filenames.push_back(PARAMS.filenames1[i]);
			}
		}
		PARAMS.filenames1 = tmp_filenames;
    }
    if(PARAMS.chains2.size() == 0 && !PARAMS.HBOND_RESTRAINTS){
        tmp_filenames.clear();
        for(unsigned int i=0; i<PARAMS.filenames2.size(); i++){
            tmp_chains = get_chains(PARAMS.error_file,PARAMS.filenames2[i],PARAMS.GENERIC);
			for(unsigned int j=0; j<tmp_chains.size(); j++){
				PARAMS.chains2.push_back(tmp_chains[j]);
				tmp_filenames.push_back(PARAMS.filenames2[i]);
			}
		}
		PARAMS.filenames2 = tmp_filenames;
    }	

	if(PARAMS.HELIX=="1"){										//fragment type id
		//read library config file
        string library = make_valid_directory(PARAMS.error_file,PARAMS.LIB_DIR);
		string filename;
		if(PARAMS.library_config.size()==0){
            filename = library + "config.txt";
		} else {
			filename = PARAMS.library_config;
		}
        
		if(check_file_exists(filename)==0){
			xml_entry(PARAMS.error_file,0,13);
			cout << endl << "Error - cannot find/open library configuration file: " << filename << ". Program terminated." << endl;
			return 0;
		}
		PARAMS.config = read_file_lines(filename);
		for(unsigned int i=0; i<PARAMS.config.size(); i++){
			if(PARAMS.config[i].size()!=8){
				PARAMS.config.erase(PARAMS.config.begin()+i);
				i--;
				continue;
			}
			if(PARAMS.HELIX_ONLY=="1" && PARAMS.config[i][0]=="helix"){continue;}
			if(PARAMS.STRAND_ONLY=="1" && PARAMS.config[i][0]=="strand"){continue;}
			if(PARAMS.HELIX_ONLY=="1" || PARAMS.STRAND_ONLY=="1"){
				PARAMS.config.erase(PARAMS.config.begin()+i);
				i--;
			}
		}
		for(unsigned int i=0; i<PARAMS.config.size(); i++){
			filename = library + PARAMS.config[i][0] + ".pdb";
			filenames3.push_back(filename);
			chains3.push_back(PARAMS.config[i][1]);
			if(PARAMS.OVERRIDE_LIB_SCORE.size()>0){
				PARAMS.lib_score.push_back(PARAMS.OVERRIDE_LIB_SCORE);
			} else {
				PARAMS.lib_score.push_back(PARAMS.config[i][3]);
			}
			if(PARAMS.OVERRIDE_LIB_FRAGLEN.size()>0){
				PARAMS.lib_fraglen.push_back(PARAMS.OVERRIDE_LIB_FRAGLEN);
			} else {
				PARAMS.lib_fraglen.push_back(PARAMS.config[i][4]);
			}
		}
		PARAMS.filenames2 = filenames3;
		PARAMS.chains2 = chains3;
    } else if(PARAMS.filenames2.size() == 0){
        if(PARAMS.filenames1.size()==1 && PARAMS.chains1.size()==1){		//one-on-all alignment
            tmp_chains = get_chains(PARAMS.error_file,PARAMS.filenames1[0],PARAMS.GENERIC);
			for(unsigned int j=0; j<tmp_chains.size(); j++){
				if(PARAMS.chains1[0]!=tmp_chains[j]){
                    PARAMS.filenames2.push_back(PARAMS.filenames1[0]);
					PARAMS.chains2.push_back(tmp_chains[j]);
				}
			}
			if(PARAMS.filenames2.size()==0){		//cannot have all-on-all alignment with only 1 chain
				PARAMS.filenames2 = PARAMS.filenames1;
				PARAMS.chains2 = PARAMS.chains1;
			}
        } else {											//all-on-all alignment
			PARAMS.ALL_ON_ALL = 1;
			PARAMS.filenames2 = PARAMS.filenames1;
			PARAMS.chains2 = PARAMS.chains1;
		}
    }
    
    //validate -align
    if(PARAMS.ALIGN_PDB.size()>0){
        for(unsigned int i=0; i<PARAMS.ALIGN_PDB.size(); i++){
            if(PARAMS.ALIGN_PDB[i] == "ALL"){
                continue;
            }
            PARAMS.ALIGN_PDB[i] = replace_spaces(PARAMS.ALIGN_PDB[i],"_");
            VALID = 0;
            for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                if(get_full_filename(PARAMS.filenames1[j]) == PARAMS.ALIGN_PDB[i]){
                    VALID = 1;
                    break;
                }
            }
            for(unsigned int j=0; j<PARAMS.filenames2.size(); j++){
                if(get_full_filename(PARAMS.filenames2[j]) == PARAMS.ALIGN_PDB[i]){
                    VALID = 1;
                    break;
                }
            }
            if(VALID==0){
                xml_entry(PARAMS.error_file,0,1);
                cout << endl << "Invalid Syntax: invalid specification of filename argument for -align:  " << PARAMS.ALIGN_PDB[i];
                cout << endl << "Possible options are:";
                for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                    VALID = 1;
                    for(unsigned int k=0; k<j; k++){
                        if(get_full_filename(PARAMS.filenames1[j])==get_full_filename(PARAMS.filenames1[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    if(VALID==1){
                        cout << endl << "\t" << get_full_filename(PARAMS.filenames1[j]);
                    }
                }
                for(unsigned int j=0; j<PARAMS.filenames2.size(); j++){
                    VALID = 1;
                    for(unsigned int k=0; k<j; k++){
                        if(get_full_filename(PARAMS.filenames2[j])==get_full_filename(PARAMS.filenames2[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    for(unsigned int k=0; k<PARAMS.filenames1.size(); k++){
                        if(get_full_filename(PARAMS.filenames2[j])==get_full_filename(PARAMS.filenames1[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    if(VALID==1){
                        cout << endl << "\t" << get_full_filename(PARAMS.filenames2[j]);
                    }
                }
                cout << endl << "Program terminated." << endl;
               return 0;
            }
        }
    }
    //validate -align_rm
    if(PARAMS.ALIGN_RM_PDB.size()>0){
        for(unsigned int i=0; i<PARAMS.ALIGN_RM_PDB.size(); i++){
            if(PARAMS.ALIGN_RM_PDB[i] == "ALL"){
                continue;
            }
            PARAMS.ALIGN_RM_PDB[i] = replace_spaces(PARAMS.ALIGN_RM_PDB[i],"_");
            VALID = 0;
            for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                if(get_full_filename(PARAMS.filenames1[j]) == PARAMS.ALIGN_RM_PDB[i]){
                    VALID = 1;
                    break;
                }
            }
            for(unsigned int j=0; j<PARAMS.filenames2.size(); j++){
                if(get_full_filename(PARAMS.filenames2[j]) == PARAMS.ALIGN_RM_PDB[i]){
                    VALID = 1;
                    break;
                }
            }
            if(VALID==0){
                xml_entry(PARAMS.error_file,0,1);
                cout << endl << "Invalid Syntax: invalid specification of filename argument for -align_rm:  " << PARAMS.ALIGN_RM_PDB[i];
                cout << endl << "Possible options are:";
                for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                    VALID = 1;
                    for(unsigned int k=0; k<j; k++){
                        if(get_full_filename(PARAMS.filenames1[j])==get_full_filename(PARAMS.filenames1[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    if(VALID==1){
                        cout << endl << "\t" << get_full_filename(PARAMS.filenames1[j]);
                    }
                }
                for(unsigned int j=0; j<PARAMS.filenames2.size(); j++){
                    VALID = 1;
                    for(unsigned int k=0; k<j; k++){
                        if(get_full_filename(PARAMS.filenames2[j])==get_full_filename(PARAMS.filenames2[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    for(unsigned int k=0; k<PARAMS.filenames1.size(); k++){
                        if(get_full_filename(PARAMS.filenames2[j])==get_full_filename(PARAMS.filenames1[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    if(VALID==1){
                        cout << endl << "\t" << get_full_filename(PARAMS.filenames2[j]);
                    }
                }
                cout << endl << "Program terminated." << endl;
               return 0;
            }
        }
    }
    //validate -restrain
    if(PARAMS.RESTRAIN_PDB.size()>0){
        for(unsigned int i=0; i<PARAMS.RESTRAIN_PDB.size(); i++){
            if(PARAMS.RESTRAIN_PDB[i] == "ALL"){
                continue;
            }
            PARAMS.RESTRAIN_PDB[i] = replace_spaces(PARAMS.RESTRAIN_PDB[i],"_");
            VALID = 0;
            for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                if(get_full_filename(PARAMS.filenames1[j]) == PARAMS.RESTRAIN_PDB[i]){
                    VALID = 1;
                    break;
                }
            }
            if(VALID==0){
                xml_entry(PARAMS.error_file,0,1);
                cout << endl << "Invalid Syntax: invalid specification of filename argument for -restrain:  " << PARAMS.RESTRAIN_PDB[i];
                cout << endl << "Possible options are:";
                for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                    VALID = 1;
                    for(unsigned int k=0; k<j; k++){
                        if(get_full_filename(PARAMS.filenames1[j])==get_full_filename(PARAMS.filenames1[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    if(VALID==1){
                        cout << endl << "\t" << get_full_filename(PARAMS.filenames1[j]);
                    }
                }
                cout << endl << "Program terminated." << endl;
                return 0;
            }
        }
    }
    //validate -restrain_rm
    if(PARAMS.RESTRAIN_RM_PDB.size()>0){
        for(unsigned int i=0; i<PARAMS.RESTRAIN_RM_PDB.size(); i++){
            if(PARAMS.RESTRAIN_RM_PDB[i] == "ALL"){
                continue;
            }
            PARAMS.RESTRAIN_RM_PDB[i] = replace_spaces(PARAMS.RESTRAIN_RM_PDB[i],"_");
            VALID = 0;
            
            for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                if(get_full_filename(PARAMS.filenames1[j]) == PARAMS.RESTRAIN_RM_PDB[i]){
                    VALID = 1;
                    break;
                }
            }
            if(VALID==0){
                xml_entry(PARAMS.error_file,0,1);
                cout << endl << "Invalid Syntax: invalid specification of filename argument for -restrain_rm:  " << PARAMS.RESTRAIN_RM_PDB[i];
                cout << endl << "Possible options are:";
                for(unsigned int j=0; j<PARAMS.filenames1.size(); j++){
                    VALID = 1;
                    for(unsigned int k=0; k<j; k++){
                        if(get_full_filename(PARAMS.filenames1[j])==get_full_filename(PARAMS.filenames1[k])){
                            VALID = 0;
                            break;
                        }
                    }
                    if(VALID==1){
                        cout << endl << "\t" << get_full_filename(PARAMS.filenames1[j]);
                    }
                }
                cout << endl << "Program terminated." << endl;
                return 0;
            }
        }
    }
   
   if(PARAMS.HBOND_RESTRAINTS){
      PARAMS.filenames2 = PARAMS.filenames1;
      PARAMS.chains2 = PARAMS.chains1;
      if(USE_DNARNA){
         cout << endl << "Error - cannot generate h-bond type restraints for DNA/RNA." << endl;
         cout << endl << "Program terminated." << endl;
         return 0;
      }
   }
	
	return 1;
}

bool combine_chains(params &PARAMS)
{
    combine_chains(PARAMS.filenames1,PARAMS.chains1,PARAMS.INPUTCHAINS_dir,"1");
    if(PARAMS.ALL_ON_ALL){
        if(PARAMS.filenames1.size()==1){
            PARAMS.ALL_ON_ALL = 0;  //cannot have all-on-all comparison with only one chain.
        }
        PARAMS.filenames2 = PARAMS.filenames1;
        PARAMS.chains2 = PARAMS.chains1;
    } else {
        combine_chains(PARAMS.filenames2,PARAMS.chains2,PARAMS.INPUTCHAINS_dir,"2");
    }
    return 1;
}

void combine_chains(vector<string> &filenames1, vector<string> &chains1, string &input_dir, string chainID)
{
    vector<string> filenames;
    vector<string> empty;
    vector<vector<string> > chains;
    for(unsigned int i=0; i<filenames1.size(); i++){
        bool file_got=0;
        for(unsigned int j=0; j<filenames.size(); j++){
            if(filenames1[i]==filenames[j]){
                file_got=1;
                chains[j].push_back(chains1[i]);
                break;
            }
        }
        if(file_got)continue;
        filenames.push_back(filenames1[i]);
        chains.push_back(empty);
        chains.back().push_back(chains1[i]);
    }

    for(unsigned int i=0; i<filenames.size(); i++){
        if(chains[i].size()>1){
            cout << endl << "Merging chains: ";
            for(unsigned int j=0; j<chains[i].size(); j++){
                cout << chains[i][j] << " ";
            }
            cout << " in PDB file: " << filenames[i] << endl;
        }
    }

    filenames1.clear();
    chains1.clear();
    
    /*for(unsigned int i=0; i<filenames.size(); i++){
        filenames1.push_back(filenames[i]);
        chains1.push_back("X");
    }*/
    
    for(unsigned int i=0; i<filenames.size(); i++){
        if(chains[i].size()==1){
            filenames1.push_back(filenames[i]);
            chains1.push_back(chains[i][0]);
            continue;
        } else {
            filenames1.push_back(input_dir + get_filename(filenames[i]) + chainID[0] + ".pdb");
            chains1.push_back(chainID);
        }
        string line;
        vector<vector<string> > lines;
        //vector<int> highest;
        //vector<int> lowest;
        for(unsigned int j=0; j<chains[i].size(); j++){
            lines.push_back(empty);
            //highest.push_back(0);
            //lowest.push_back(0);
        }

        ifstream infile(filenames[i].c_str(), ios::in);
        if(infile.is_open()){
            while(!infile.eof()){
                line.clear();
                getline(infile,line);
                
                if(line.size()<66)continue;
                
                if(line.substr(0,6)!="ATOM  "){
                    if(line.substr(0,6)!="HETATM"){
                        continue;
                    }
                }
                
                for(unsigned int j=0; j<chains[i].size(); j++){
                    if(chains[i][j][0]==line[21]){
                        line[21] = chainID[0];
                        lines[j].push_back(line);
                    }
                }                
            }
            infile.close();
        }
        
        int current = 0;
        for(unsigned int j=0; j<lines.size(); j++){
            int current_orig = atoi(lines[j][0].substr(22,4).c_str());
            current++;
            for(unsigned int k=0; k<lines[j].size(); k++){
                int resnum = atoi(lines[j][k].substr(22,4).c_str());
                if(resnum!=current_orig){
                    if(resnum==current_orig+1){         //consecutive residue
                        current++;
                    } else {                            //gap
                        current+=2;
                    }
                    current_orig = resnum;
                }
                string str = int_to_str(current);
                while(str.size()<4)str.insert(str.begin(),' '); //fill out with spaces
                while(str.size()>4)str.erase(str.begin());      //sanity check
                for(unsigned int l=0; l<4; l++){
                    lines[j][k][22+l] = str[l];
                }
            }
        }
        
        ofstream outfile;
        outfile.open(filenames1[i].c_str());
        if(outfile.is_open()){
            for(unsigned int j=0; j<lines.size(); j++){
                for(unsigned int k=0; k<lines[j].size(); k++){
                    outfile << lines[j][k] << endl;
                }
            }
            outfile.close();
        }
    }
    
    return;
}

bool param_validation2(params &PARAMS)
{
	//must have input files and chains specified.
    if(PARAMS.filenames1.size()==0 || PARAMS.chains1.size()==0 || PARAMS.filenames2.size()==0 || PARAMS.chains2.size()==0){
        xml_entry(PARAMS.error_file,0,2);
        cout << endl << "Error - input pdb not correctly specified - could not extract valid chains from input file." << endl;
       cout << endl << "Notes:";
       cout << endl << "Valid chains must contain sufficient consecutive valid residues.";
       cout << endl << "Valid residues must minimally contain all backbone atoms.";
       cout << endl << "To use DNA/RNA instead of protein macromolecules, please specify the -dna_rna keyword." << endl;
       cout << endl << "Program terminated." << endl;
		return 0;
    }
    
    //number of files must match number of chains.
    if((PARAMS.filenames1.size()!=PARAMS.chains1.size()) || (PARAMS.filenames2.size()!=PARAMS.chains2.size())){
		xml_entry(PARAMS.error_file,0,3);
        cout << endl << "Error - number of pdbs doesn't match number of chains. Program terminated." << endl;
		return 0;
    }
    
    //make sure chain numbers are not blank
    for(unsigned int i=0; i<PARAMS.chains1.size(); i++){
        if(PARAMS.chains1[i][0] == ' '){
			xml_entry(PARAMS.error_file,0,4);
            cout << endl << "Warning - chain ID(s) not specified in file: " << PARAMS.filenames1[i] << endl;
            cout << "Possible cause: invalid PDB file. This file will be removed. Continuing anyway." << endl;
            cout << "If chain ID is not present in PDB file, then use -renamechain." << endl;
            cout << "See user documentation for details of usage of -renamechain." << endl << endl;
            PARAMS.filenames1.erase(PARAMS.filenames1.begin()+i);
            PARAMS.chains1.erase(PARAMS.chains1.begin()+i);
            i--;
		}
    }
    for(unsigned int i=0; i<PARAMS.chains2.size(); i++){
        if(PARAMS.chains2[i][0] == ' '){
            xml_entry(PARAMS.error_file,0,4);
            cout << endl << "Warning - chain ID(s) not specified in file: " << PARAMS.filenames2[i] << endl;
            cout << "Possible cause: invalid PDB file. This file will be removed. Continuing anyway." << endl;
            cout << "If chain ID is not present in PDB file, then use -renamechain." << endl;
            cout << "See user documentation for details of usage of -renamechain." << endl << endl;
            PARAMS.filenames2.erase(PARAMS.filenames2.begin()+i);
            PARAMS.chains2.erase(PARAMS.chains2.begin()+i);
            i--;
        }
    }
    
#ifdef RUN_AREAIMOL
    vector<string> unique_filenames;
    for(unsigned int i=0; i<PARAMS.filenames1.size(); i++){
        bool file_got = 0;
        for(unsigned int j=0; j<unique_filenames.size(); j++){    //do this if want to combine all chains
            if(PARAMS.filenames1[i]==unique_filenames[j]){
                file_got = 1;
                break;
            }
        }
        if(!file_got)unique_filenames.push_back(PARAMS.filenames1[i]);
    }
    for(unsigned int i=0; i<PARAMS.filenames2.size(); i++){
        bool file_got = 0;
        for(unsigned int j=0; j<unique_filenames.size(); j++){    //do this if want to combine all chains
            if(PARAMS.filenames2[i]==unique_filenames[j]){
                file_got = 1;
                break;
            }
        }
        if(!file_got)unique_filenames.push_back(PARAMS.filenames2[i]);
    }
    for(unsigned int i=0; i<unique_filenames.size(); i++){
        string file_asa = PARAMS.INPUTCHAINS_dir + get_filename(unique_filenames[i]) + "_asa.pdb";
        if(!check_file_exists(file_asa)){
            run_areaimol(unique_filenames[i],file_asa);
            if(!check_file_exists(file_asa)){
                cout << endl << "Error - AREAIMOL output file \"" << file_asa << "\" not found. Program terminated." << endl;
                return 0;
            }
        }
    }
#endif

    string input_dir = PARAMS.output_dir+"Output_Files/";
    bool valid = 0;
    if(!PARAMS.GENERIC){
        //ensure that at least 1 fragment exists in each file
        //also output recoded PDB files, including information about residue renumbering
        int FRAG_LEN_INT = atoi(PARAMS.FRAG_LEN.c_str());
        vector<int> RANGE1;
        vector<int> RANGE2;
        vector<string> RM_RES;
        for(unsigned int i=0; i<PARAMS.chains1.size(); i++){
            // determine whether specific alignment ranges have been specified, for this particular PDB/chain
            RANGE1.clear();
            RANGE2.clear();
            for(unsigned int k=0; k<PARAMS.ALIGN_PDB.size(); k++){
                if(PARAMS.ALIGN_PDB[k]=="ALL" || PARAMS.ALIGN_PDB[k]==get_full_filename(PARAMS.filenames1[i])){
                    if(PARAMS.ALIGN_CHAIN[k]=="ALL"){
                        RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                        RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                    } else {
                        for(unsigned int l=0; l<PARAMS.ALIGN_CHAIN[k].size(); l++){
                            if(PARAMS.ALIGN_CHAIN[k][l]==PARAMS.chains1[i][0]){
                                RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                                RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                            }
                        }
                    }
                }
            }
            
            RM_RES.clear();
            for(unsigned int k=0; k<PARAMS.ALIGN_RM_PDB.size(); k++){
                if(PARAMS.ALIGN_RM_PDB[k]=="ALL" || PARAMS.ALIGN_RM_PDB[k]==get_full_filename(PARAMS.filenames1[i])){
                    if(PARAMS.ALIGN_RM_CHAIN[k]=="ALL"){
                        RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                    } else {
                        for(unsigned int l=0; l<PARAMS.ALIGN_RM_CHAIN[k].size(); l++){
                            if(PARAMS.ALIGN_RM_CHAIN[k][l]==PARAMS.chains1[i][0]){
                                RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                            }
                        }
                    }
                }
            }

            //this outputs re-formatted files in .Input_Chains/
            valid = validate_files(input_dir,PARAMS.filenames1[i],PARAMS.chains1[i][0],FRAG_LEN_INT,0,RANGE1,RANGE2,RM_RES,0,PARAMS.OCCUPANCY_THRESHOLD);

            if(valid == 0){
                cout << endl << "Warning - cannot find sufficient residues in: " << PARAMS.filenames1[i] << " chain " << PARAMS.chains1[i];
                cout << endl << "This chain will be ignored." << endl;
                PARAMS.filenames1.erase(PARAMS.filenames1.begin()+i);
                PARAMS.chains1.erase(PARAMS.chains1.begin()+i);
                if(PARAMS.ALL_ON_ALL==1){
                    PARAMS.filenames2.erase(PARAMS.filenames2.begin()+i);
                    PARAMS.chains2.erase(PARAMS.chains2.begin()+i);
                }
                i--;
            }
        }

        if(PARAMS.ALL_ON_ALL==0){
            bool USE_SIGMAS = 0;
            if(PARAMS.GET_ATOMIC_UNCERTAINTY == "1"){
                vector<pid_t> pIDs;
                for(unsigned int i=0; i<PARAMS.chains2.size(); i++){
                    string bonds_filename = PARAMS.filenames2[i];
                    bool bonds_exists = 0;
                    for(unsigned int j=0; j<i; j++){
                        if(bonds_filename == PARAMS.filenames2[j]){
                            bonds_exists = 1;
                            break;
                        }
                    }
                    if(bonds_exists == 0){
                        bonds_filename = PARAMS.INPUTCHAINS_dir+get_filename(PARAMS.filenames2[i])+"_sigmas.txt";
                        bonds_exists = check_file_exists(bonds_filename);	//check whether file exists
                        if(bonds_exists==0){
                            string refmac_logfile = PARAMS.output_dir + "refmac5_dpiout_logfile.html";
                            string mtz_file = delete_ext(PARAMS.filenames2[i]) + ".mtz";
                            get_atomic_uncertainty(PARAMS.error_file,bonds_filename,PARAMS.filenames2[i],mtz_file,pIDs,PARAMS.MAX_THREADS,PARAMS.REFMAC_NAME,refmac_logfile);   
                        }
                    }
                }
                int status = 0;
                for(unsigned int j=0; j<pIDs.size(); j++)waitpid(pIDs[j],&status,0); //wait for refmac
                USE_SIGMAS = 1;
            }

            for(unsigned int i=0; i<PARAMS.chains2.size(); i++){
                // determine whether specific alignment ranges have been specified, for this particular PDB/chain
                RANGE1.clear();
                RANGE2.clear();
                for(unsigned int k=0; k<PARAMS.ALIGN_PDB.size(); k++){
                    if(PARAMS.ALIGN_PDB[k]=="ALL" || PARAMS.ALIGN_PDB[k]==get_full_filename(PARAMS.filenames2[i])){
                        if(PARAMS.ALIGN_CHAIN[k]=="ALL"){
                            RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                            RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                        } else {
                            for(unsigned int l=0; l<PARAMS.ALIGN_CHAIN[k].size(); l++){
                                if(PARAMS.ALIGN_CHAIN[k][l]==PARAMS.chains2[i][0]){
                                    RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                                    RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                                }
                            }
                        }
                    }
                }
                
                RM_RES.clear();
                for(unsigned int k=0; k<PARAMS.ALIGN_RM_PDB.size(); k++){
                    if(PARAMS.ALIGN_RM_PDB[k]=="ALL" || PARAMS.ALIGN_RM_PDB[k]==get_full_filename(PARAMS.filenames2[i])){
                        if(PARAMS.ALIGN_RM_CHAIN[k]=="ALL"){
                            RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                        } else {
                            for(unsigned int l=0; l<PARAMS.ALIGN_RM_CHAIN[k].size(); l++){
                                if(PARAMS.ALIGN_RM_CHAIN[k][l]==PARAMS.chains2[i][0]){
                                    RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                                }
                            }
                        }
                    }
                }
                                
                valid = validate_files(input_dir,PARAMS.filenames2[i],PARAMS.chains2[i][0],FRAG_LEN_INT,(bool)str_to_int(PARAMS.HELIX),RANGE1,RANGE2,RM_RES,USE_SIGMAS,PARAMS.OCCUPANCY_THRESHOLD);
                
                if(valid == 0){
                    cout << endl << "Warning - cannot find sufficient residues in: " << PARAMS.filenames2[i] << " chain " << PARAMS.chains2[i];
                    cout << endl << "This chain will be ignored." << endl;
                    PARAMS.filenames2.erase(PARAMS.filenames2.begin()+i);
                    PARAMS.chains2.erase(PARAMS.chains2.begin()+i);
                    i--;
                }
            }
        }
        
    } else {
        //generic self-restraints
        PARAMS.filenames2 = PARAMS.filenames1;
        PARAMS.chains2 = PARAMS.chains1;
        
        //copy PDB file(s) to .Input_Chains/ so that ProSMART_RESTRAIN can access them.
        for(unsigned int i=0; i<PARAMS.filenames1.size(); i++){
            valid = 1;
            for(unsigned int j=0; j<i; j++){
                if(PARAMS.filenames1[i]==PARAMS.filenames1[j]){
                    valid = 0;  //dont copy same PDB file more than once.
                    break;
                }
            }
            if(valid==0)continue;
            stream_file_copy(PARAMS.error_file,PARAMS.filenames1[i],(input_dir + ".Input_Chains/" + get_filename(PARAMS.filenames1[i])));
        }
    }
	
	//must have input files and chains specified.
    if(PARAMS.filenames1.size()==0 || PARAMS.filenames2.size()==0){
        xml_entry(PARAMS.error_file,0,2);
       cout << endl << "Error - insufficient valid input chains\n";
       cout << "Note - to use DNA/RNA chains, use the keyword: -dna_rna\n";
        cout << "Program terminated." << endl;
		return 0;
    }
	if(PARAMS.ALL_ON_ALL==1 && PARAMS.filenames1.size()==1){
        xml_entry(PARAMS.error_file,0,2);
      cout << endl << "Error - insufficient valid input chains\n";
      cout << "Note - to use DNA/RNA chains, use the keyword: -dna_rna\n";
      cout << "Program terminated." << endl;
		return 0;
    }
	
	//if there is only one chain in target and external, then self-restraining is allowed
	if(PARAMS.filenames1.size()==1 && PARAMS.filenames2.size()==1){
		PARAMS.RESTRAIN_TO_SELF = 1;
	}
	
	return 1;
}

bool check_for_replication(params &PARAMS)
{
    //construct list of unique chains
    vector<string> files;
    vector<string> chains;
    for(unsigned int i=0; i<PARAMS.filenames1.size(); i++){
        bool VALID = 1;
        for(unsigned int j=0; j<files.size(); j++){
            if(PARAMS.filenames1[i]==files[j] && PARAMS.chains1[i]==chains[j]){
                VALID = 0;
                break;
            }
        }
        if(VALID){
            files.push_back(PARAMS.filenames1[i]);
            chains.push_back(PARAMS.chains1[i]);
        }
    }
    for(unsigned int i=0; i<PARAMS.filenames2.size(); i++){
        bool VALID = 1;
        for(unsigned int j=0; j<files.size(); j++){
            if(PARAMS.filenames2[i]==files[j] && PARAMS.chains2[i]==chains[j]){
                VALID = 0;
                break;
            }
        }
        if(VALID){
            files.push_back(PARAMS.filenames2[i]);
            chains.push_back(PARAMS.chains2[i]);
        }
    }
    
    
    vector<string> args1;
    args1.push_back(PARAMS.align_location);
    args1.push_back("");                            //File 1
    args1.push_back("");                            //File 2
    args1.push_back("");                            //Chain identifier from file 1. Must be of length 1.
    args1.push_back("");                            //Chain identifier from file 2. Must be of length 1.
    args1.push_back(PARAMS.WORKING_DIRECTORY);
    args1.push_back(PARAMS.FRAG_LEN);				//Fragment length, in residues.
    args1.push_back(PARAMS.CLUSTER_DEGREES);
    args1.push_back("0");                           //Fragment alignment distance cutoff
    args1.push_back(PARAMS.CLUSTER);				//indicates whether or not to perform clustering.
    args1.push_back(PARAMS.LIB_DIR);
    args1.push_back(PARAMS.ALIGN_SCORE);			//score that defines similar residues.
    args1.push_back(PARAMS.ALIGN_MODE);				//use ca's or all mainchain atoms.
    args1.push_back("0");	//do not refine alignment.
    args1.push_back(PARAMS.SIDE_SCORE);
    args1.push_back("0");   //HELIX == 0
    args1.push_back("0");   //IDENTICAL == 0
    args1.push_back(PARAMS.HELIX_CUTOFF);
    args1.push_back(PARAMS.HELIX_PENALTY);
    args1.push_back(PARAMS.CLUSTER_MIN);
    args1.push_back(PARAMS.CLUSTER_LINKAGE);
    args1.push_back(PARAMS.CLUSTER_RIGIDITY);
    args1.push_back(PARAMS.CLUSTER_COLOR);
    args1.push_back(PARAMS.G1);
    args1.push_back(PARAMS.G2);
    args1.push_back(PARAMS.G3);
    args1.push_back(PARAMS.R1);
    args1.push_back(PARAMS.R2);
    args1.push_back(PARAMS.R3);
    args1.push_back(PARAMS.OUT_PDB);
    args1.push_back(PARAMS.OUT_COLOR);
    args1.push_back(PARAMS.SUPERPOSE);
    args1.push_back(PARAMS.CLUSTER_SCORE);
    args1.push_back(PARAMS.OUTPUT_CLUSTER_DM);
    args1.push_back("-");    //RANGE1 == "-"
    args1.push_back(PARAMS.SCORE_THRESHOLD);
    args1.push_back(PARAMS.DISPLAY_AS_COSINE_DISTANCE);
    args1.push_back(PARAMS.OUT_PDB_FULL);
    args1.push_back(PARAMS.USE_MAINCHAIN_FOR_SIDERMSD);
    args1.push_back(PARAMS.NUM_DIST_THRESHOLD);
    args1.push_back("1");   //search for internal replication
    args1.push_back(PARAMS.FIX_SIDE_CHAIN_ERRORS);
    args1.push_back(PARAMS.NO_REWARD_SEQ);
    args1.push_back(PARAMS.REWARD_SEQ);
    args1.push_back("");


    vector<string> sys_string;
    vector<string> message;
    for(unsigned int i=0; i<files.size(); i++){
        //make subdirectories
        string tmp = get_filename(files[i])+"_"+chains[i];
        string tmp2 = tmp+"_"+tmp;
		string sub_dir = make_subdirectory(PARAMS.error_file,PARAMS.ALIGN_dir,tmp);
		sub_dir = make_subdirectory(PARAMS.error_file,PARAMS.TRANS_dir,tmp);
        if(PARAMS.OUT_PDB=="1"){
            sub_dir = make_subdirectory(PARAMS.error_file,PARAMS.PDB_dir,tmp2);
        }
        if(PARAMS.OUT_COLOR=="1"){
            sub_dir = make_subdirectory(PARAMS.error_file,PARAMS.PYMOL_dir,tmp2);
        }
        //get command strings
        args1[1] = files[i];
        args1[2] = files[i];
        args1[3] = chains[i];
        args1[4] = chains[i];
        args1[args1.size()-1] = "> \""+PARAMS.WORKING_DIRECTORY+"out"+int_to_str(i)+"\"";
        sys_string.push_back(vector_to_string(args1));
        message.push_back("Searching for internal replication in " + tmp);
    }
    
    vector<pid_t> pIDs;
    vector<pid_t> newpIDs;
    cout << endl;
    spawn_external(PARAMS.error_file,sys_string,message,pIDs,newpIDs,PARAMS.MAX_THREADS);	//launch prosmart align, and wait for all instances to terminate before continuing.
    
    return 1;
}

void print_parameters(params &PARAMS, string &outputdirectory)
{
	vector<int> RANGE1;
    vector<int> RANGE2;
    vector<string> RM_RES;
    cout << endl << endl << "ProSMART Launched with Arguments:" << endl;
    if(PARAMS.ALL_ON_ALL==1){cout << endl << "All-On-All";}
	if(PARAMS.filenames1.size() > 1){
		cout << endl << "Target Chains:\t\t\t";
	} else {
		cout << endl << "Target Chain:\t\t\t";
	}
    for(unsigned int i=0; i<PARAMS.filenames1.size(); i++){
        if(i!=0){
            cout << endl << "\t\t\t\t";
        }
        cout << PARAMS.filenames1[i] << "  " << PARAMS.chains1[i];
        //list custom residue ranges
        if(PARAMS.ALIGN_PDB.size()>0){
            RANGE1.clear();
            RANGE2.clear();
            for(unsigned int k=0; k<PARAMS.ALIGN_PDB.size(); k++){
                if(PARAMS.ALIGN_PDB[k]=="ALL" || PARAMS.ALIGN_PDB[k]==get_full_filename(PARAMS.filenames1[i])){
                    if(PARAMS.ALIGN_CHAIN[k]=="ALL"){
                        RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                        RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                    } else {
                        for(unsigned int l=0; l<PARAMS.ALIGN_CHAIN[k].size(); l++){
                            if(PARAMS.ALIGN_CHAIN[k][l]==PARAMS.chains1[i][0]){
                                RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                                RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                            }
                        }
                    }
                }
            }
            if(RANGE1.size()>0){
                cout << "  residue range:";
            }
            for(unsigned int i=0; i<RANGE1.size(); i++){
                cout << " [" << RANGE1[i] << "," << RANGE2[i] << "]";
            }
        }
        //list residue removed
        RM_RES.clear();
        if(PARAMS.ALIGN_RM_PDB.size()>0){
            for(unsigned int k=0; k<PARAMS.ALIGN_RM_PDB.size(); k++){
                if(PARAMS.ALIGN_RM_PDB[k]=="ALL" || PARAMS.ALIGN_RM_PDB[k]==get_full_filename(PARAMS.filenames1[i])){
                    if(PARAMS.ALIGN_RM_CHAIN[k]=="ALL"){
                        RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                    } else {
                        for(unsigned int l=0; l<PARAMS.ALIGN_RM_CHAIN[k].size(); l++){
                            if(PARAMS.ALIGN_RM_CHAIN[k][l]==PARAMS.chains1[i][0]){
                                RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                            }
                        }
                    }
                }
            }
            if(RM_RES.size()>0){
                cout << "  removing:";
            }
            for(unsigned int i=0; i<RM_RES.size(); i++){
                cout << " " << RM_RES[i];
            }
        }
        
    }
    if(PARAMS.ALL_ON_ALL==0){
		if(PARAMS.filenames2.size() > 1){
			cout << endl << endl << "Secondary Chains:\t\t";
		} else {
			cout << endl << endl << "Secondary Chain:\t\t";
		}
        for(unsigned int i=0; i<PARAMS.filenames2.size(); i++){
			if(i!=0){
                cout << endl << "\t\t\t\t";
            }
            cout << PARAMS.filenames2[i] << "  " << PARAMS.chains2[i];
            //list custom residue ranges
            if(PARAMS.ALIGN_PDB.size()>0){
                RANGE1.clear();
                RANGE2.clear();
                for(unsigned int k=0; k<PARAMS.ALIGN_PDB.size(); k++){
                    if(PARAMS.ALIGN_PDB[k]=="ALL" || PARAMS.ALIGN_PDB[k]==get_full_filename(PARAMS.filenames2[i])){
                        if(PARAMS.ALIGN_CHAIN[k]=="ALL"){
                            RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                            RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                        } else {
                            for(unsigned int l=0; l<PARAMS.ALIGN_CHAIN[k].size(); l++){
                                if(PARAMS.ALIGN_CHAIN[k][l]==PARAMS.chains2[i][0]){
                                    RANGE1.push_back(PARAMS.ALIGN_RANGE1[k]);
                                    RANGE2.push_back(PARAMS.ALIGN_RANGE2[k]);
                                }
                            }
                        }
                    }
                }
                if(RANGE1.size()>0){
                    cout << "  residue range:";
                }
                for(unsigned int i=0; i<RANGE1.size(); i++){
                    cout << " [" << RANGE1[i] << "," << RANGE2[i] << "]";
                }
            }
            //list residue removed
            RM_RES.clear();
            if(PARAMS.ALIGN_PDB.size()>0){
                for(unsigned int k=0; k<PARAMS.ALIGN_RM_PDB.size(); k++){
                    if(PARAMS.ALIGN_RM_PDB[k]=="ALL" || PARAMS.ALIGN_RM_PDB[k]==get_full_filename(PARAMS.filenames2[i])){
                        if(PARAMS.ALIGN_RM_CHAIN[k]=="ALL"){
                            RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                        } else {
                            for(unsigned int l=0; l<PARAMS.ALIGN_RM_CHAIN[k].size(); l++){
                                if(PARAMS.ALIGN_RM_CHAIN[k][l]==PARAMS.chains2[i][0]){
                                    RM_RES.push_back(PARAMS.ALIGN_RM_RESIDUE[k]);
                                }
                            }
                        }
                    }
                }
                if(RM_RES.size()>0){
                    cout << "  removing:";
                }
                for(unsigned int i=0; i<RM_RES.size(); i++){
                    cout << " " << RM_RES[i];
                }
            }
        }
    }
    
	cout << endl << endl << "Output Directory:\t\t" << outputdirectory;
	cout << endl << "Use Fragment Library:\t\t";
	if(PARAMS.HELIX=="1"){
		cout << "Yes";
	} else {
		cout << "No";
	}
	cout << endl << "Max Parallel Threads:\t\t" << PARAMS.MAX_THREADS;
	cout << endl << "Output XML File:\t\t";
	if(PARAMS.error_file.size()>0){
		cout << PARAMS.error_file;
	} else {
		cout << "No";
	}
	if(PARAMS.ALIGN ==1){
		cout << endl << "Similar Colour:   \t\t(" << PARAMS.G1 << ", " << PARAMS.G2 << ", " << PARAMS.G3 << ")";
		cout << endl << "Dissimilar Colour:\t\t(" << PARAMS.R1 << ", " << PARAMS.R2 << ", " << PARAMS.R3 << ")";
	}
	
	cout << endl << endl << "Align Chains:\t\t\t";
    if(PARAMS.ALIGN == 0){
        cout << "No";
    } else {
        cout << "Yes";
		cout << endl << "ALIGN Options:";
        
		cout << endl << "\tAlignment Mode:\t\t" << PARAMS.ALIGN_MODE;
		cout << endl << "\tOutput Superposed PDBs:\t";
		if(PARAMS.OUT_PDB == "1"){
			cout << "Yes";
		} else {
			cout << "No";
		}
		cout << endl << "\tOutput Colour Scripts:\t";
		if(PARAMS.OUT_COLOR == "1"){
			cout << "Yes";
		} else {
			cout << "No";
		}
        cout << endl << "\tProcrustes Cutoff:\t";
        if(PARAMS.SCORE_THRESHOLD == "1000.0"){
            cout << "No";
        } else {
            cout << PARAMS.SCORE_THRESHOLD;
        }
		if(PARAMS.HELIX == "1"){
			cout << endl << "\tSidechain Cutoff:\t" << PARAMS.SIDE_SCORE;
		} else {
			cout << endl << "\tFragment Length:\t" << PARAMS.FRAG_LEN;
			cout << endl << "\tAssume Seq. Identical:\t";
			if(PARAMS.IDENTICAL == "1"){
				cout << "Yes";
			} else {
				cout << "No";
                cout << endl << "\tSuperpose Cutoff:\t" << PARAMS.SUPERPOSE;
				cout << endl << "\tRefine Alignment:\t";
				if(PARAMS.EXTRA_REFINEMENT == "1"){
					cout << "Yes";
				} else {
					cout << "No";
				}
				cout << endl << "\tHelix Cutoff:\t\t" << PARAMS.HELIX_CUTOFF;
				cout << endl << "\tHelix Penalty:\t\t" << PARAMS.HELIX_PENALTY;
				cout << endl << "\tColour Score Cutoff:\t" << PARAMS.ALIGN_SCORE;
				cout << endl << "\tColour SideAV Cutoff:\t" << PARAMS.SIDE_SCORE;
			}
			cout << endl << "Identify Rigid Substructures:\t";
			if(PARAMS.CLUSTER == "1"){
				cout << "Yes";
				cout << endl << "\tProcrustes Cutoff:\t" << PARAMS.CLUSTER_SCORE;
				cout << endl << "\tRotation Score Cutoff:\t" << PARAMS.CLUSTER_DEGREES;
				cout << endl << "\tMinimum Cluster Size:\t" << PARAMS.CLUSTER_MIN;
				cout << endl << "\tSingle Linkage Cutoff:\t" << PARAMS.CLUSTER_LINKAGE;
				cout << endl << "\tCluster Rigidity:\t" << PARAMS.CLUSTER_RIGIDITY;
				cout << endl << "\tCluster Colour Res:\t" << PARAMS.CLUSTER_COLOR;
			} else {
				cout << "No";
			}
		}
    }
    cout << endl << endl << "Generate Restraints:\t\t";
    if(PARAMS.RESTRAIN == 0){
        cout << "No";
    } else {
		cout << "Yes";
		cout << endl << "RESTRAIN Options:";
		cout << endl << "\tOnly Use Best Chain:\t";
		if(PARAMS.RESTRAIN_OPTION == 0){
			cout << "Yes";
		} else {
			cout << "No";
		}
		cout << endl << "\tUse All Chains:\t\t";
		if(PARAMS.RESTRAIN_OPTION == 1){
			cout << "Yes";
			cout << endl << "\tAllow Self-Restraining:\t";
			if(PARAMS.RESTRAIN_TO_SELF == 1){
				cout << "Yes";
			} else {
				cout << "No";
			}
		} else {
			cout << "No";
		}
		cout << endl << "\tMax Restraint Dist:\t" << PARAMS.sphere;
		cout << endl << "\tMin Restraint Dist:\t" << PARAMS.MIN_DISTANCE;
		cout << endl << "\tDefault Sigma:\t\t" << PARAMS.sigma;
		cout << endl << "\tMinimum Sigma:\t\t" << PARAMS.MINIMUM_SIGMA;
		cout << endl << "\tBfactor Sigma Scaling:\t";
		if(PARAMS.BFAC1 == "1"){
			if(PARAMS.BFAC2 == "1"){
				cout << "Target and External";
			} else {
				cout << "Target Only";
			}
		} else if(PARAMS.BFAC2 == "1"){
			cout << "External Only";
		} else {
			cout << "No";
		}
		cout << endl << "\tSigma Estimate Type:\t" << PARAMS.sigma_method;
		cout << endl << "\tSigma Multiplier:\t" << PARAMS.RESTRAIN_MULTIPLIER_CUTOFF;
		cout << endl << "\tAlign Score Cutoff:\t" << PARAMS.score_cutoff;
		cout << endl << "\tSideAV Score Cutoff:\t" << PARAMS.RESTRAIN_SIDE_CUTOFF;
		cout << endl << "\tSigma Weight:\t\t" << PARAMS.SIGMA_WEIGHT;
		cout << endl << "\tRemove Bonds/Angles:\t";
		if(PARAMS.GET_BONDS == "1"){
			cout << "Yes";
		} else {
			cout << "No";
		}
		cout << endl << "\tRestrain Sidechains:\t";
		if(PARAMS.MAINCHAIN_ONLY == "0"){
			cout << "Yes";
		} else {
			cout << "No";
		}
    }
	
    cout << endl << endl << "///////////////////////////////////////////////////////////////" << endl << endl;
	
	return;
}

string get_restraints_range(params &PARAMS, int i)
{
    stringstream ss;
    ss.str("");
    for(unsigned int k=0; k<PARAMS.RESTRAIN_PDB.size(); k++){
        if(PARAMS.RESTRAIN_PDB[k]=="ALL" || PARAMS.RESTRAIN_PDB[k]==get_full_filename(PARAMS.filenames1[i])){
            if(PARAMS.RESTRAIN_CHAIN[k]=="ALL" || PARAMS.RESTRAIN_CHAIN[k]==PARAMS.chains1[i]){
                ss << PARAMS.RESTRAIN_RANGE1[k] << "_" << PARAMS.RESTRAIN_RANGE2[k] << "__";
            }
        }
    }
    if(ss.str()==""){
        return "-";
    } else {
        return ss.str();
    }
}

string get_restraints_range_rm(params &PARAMS, int i)
{
    stringstream ss;
    ss.str("");
    for(unsigned int k=0; k<PARAMS.RESTRAIN_RM_PDB.size(); k++){
        if(PARAMS.RESTRAIN_RM_PDB[k]=="ALL" || PARAMS.RESTRAIN_RM_PDB[k]==get_full_filename(PARAMS.filenames1[i])){
            if(PARAMS.RESTRAIN_RM_CHAIN[k]=="ALL" || PARAMS.RESTRAIN_RM_CHAIN[k]==PARAMS.chains1[i]){
                ss << PARAMS.RESTRAIN_RM_RESIDUE[k] << "_";
            }
        }
    }
    if(ss.str()==""){
        return "-";
    } else {
        return ss.str();
    }
}


            
