//
// This file is part of ProSMART.
//

#include "prosmart_process_management.h"

void validate_pdb_file(string &error_file, vector<string> &files, string &input_dir)
//check files exist, and fill spaces with "_"
{
    string temp;	
    ifstream infile;
	for(unsigned int i=0; i<files.size(); i++){
		infile.open(files[i].c_str());
		if(infile.is_open()){
			infile.close();
		} else {
			xml_entry(error_file,0,13,files[i]);
			cout << "Unable to open file " << files[i] << " for reading. Program terminated." << endl;
			exit(-1);
		}
	}
    
    //fill spaces with "_"
#ifndef WINDOWS
    for(unsigned int i=0; i<files.size(); i++){
        temp = replace_spaces(get_full_filename(files[i]),"_");
        if(temp != get_full_filename(files[i])){
            system(("cp " + terminal_spaces(files[i]) + " " + input_dir + temp).c_str());
            files[i] = input_dir + temp;
        }
    }
#endif
    return;
}

bool check_file(string filename)
{
    string line;
    bool result=0;
    const char *filein = filename.c_str();
    ifstream infile(filein, ios::in);
    if(infile.is_open()){
        while(!infile.eof()){
            line.clear();
            getline(infile,line);
            line.clear();
            getline(infile,line);
            if(line[0] == '#'){		//valid bonds files have a # as the first character of the second line.
                result = 1;
            }
            break;
        }
        infile.close();
    }
    return result;
}

void get_bonds(string &error_file, string &fileout_bonds, string &filein, vector<pid_t> &pIDs, int MAX_THREADS, string &REFMAC_NAME, string &logfile)
{
    vector<string> message1;
    vector<string> message2;
    vector<string> message3;
    message1.push_back("REFMAC5 launched to generate bonds for " + get_filename(filein) + " : ");
    message2.push_back("REFMAC5 completed generating bonds for " + get_filename(filein) + " : ");
    message3.push_back("See " + logfile + " for details.");
    
    //write REFMAC keywords to file - this file is subsequently overwritten by REFMAC output
    ofstream outfile;
    outfile.open(fileout_bonds.c_str());
    if(outfile.is_open()){
        outfile << "make hydr no - " << endl << "newligand continue" << endl << "end" << endl;
        outfile.close();
    } else {
        xml_entry(error_file,0,5);
        cout << endl << "Error - unable to open output file " << fileout_bonds << " for writing" << endl;
        exit(0);
    }
    
    stringstream ss;
    ss << REFMAC_NAME << " > \"" << logfile << "\" xyzin \"" << filein << "\" intdist \"" << fileout_bonds << "\" < \"" << fileout_bonds << "\"";
    
#ifdef WINDOWS
    cout << message1[0] << get_time() << endl;
    system(("\"" + ss.str() + "\"").c_str());
    cout << message2[0] << get_time() << endl << message3[0] << endl;
#else
    vector<pid_t> newpIDs;
    vector<string> sys_string;    
    sys_string.push_back(ss.str());    
    spawn_external(error_file,sys_string,message1,message2,message3,1,pIDs,newpIDs,MAX_THREADS);
#endif
    return;
}

void get_atomic_uncertainty(string &error_file, string &fileout, string &filein, string &mtzin, vector<pid_t> &pIDs, int MAX_THREADS, string &REFMAC_NAME, string &logfile)
{
    vector<string> message1;
    vector<string> message2;
    vector<string> message3;
    message1.push_back("REFMAC5 launched to estimate atomic uncertainty for " + get_filename(filein) + " : ");
    message2.push_back("REFMAC5 completed estimating atomic uncertainty for " + get_filename(filein) + " : ");
    message3.push_back("See " + logfile + " for details.");
    
    //write REFMAC keywords to file - this file is subsequently overwritten by REFMAC output
    ofstream outfile;
    outfile.open(fileout.c_str());
    if(outfile.is_open()){
        outfile << "make hydr no - " << endl << "newligand continue" << endl << "NCYC 0" << endl << "end" << endl;
        outfile.close();
    } else {
        xml_entry(error_file,0,5);
        cout << endl << "Error - unable to open output file " << fileout << " for writing" << endl;
        exit(0);
    }
    
    stringstream ss;
    string pdbout = delete_ext(fileout) + ".pdb";
    string mtzout = delete_ext(fileout) + ".mtz";
    ss << REFMAC_NAME << " > \"" << logfile << "\" xyzin \"" << filein << "\" hklin \"" << mtzin << "\" xyzout \"" << pdbout << "\" hklout \"" << mtzout << "\" dpiout \"" << fileout << "\" < \"" << fileout << "\"";
    ss << "; rm " << pdbout << " " << mtzout;
#ifdef WINDOWS
    cout << message1[0] << get_time() << endl;
    system(("\"" + ss.str() + "\"").c_str());
    cout << message2[0] << get_time() << endl << message3[0] << endl;
#else
    vector<pid_t> newpIDs;
    vector<string> sys_string;    
    sys_string.push_back(ss.str());    
    spawn_external(error_file,sys_string,message1,message2,message3,1,pIDs,newpIDs,MAX_THREADS);
#endif
    return;
}

void update_pIDs(vector<pid_t> &pIDs)
{
    pid_t pIDstatus;
    int status=0;
    
    for(unsigned int i=0; i<pIDs.size(); i++){
        pIDstatus = waitpid(pIDs[i],&status,WNOHANG);	//returns 0 if process pIDs[i] exists and has not yet completed
        if(pIDstatus != 0){								//if process either no longer exists, or has completed
            pIDs.erase(pIDs.begin()+i);					//stop tracking this process
            i--;
            continue;
        }
        //cout << endl << i << " " << pIDs[i] << " " << pIDstatus << " " << status;
    }
    return;
}

void wait_for_pIDs(vector<pid_t> &pIDs, vector<pid_t> &newpIDs)
{
    //ensure all child process have terminated before continuing.
    int status = 0;
    for(unsigned int i=0; i<newpIDs.size(); i++)waitpid(newpIDs[i],&status,0);
    newpIDs.clear();
    update_pIDs(pIDs);					//update list of active processes
    return;
}

void spawn_external(string &error_file, vector<string> &sys_string, vector<string> &message, vector<pid_t> &pIDs, vector<pid_t> &newpIDs, int max_threads)
{
#ifdef WINDOWS
   string tmp_str = "TITLE Running ProSMART...";
   for(unsigned int i=0; i<sys_string.size(); i++){
      cout << message[i] << endl;
      tmp_str += " & " + replace_chars(sys_string[i],"/","\\");
   }
   system(tmp_str.c_str());
    return;
#else
    vector<string> empty;
    spawn_external(error_file, sys_string, message, empty, empty, 0, pIDs, newpIDs, max_threads);   //display message at beginning, not at end; don't display time.
    return wait_for_pIDs(pIDs,newpIDs); //ensure all child process have terminated before continuing.
#endif
}

#ifndef WINDOWS
void spawn_external(string &error_file, vector<string> &sys_string, vector<string> &message1, vector<string> &message2, vector<string> &message3, bool DISPLAY_TIME, vector<pid_t> &pIDs, vector<pid_t> &newpIDs, int max_threads)
{  
    pid_t pID;
    int status = 0;
    unsigned int N = sys_string.size();
    bool M1 = 0;
    bool M2 = 0;
    bool M3 = 0;
    if(message1.size()==N)M1 = 1;   //cout before system call
    if(message2.size()==N)M2 = 1;   //cout after system call (and directly before display of current time, if DISPLAY_TIME==1)
    if(message3.size()==N)M3 = 1;   //cout on next line after message2.
    
    //launch multiple child processes, but proceed without waiting for processes to finish.
    for(unsigned int i=0; i<N; i++){
		update_pIDs(pIDs);					//update list of active processes
		update_pIDs(newpIDs);				//update list of active processes
		if((int)pIDs.size() >= max_threads){		//make sure that not too many threads are started
			wait(&status);						//wait for any process to terminate
		}
		pID = fork();
		if(pID == 0){		//child process
            if(M1){
                cout << message1[i];
                if(DISPLAY_TIME)cout << get_time();
                cout << endl;
            }
#ifdef PERFORMANCE
            double t1 = get_time_sec();
#endif
            status = system(sys_string[i].c_str());
#ifdef PERFORMANCE
			double t2 = get_time_sec();
            string file = get_full_filename(string_to_vector(sys_string[i])[0]) + ".txt";
			write_performance_file(file,"NA","NA",0,0,0,0,t2-t1);
#endif
            if(M2){
                cout << message2[i];
                if(DISPLAY_TIME)cout << get_time();
                cout << endl;
            }
            if(M3)cout << message3[i] << endl;
            _exit(status);	//ensure child process is terminated.
		} else if(pID < 0){
			xml_entry(error_file,0,15);
			cout << endl << "Error spawning child process. Program terminated.";
			exit(0);
		}
		pIDs.push_back(pID);				//store process ID of this process
		newpIDs.push_back(pID);				//store process ID of this process
    }
	
    return;
}
#endif

