//
// This file is part of ProSMART.
//

#include "restrain_hbond.h"
#include "restrain_bfgs_optimisation.h"
#include "restrainClass_Atomic_Bonds.h"

//#define OUTPUT_DIFFERENCE_FILES
//#define COUT_CLOSE_ATOMS

int main (int argc, char *argv[])
{
    cout << "////////////////////////////////////////////////" << endl;
    cout << endl << "ProSMART RESTRAIN" << endl; 
    
    if(argc != 25){
        cout << endl << "Error - incorrect arguments passed to ProSMART RESTRAIN."
		<< endl << "Instance launched with command: " << endl;
		for(int i=0; i<argc; i++){
			cout << " " << argv[i];
		}
		cout << endl << "Note - 'prosmart_restrain' should not be manually executed. Please use 'prosmart'."
		<< endl << "Program terminated." << endl;
		return 0;
    }
	
	initialise_reflectM();	//set up global variable, which stores 3x3 reflection matrix
    
    string filein1 = argv[1];
    string filename1 = get_full_filename(filein1);
    string filein2 = argv[2];
    string filename2 = get_full_filename(filein2);
    char chain1 = argv[3][0];
    char chain2 = argv[4][0];
    string workingdirectory = get_folder(argv[5]);
	
    //############### parameters:
    
    double dist_param = 5.0;
    double sigma = 0.4;
    double MIN_SIGMA = 0.01;
    bool REMOVE_BONDS = 0;
    double align_cutoff = 1.0;
    double side_cutoff = 0.5;
    double sigma_cutoff_multiplier = 3.0;
    double SIGMA_WEIGHT = 1.0;
    int HELIX = 0;
    double MIN_DISTANCE = 0.0;
    int SIGMA_METHOD = 2;				// 0 : user-chosen sigmas
	// 1 : constant sigmas
	// 2 : distance-dependent sigmas
    double MAX_DIST_MODIFIER = 2;		// increases sphere size for target structure (reduces the negative effect of sphere-size during sigma estimation by including more restraints).
    double PERFECT_SCORE_CUTOFF = 0.1;	//specifies that if the sum of scores is less than this value, then sigma refinement doesn't make sense.
	bool BFAC1 = 0;
	bool BFAC2 = 0;
	bool MAINCHAIN_ONLY = 0;
    string RANGE_STR = "";
    string RM_STR = "";
    int TYPE = -1;
    bool GENERIC = 0;
    bool HBOND_RESTRAINTS = 0;
	
    dist_param = atof(argv[6]);
    sigma = atof(argv[7]);
    MIN_DISTANCE = atof(argv[8]);
    SIGMA_METHOD = atoi(argv[9]);
    align_cutoff = atof(argv[10]);
    side_cutoff = atof(argv[11]);
    sigma_cutoff_multiplier = atof(argv[12]);
    SIGMA_WEIGHT = atof(argv[13]);
    HELIX = atoi(argv[14]);
    MIN_SIGMA = atof(argv[15]);
    REMOVE_BONDS = atoi(argv[16]);
	BFAC1 = atoi(argv[17]);
	BFAC2 = atoi(argv[18]);
	MAINCHAIN_ONLY = atoi(argv[19]);
    RANGE_STR = argv[20];
    RM_STR = argv[21];
    TYPE = atoi(argv[22]);
    GENERIC = atoi(argv[23]);
    
    /*if(dist_param < 2.0){
     cout << endl << "Sphere size cannot be less than 2." << endl;
     return(-1);
     }*/
    
    //###############	h-bond parameters:
    
    string hbond_filein = argv[24];
    if(hbond_filein!="-")HBOND_RESTRAINTS = 1;
    
    hbond hbond_params;
    hbond_params.DIST = 2.8;
    hbond_params.SIGMA = sigma;                 //controlled with -sigma
    hbond_params.TROUBLESHOOT = 0;
    hbond_params.MAX_DIST = 3.5;                      //maximum restraint distance (controlled with -rmax?)
    hbond_params.MIN_DIST = 2.0;                      //minimum restraint distance (controlled with -rmin?)
    hbond_params.MIN_RESIDUE_SEPARATION = 3;             //minimum residue separation
    hbond_params.MAX_RESIDUE_SEPARATION = -1;            //minimum residue separation
    hbond_params.ALLOWED_RESIDUE_SEPARATIONS.clear();
    hbond_params.REMOVED_RESIDUE_SEPARATIONS.clear();
    hbond_params.RESTRAINT_TYPE = 2;                     //1: helix, 2: sheet (or all main-chain), N: ignore requirement for donor-acceptor
    hbond_params.OVERRIDE_MAX_BONDS = 0;                 //values >0 override max bonds (indiscriminately for all atom types) although don't alter which atom-pairs are used (i.e. doesn't override RESTRAINT_TYPE).
    hbond_params.STRICT_FRAG = 0;
    
    if(HBOND_RESTRAINTS)
        if(!read_hbond_parameters_file(hbond_params,hbond_filein)){
            cout << endl << endl << "Error - could not successfully process input file " << hbond_filein << endl << endl;
            cout << "Program Terminated." << endl << endl;
            return -1;
        }


	
    //###############	PDB file interface:
	
    cout << endl << "Reading PDB files..." << endl;
    cout << "pdb1: " << filein1 << "\tchain " << chain1 << endl;
    cout << "pdb2: " << filein2 << "\tchain " << chain2 << endl;
    /*PDBfile pdb1;
     pdb1.readPDB(filein1, chain1);	// read pdbfile
     PDBfile pdb2;
     pdb2.readPDB(filein2, chain2);
     
     if(MAINCHAIN_ONLY==1){
     PDBfile pdbtmp;
     pdb1.mainchain(pdbtmp);
     pdb1 = pdbtmp;
     pdb2.mainchain(pdbtmp);
     pdb2 = pdbtmp;
     }
	 
     cout << endl << "Any atoms with alternative positions have been removed." << endl;
     pdb1.remove_all_alt_atoms();
     pdb2.remove_all_alt_atoms();
	 
	 if(MAINCHAIN_ONLY==1){
	 PDBfile pdbtmp;
	 pdb1.mainchain(pdbtmp);
	 pdb1 = pdbtmp;
	 pdb2.mainchain(pdbtmp);
	 pdb2 = pdbtmp;
	 }
	 
     */
	
	PDBfile pdb1;
	PDBfile pdb2;
	string obj1 = get_filename(filein1) + "_" + chain1;
	string obj2 = get_filename(filein2) + "_" + chain2;
    vector<res_corresp> original_res1;
    vector<res_corresp> original_res2;
	
    if(GENERIC){
        pdb1.readPDB(filein1, chain1);
        pdb2.readPDB(filein2, chain2);
        pdb1.rename_resnum(original_res1,1);
        pdb2.rename_resnum(original_res2,1);
    } else {
        string input1 = workingdirectory + ".Input_Chains/" + obj1;
        string input2 = workingdirectory + ".Input_Chains/" + obj2;
        if(MAINCHAIN_ONLY==1){
            pdb1.read_formatted_pdb_mainchain(input1);	//format that ProSMART launcher outputs.
            pdb2.read_formatted_pdb_mainchain(input2);
        } else {
            pdb1.read_formatted_pdb(input1);	//format that ProSMART launcher outputs.
            pdb2.read_formatted_pdb(input2);
        }
        original_res1 = read_original_res(input1+"_orig");
        original_res2 = read_original_res(input2+"_orig");
    }
        
	int MAX_RESNUM = original_res1[pdb1.get_highest_resnum()].res;

    cout << "\tpdb1: " << pdb1.size() << " atoms." << endl;
    cout << "\tpdb2: " << pdb2.size() << " atoms." << endl;
    
    if(pdb1.size() == 0 || pdb2.size() == 0){
		cout << endl << endl << "Atoms not successfully read."
        << endl << "ProSMART RESTRAIN completed."
        << endl << endl;
		return 0; 
    }
    
    vector<string> info;	//info vector is not used at current, but could be used to place headers in restraint files
    
    //###############
    
    /*cout << endl << "Sphere size: " << dist_param;
    cout << endl << "Sigma: " << sigma;
    cout << endl << "Alignment score cutoff: " << align_cutoff;
    cout << endl;*/
    
    //############### get alignment info:
    
    vector<residue_alignment> res_align_original;
    if(GENERIC){
        res_align_original = align_to_self(pdb1);
    } else {
        string align_filepath = workingdirectory + "Residue_Alignment_Scores/" + delete_ext(filename1) + "_" + chain1 + "/"+ delete_ext(filename1) + "_" + chain1 + "_" + delete_ext(filename2) + "_" + chain2 + ".txt";
        res_align_original = file_read_alignment(align_filepath,original_res1,original_res2,HELIX);
    }
    
    cout << res_align_original.size() << " residues aligned." << endl;

    vector<residue_alignment> res_align_filtered;
	vector<residue_alignment> restraints1;
    vector<residue_alignment> restraints2;
	vector<double> sigmas;
    vector<int> RANGE_MIN;
    vector<int> RANGE_MAX;
    vector<string> RM_RESIDUES;
    string tmp_str;
	
	if(res_align_original.size() == 0){
		cout << endl << endl << "Warning - residue alignment is of length zero."
		<< endl << "Empty restraints files will be created." << endl << endl;
	} else {		//if no residues are aligned, output empty restraints files.
		
        // allow users to edit restraints lists by selecting certain residues.
        if(RANGE_STR=="-" && RM_STR=="-"){
            res_align_filtered = res_align_original;
        } else {
            if(RANGE_STR!="-" && RANGE_STR.size()>1){
                //convert RANGE_STR to residue ranges
                for(unsigned int i=1; i<RANGE_STR.size()-1; i++){
                    if(RANGE_STR[i]=='_'){
                        if(RANGE_STR[i+1]=='_'){
                            tmp_str = RANGE_STR.substr(0,i);
                            RANGE_MAX.push_back(str_to_int(tmp_str));
                            RANGE_STR.erase(RANGE_STR.begin(),RANGE_STR.begin()+i+2);
                            if(RANGE_STR.size()<2){break;}
                            i=0;
                            continue;
                        }
                        tmp_str = RANGE_STR.substr(0,i);
                        RANGE_MIN.push_back(str_to_int(tmp_str));
                        RANGE_STR.erase(RANGE_STR.begin(),RANGE_STR.begin()+i+1);
                        i=0;
                    }
                }
                /*for(unsigned int i=0; i<RANGE_MIN.size(); i++){
                 cout << endl << RANGE_MIN[i];
                 }
                 cout << endl;
                 for(unsigned int i=0; i<RANGE_MAX.size(); i++){
                 cout << endl << RANGE_MAX[i];
                 }
                 cout << endl;*/
                
                if(RANGE_MIN.size()!=RANGE_MAX.size()){
                    cout << endl << "Warning - residue ranges could not be correctly interpreted."
                    << endl << "Warning - restraints will be generated using all residues." << endl << endl;
                    RANGE_MIN.clear();
                    RANGE_MAX.clear();
                }
            }
            if(RM_STR!="-" && RM_STR.size()>1){
                //convert RM_STR to residues
                for(unsigned int i=1; i<RM_STR.size(); i++){
                    if(RM_STR[i]=='_'){
                        tmp_str = RM_STR.substr(0,i);
                        RM_RESIDUES.push_back(tmp_str);
                        RM_STR.erase(RM_STR.begin(),RM_STR.begin()+i+1);
                        i=0;
                    }
                }
                /*cout << endl;
                 for(unsigned int i=0; i<RM_RESIDUES.size(); i++){
                 cout << endl << RM_RESIDUES[i];
                 }
                 cout << endl;*/
            }
            
        }
        
        /*for(unsigned int i=0; i<res_align_filtered.size(); i++){                    
         cout << res_align_filtered[i] << "\t"
         << original_res1[res_align_filtered[i].res1].res << "'" << original_res1[res_align_filtered[i].res1].ins << "'" << endl;
         }*/
        
		Residues res1;
		Residues res2;
		res1.initialise(pdb1);				//set up residue objects to point to the pdb objects
		res2.initialise(pdb2);				//for easy residue-atom access.
		vector<residue_alignment> res_align;
        bool VALID;
        
        if(RANGE_MIN.size()>0){
            cout << endl << "Only generating restraints using residues in range";
            if(RANGE_MIN.size()>1){
                cout << "s";
            }
            cout << ":  ";
            for(unsigned int i=0; i<RANGE_MIN.size(); i++){
                cout << "[" << RANGE_MIN[i] << "," << RANGE_MAX[i] << "] ";
            }
        }
        if(RM_RESIDUES.size()>0){
            cout << endl << "Restraints will not be generated for ";
            if(RM_RESIDUES.size()>1){
                cout << "these residues:  ";
            } else {
                cout << "this residue:  ";
            }
            for(unsigned int i=0; i<RM_RESIDUES.size(); i++){
                cout << RM_RESIDUES[i] << " ";
            }
        }
        
        if(HELIX==1){
			res_align = res_align_original; //alignment filtering is done in ALIGN for fragment type id.
			string align_filepath2 = workingdirectory + "Residue_Alignment_Scores/" + delete_ext(filename1) + "_" + chain1 + "/."+ delete_ext(filename1) + "_" + chain1 + "_" + delete_ext(filename2) + "_" + chain2 + ".res";
            get_helix_restraints(res_align, pdb1, pdb2, res1, res2, restraints1, restraints2, MIN_DISTANCE, dist_param, align_filepath2, original_res1, RANGE_MIN, RANGE_MAX, RM_RESIDUES);
        } else {
            if(RANGE_MIN.size()>0 || RM_RESIDUES.size()>0){
                for(unsigned int i=0; i<res_align_original.size(); i++){
                    VALID = 1;
                    if(RM_RESIDUES.size()>0){
                        for(unsigned int j=0; j<RM_RESIDUES.size(); j++){
                            if(original_res1[res_align_original[i].res1].ins == ' '){
                                tmp_str = int_to_str(original_res1[res_align_original[i].res1].res);
                            } else {
                                tmp_str = int_to_str(original_res1[res_align_original[i].res1].res) + original_res1[res_align_original[i].res1].ins;
                            }
                            if(tmp_str == RM_RESIDUES[j]){
                                VALID = 0;
                            }
                        }
                    }
                    if(VALID==0){
                        continue;
                    }
                    if(RANGE_MIN.size()==0){
                        res_align_filtered.push_back(res_align_original[i]);
                    } else {
                        for(unsigned int j=0; j<RANGE_MIN.size(); j++){
                            if(original_res1[res_align_original[i].res1].res >= RANGE_MIN[j]){
                                if(original_res1[res_align_original[i].res1].res <= RANGE_MAX[j]){
                                    res_align_filtered.push_back(res_align_original[i]);
                                    break;
                                }
                            }
                        }
                    }
                }
                
                cout << endl << res_align_filtered.size() << " residues aligned after custom filtering." << endl;
            }
            
			cout << endl << "Filtering alignment... (flexible < " << align_cutoff << ", sideAV < " << side_cutoff << ")" << endl;
			res_align = filter_alignment(res_align_filtered,align_cutoff,side_cutoff);
			cout << res_align.size() << " residues aligned after score filtering" << endl;
		}
        
		double sum_scores = 0.0;
		for(unsigned int i=0; i<res_align.size(); i++){
			sum_scores += res_align[i].dist;
		}
		
		if(HELIX==0){
			
			//###############	boxing algorithm:
			
			vector<coord> outer_box1 = get_outer_box(pdb1);
			vector<coord> outer_box2 = get_outer_box(pdb2);
            
			vector<vector<vector<vector<int> > > > boxed_atoms1 = get_boxed_atoms(pdb1, outer_box1, dist_param*MAX_DIST_MODIFIER);
			
            //cout << endl << "Got to here" << endl;
            //return 0;
            
            vector<vector<vector<vector<int> > > > boxed_atoms2 = get_boxed_atoms(pdb2, outer_box2, dist_param);
			
			Array1D<coord> atom_boxes1 = get_atom_boxes(boxed_atoms1, pdb1.size());
			Array1D<coord> atom_boxes2 = get_atom_boxes(boxed_atoms2, pdb2.size());
			
			vector<vector<vector<vector<int> > > > array_adjacent_atoms1 = get_adjacent_atoms_array(boxed_atoms1);
			vector<vector<vector<vector<int> > > > array_adjacent_atoms2 = get_adjacent_atoms_array(boxed_atoms2);
			
			//############### get close atoms
			
			vector<vector<int> > close_atoms1;
			vector<vector<int> > close_atoms2;
			
			for(int i=0; i<atom_boxes1.dim1(); i++){
				close_atoms1.push_back(get_close_atoms(i,atom_boxes1,array_adjacent_atoms1,pdb1,dist_param*MAX_DIST_MODIFIER));
			}
			for(int i=0; i<atom_boxes2.dim1(); i++){
				close_atoms2.push_back(get_close_atoms(i,atom_boxes2,array_adjacent_atoms2,pdb2,dist_param));
			}
            
#ifdef COUT_CLOSE_ATOMS
            for(unsigned int i=0; i<close_atoms1.size(); i++){
                cout << endl << i << "  " << pdb1.get_atom_resnum(i) << pdb1.get_atom_atom(i);
                for(unsigned int j=0; j<close_atoms1[i].size(); j++){
                    cout << endl << "\t" << close_atoms1[i][j] << "  " << pdb1.get_atom_resnum(close_atoms1[i][j]) << pdb1.get_atom_atom(close_atoms1[i][j]) << "  " << pdb1.get_distance(i,close_atoms1[i][j]);
                }
            }
#endif
            			
			//############### get atomic alignment
			
			vector<residue_alignment> atom_align = get_atomic_alignment(res_align,res1,res2,HELIX);
			                  
            /*cout << endl << "Alignment:";
			for(int i=0; i<atom_align.size(); i++){
			 cout << atom_align[i].res1
			 << " " << atom_align[i].res2
			 << " " << pdb1.get_resnum(atom_align[i].res1)
			 << " " << pdb1.get_resid(atom_align[i].res1)
			 << " " << pdb1.get_atom(atom_align[i].res1)
			 << " " << pdb2.get_resnum(atom_align[i].res2)
			 << " " << pdb2.get_resid(atom_align[i].res2)
			 << " " << pdb2.get_atom(atom_align[i].res2)
			 << endl;
			 }*/
			
			//############### identify atomic restraints
			
			//get array indexing the atoms in 'atom_align'
			Array1D<int> atom_idx = get_atom_indexes(atom_align,atom_boxes1.dim1());
			//cout << atom_idx;
			
			get_atom_restraints(restraints1,restraints2,atom_align,close_atoms1,close_atoms2,atom_idx,pdb1,pdb2,MIN_DISTANCE,HELIX);
			
            /*cout << endl << "Restraints:";
            for(unsigned int i=0; i<restraints1.size(); i++){
			 cout << endl << restraints1[i] << "\t" << restraints2[i];
			 }*/
			
		}
        
        if(HBOND_RESTRAINTS==1){
                        
            vector<vector<int> > close_atoms_hbond = h_bond_restraints(pdb1,original_res1,hbond_params,res_align,RANGE_MIN,RANGE_MAX,RM_RESIDUES);
            
            string tmp_filestr = workingdirectory + "Restraints/Chain_Chain/"	+ delete_ext(filename1) + "_" + chain1 + "_" + delete_ext(filename2) + "_" + chain2;
            string fileout_hbond = tmp_filestr + "_distances.pml";
            write_interatomic_distances(fileout_hbond,obj1,pdb1,original_res1,close_atoms_hbond);
            
            fileout_hbond = tmp_filestr + ".txt";
            write_restraints_hbond(info, fileout_hbond, pdb1, original_res1, close_atoms_hbond, hbond_params.DIST, hbond_params.SIGMA, TYPE);
            
            tmp_filestr = workingdirectory + "Colour_Scripts/"	+ delete_ext(filename1) + "_" + chain1 + "_" + delete_ext(filename2) + "_" + chain2;
            fileout_hbond = tmp_filestr + "/bond_residue_separation.pml";
            write_residue_separation(fileout_hbond,obj1,pdb1,original_res1,close_atoms_hbond);
            
            fileout_hbond = tmp_filestr + "/bond_restraints_per_atom.pml";
            write_no_restraints_per_atom(fileout_hbond,obj1,pdb1,original_res1,close_atoms_hbond);
            
            cout << endl << "ProSMART RESTRAIN completed." << endl << endl;
            return 0;
        }
		
		//############### Everything from here is performed the same for both pairwise and fragment type id.
		
		
		if(MIN_DISTANCE > 0.0){
			cout << endl << "Any restraints below " << MIN_DISTANCE << "A have been removed.";
		}
		cout << endl << restraints1.size() << " atomic restraints identified." << endl;
		
		//############### Filtering, generate sigmas, write difference files
		
		vector<double> res_scores1 = get_res_scores(res_align,1);		//creates a vector that gives alignment scores, indexed by residue rather than residue index.
		vector<double> res_scores2 = get_res_scores(res_align,2);		//e.g. if residue 2 happens to be the first residue in the vector res_align, then res_scores[2] will
		//be the score corresponding to residue 2, not the score corresponding to residue res_align[2].
		
		string filein_bonds = workingdirectory + delete_ext(filename1) + "_bonds.txt";
        
        if(MAINCHAIN_ONLY==0){	//use refmac to remove bonds/angles involving sidechains
            if(REMOVE_BONDS==1){
                filter_bonds(filein_bonds,chain1,original_res1,restraints1,restraints2,pdb1,MAX_RESNUM,SIGMA_METHOD);
            } else if(SIGMA_METHOD==1 || SIGMA_METHOD==2){
                cout << endl << "Bonded and angled atom-pairs not removed." << endl
                << "Default sigmas will be used." << endl;
                SIGMA_METHOD = 0;
            }
        } else {
            //don't need refmac for removing bonds if only using mainchain atoms.
            remove_bonds_mainchain(restraints1,restraints2,pdb1);
            cout << endl << "Bonded and angled atom-pairs removed - " << restraints1.size() << " restraints remaining." << endl;
        }
        
		if(restraints1.size()<3){
			cout << endl << endl << "Insufficient restraints created."
			<< endl << "Possible solution: adjust parameter values, e.g. -rmin, -rmax, -cutoff, -side_cutoff."
			<< endl << "ProSMART RESTRAIN completed."
			<< endl << endl;
			return 0;
		}
		
#ifdef OUTPUT_DIFFERENCE_FILES
		
		vector<double> diff_dist;
		//get differences between distances.
		//filter_restraints(restraints1,restraints2,diff_dist,pdb1,pdb2);	//this does not actually filter restraints, it generates diff_dist.
        
		string fileoutdiff = workingdirectory + delete_ext(filename1) + chain1 + "_" + delete_ext(filename2) + chain2 + "_differences.txt";
		write_diff(fileoutdiff,restraints1,restraints2, pdb1,pdb2,res_scores1,res_scores2,diff_dist,original_res1);
		//return 0;
#endif
		
		//cout << endl << pdb1 << endl << pdb2 << endl;
		
		
        /*pdbline temp1;
        pdbline temp2;
        pdbline temp3;
        pdbline temp4;
        for(unsigned int i=0; i<restraints1.size(); i++){
            temp1 = pdb1.line(restraints1[i].res1);
            temp2 = pdb1.line(restraints1[i].res2);
            temp3 = pdb2.line(restraints2[i].res1);
            temp4 = pdb2.line(restraints2[i].res2);
            cout << endl << temp1.res_num << " "
            << temp1.atom << "\t"
            << temp2.res_num << " "
            << temp2.atom << "\t"
            << temp3.res_num << " "
            << temp3.atom << "\t"
            << temp4.res_num << " "
            << temp4.atom << "\t"
            << restraints1[i].dist << " "
            << restraints2[i].dist;
            cout << "\t" << temp1.sigma;
            cout << " " << temp2.sigma;
            cout << "\t" << temp3.sigma;
            cout << " " << temp4.sigma;
        }*/
		 
        
		vector<double> sigma_params;
		if(SIGMA_METHOD==1 || SIGMA_METHOD==2){
			if(sum_scores > PERFECT_SCORE_CUTOFF){	//make sure structures are not identical (actually structurally identical) - otherwise maximum likelihood estimation would fail.
                sigma_params = set_k0(SIGMA_METHOD);	//this initialises the sigma parameter(s).
                sigma_params = generate_sigmas(sigma_params,restraints1, restraints2, pdb1, pdb2, res_scores1, res_scores2,dist_param);
                if(sigma_params.size()==0){	//size=0 implies optimisation failed.
					cout << endl << endl << "Sigma refinement failed - using default." << endl;
                    SIGMA_METHOD = 0;
                } else if(sigma_params.size()==2){
					if(sigma_params[1] < 0){	//if distance-dependent sigmas have negative gradient (against assumption) then recalculate with flat sigmas.
						cout << endl << "Negative distance gradient encountered - reoptimising parameters using non-distance-dependent sigmas." << endl;
						SIGMA_METHOD = 1;
						sigma_params = set_k0(SIGMA_METHOD);
						sigma_params = generate_sigmas(sigma_params,restraints1, restraints2, pdb1, pdb2, res_scores1, res_scores2,dist_param);
						if(sigma_params.size()==0){	//size=0 implies optimisation failed.
							cout << endl << endl << "Sigma refinement failed - using default." << endl;
							SIGMA_METHOD = 0;
						}
					}
				}
			} else {
				cout << endl << endl << "Aligned residues are too similar - using default sigmas." << endl;
				SIGMA_METHOD = 0;
			}
		}
        
		int no_restraints_with_outliers = restraints1.size();
		
		double temp_sigma;
		double ddist;
        double sigma1;
        double sigma2;
		switch(SIGMA_METHOD){
			case 1:		//constant estimated sigmas
				temp_sigma = sqrt(sigma_params[0])*SIGMA_WEIGHT;
				if(temp_sigma < MIN_SIGMA){
					temp_sigma = MIN_SIGMA;
				}
				for(unsigned int i=0; i<restraints1.size(); i++){
					ddist = restraints1[i].dist - restraints2[i].dist;
					if(ddist > temp_sigma*sigma_cutoff_multiplier || 0.0-ddist > temp_sigma*sigma_cutoff_multiplier){
						restraints1.erase(restraints1.begin()+i);
						restraints2.erase(restraints2.begin()+i);
						i--;
					} else {
						sigmas.push_back(temp_sigma);
					}
				}
				break;
			case 2:		//distance-dependent sigmas
				for(unsigned int i=0; i<restraints1.size(); i++){
					temp_sigma = sqrt(sigma_params[0] + (sigma_params[1]*restraints2[i].dist))*SIGMA_WEIGHT;
					ddist = restraints1[i].dist - restraints2[i].dist;
					if(ddist > temp_sigma*sigma_cutoff_multiplier || 0.0-ddist > temp_sigma*sigma_cutoff_multiplier){
						restraints1.erase(restraints1.begin()+i);
						restraints2.erase(restraints2.begin()+i);
						i--;
					} else if(temp_sigma < MIN_SIGMA){
						sigmas.push_back(MIN_SIGMA);
					} else {
						sigmas.push_back(temp_sigma);
					}
				}
				break;
            case 3:		//sum of atomic uncertainties
				for(unsigned int i=0; i<restraints1.size(); i++){
					ddist = restraints1[i].dist - restraints2[i].dist;
                    sigma1 = pdb2.line(restraints2[i].res1).sigma;
                    sigma2 = pdb2.line(restraints2[i].res2).sigma;  //note: these are sigma^2 not sigma!!!!
                    temp_sigma = sqrt(sigma1+sigma2)*SIGMA_WEIGHT;
                    //cout << endl << restraints2[i].res1 << " " << restraints2[i].res2 << " " << sigma1 << " " << sigma2;
					if(ddist > temp_sigma*sigma_cutoff_multiplier || 0.0-ddist > temp_sigma*sigma_cutoff_multiplier){
						restraints1.erase(restraints1.begin()+i);
						restraints2.erase(restraints2.begin()+i);
						i--;
					} else {
						sigmas.push_back(temp_sigma);
					}
				}
				break;
			default:	//constant pre-specified sigmas
				temp_sigma = sigma*SIGMA_WEIGHT;
				for(unsigned int i=0; i<restraints1.size(); i++){
					ddist = restraints1[i].dist - restraints2[i].dist;
					if(ddist > temp_sigma*sigma_cutoff_multiplier || 0.0-ddist > temp_sigma*sigma_cutoff_multiplier){
						restraints1.erase(restraints1.begin()+i);
						restraints2.erase(restraints2.begin()+i);
						i--;
					} else {
						sigmas.push_back(temp_sigma);
					}
				}
		}
		
		cout << endl << no_restraints_with_outliers-restraints1.size() << " outliers removed - " << restraints1.size() << " restraints remaining." << endl;
		
	}
	
	if(BFAC1==1 || BFAC2==1){
		modify_sigmas(restraints1, restraints2, sigmas,BFAC1,BFAC2);
	}
    
    //############### write restraints
    
	//write info to vector<string>
    //vector<string> info = info_vector(filename1, filename2, chain1, chain2, pdb1.size(), pdb2.size(), dist_param, sigma, MIN_DISTANCE, restraints1.size(), PROGRAM_VERSION, align_cutoff, side_cutoff, sigma_cutoff_multiplier, SIGMA_WEIGHT);
    
    string fileout = workingdirectory + "Restraints/Chain_Chain/" + obj1 + "_" + obj2 + ".txt";
    write_restraints(info,fileout,restraints1,restraints2,pdb1,sigmas,original_res1,pdb2,original_res2, TYPE);
    
    cout << endl << endl << "ProSMART RESTRAIN completed." << endl << endl;
    
    return 0;
}
