//
// This file is part of ProSMART.
//

#include "prosmart_align_tools.h"

ostream& operator<<(ostream& out, res_alignment &v)
{
    cout.precision(3);
    cout << v.res1 << "\t" << v.res2 << "\t" << v.min << "\t" << v.central << "\t" << v.rot << "\t" << v.sideRMS << "\t" << v.sideAV << endl;
    return out;
}

vector<res_alignment> file_read_alignment(string &filein)
{
    ifstream infile(filein.c_str(), ios::in);
    string line;
    vector<string> line_v;
    
    res_alignment temp;
    vector<res_alignment> result;
    
    if(infile.is_open()){
        while(!infile.eof()){
            line.clear();
            getline(infile,line);
            line_v = string_to_vector(line);
            
            if(line_v.size()>11 && line[0]!='#'){		//if valid line
                if(line_v[0]=="Res1")continue;          //don't want header
                temp.res1 = line_v[0];
                temp.res2 = line_v[1];
                if(line_v[6] == "NA"){
                    temp.min = -1.0;
                } else {
                    temp.min = str_to_double(line_v[6]);
                }
                if(line_v[7] == "NA"){
                    temp.central = -1.0;
                } else {
                    temp.central = str_to_double(line_v[7]);
                }
                if(line_v[8] == "NA"){
                    temp.rot = -1.0;
                } else {
                    temp.rot = str_to_double(line_v[8]);
                }
                if(line_v[9] == "NA"){
                    temp.sideRMS = -1.0;
                } else {
                    temp.sideRMS = str_to_double(line_v[9]);
                }
                if(line_v[10] == "NA"){
                    temp.sideAV = -1.0;
                } else {
                    temp.sideAV = str_to_double(line_v[10]);
                }
                if(line_v[11] == "NA"){
                    temp.MaxDist = -1.0;
                } else {
                    temp.MaxDist = str_to_double(line_v[11]);
                }
                temp.NumDist = -1;
                temp.asa1 = -1.0;
                temp.asa2 = -1.0;
                if(line_v.size()>12){
                    temp.NumDist = str_to_int(line_v[12]);
                    if(line_v.size()>14){
                        temp.asa1 = str_to_double(line_v[13]);
                        temp.asa2 = str_to_double(line_v[14]);
                    }
                }
                result.push_back(temp);
            }
        }
        infile.close();
    } 
    else cout << endl << "Unable to open " << filein << " for reading" << endl; 
    
    return result;
}

vector<res_alignment> file_read_align_types(string &filein)
{
    ifstream infile(filein.c_str(), ios::in);
    string line;
    vector<string> line_v;
    
    res_alignment temp;
    vector<res_alignment> result;
    
    if(infile.is_open()){
        while(!infile.eof()){
			line.clear();
			getline(infile,line);
			line_v = string_to_vector(line);
			
			if(line_v.size()==2 && line[0]!='#'){		//if valid line
				temp.res1 = line_v[0];
				temp.min = str_to_double(line_v[1]);
				result.push_back(temp);
			}
        }
        infile.close();
    } 
    else cout << endl << "Unable to open " << filein << " for reading" << endl; 
    
    return result;
}

void write_align(vector<res_alignment> &align, string &file)
{
    ofstream outfile;
    
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Alignment File" << endl;
        outfile << "#" << endl;
        /*for(unsigned int i=0; i<info.size(); i++){
         outfile << info[i] << endl;
         }*/
        outfile << "# Res1\tRes2\tFlexi\tProcr\tHinge\tSideRMS\tSideAV\tMaxDist" << endl;
        for(unsigned int i=0; i<align.size(); i++){
			outfile.precision(3);
			outfile << align[i].res1 << "\t"
			<< align[i].res2 << "\t"
           << align[i].min << "\t"
           << align[i].central << "\t"
           << align[i].rot << "\t"
           << align[i].sideRMS << "\t"
			  << align[i].sideAV << "\t"
           << align[i].MaxDist << "\t"
			 << endl;
		}
        outfile.close();
        //cout << "Extra alignment file written to: " << file << endl << endl;
    } else {
        cout << "Unable to open output file " << file << " for writing" << endl << endl;
    }
    
    return;
}

vector<res_alignment> max_align_scores(vector<vector<res_alignment> > &align_v)
{
    vector<res_alignment> result;
    unsigned int idx;
	int offset = str_to_int(align_v[0][0].res1);
	res_alignment empty;
	empty.res1 = "";
	empty.res2 = "";
	empty.min = -1.0;
	empty.central = -1.0;
	empty.rot = -1.0;
	empty.sideRMS = -1.0;
	empty.sideAV = -1.0;
	empty.MaxDist = -1.0;
	
	for(unsigned int i=0; i<align_v.size(); i++){
		if(str_to_int(align_v[i][0].res1) < offset){
			offset = str_to_int(align_v[i][0].res1);		//this normalises so that smallest resnum is always 0 - this means it wont crash if resnum is negative
		}
	}
	
	for(unsigned int i=0; i<align_v.size(); i++){
		for(unsigned int j=0; j<align_v[i].size(); j++){
			idx = str_to_int(align_v[i][j].res1) - offset;
			while(result.size()<=idx){
				result.push_back(empty);
			}
			if(result[idx].min < align_v[i][j].min || result[idx].min < 0.0){
				result[idx].min = align_v[i][j].min;
			}
			if(result[idx].central < align_v[i][j].central || result[idx].central < 0.0){
				result[idx].central = align_v[i][j].central;
			}
			if(result[idx].rot < align_v[i][j].rot || result[idx].rot < 0.0){
				result[idx].rot = align_v[i][j].rot;
			}
			if(result[idx].sideRMS < align_v[i][j].sideRMS || result[idx].sideRMS < 0.0){
				result[idx].sideRMS = align_v[i][j].sideRMS;
			}
			if(result[idx].sideAV < align_v[i][j].sideAV || result[idx].sideAV < 0.0){
				result[idx].sideAV = align_v[i][j].sideAV;
			}
         if(result[idx].MaxDist < align_v[i][j].MaxDist || result[idx].MaxDist < 0.0){
				result[idx].MaxDist = align_v[i][j].MaxDist;
			}
		}
	}
	
	for(unsigned int i=0; i<result.size(); i++){
		result[i].res1 = int_to_str(i+offset);
		result[i].res2 = result[i].res1;
		//cout << result[i] << endl;
	}
    
    return result;
}

coord prosmart_green1;
coord prosmart_red1;

void define_pymol_colors(coord &col1, coord &col2)
{
	prosmart_green1 = col1;
	prosmart_red1 = col2;
	return;
}

void write_pymol(int OPTION, vector<res_alignment> &align, string &file, vector<string> &obj1, vector<char> &domain1, double score_cutoff)
{
    ofstream outfile;
    double score = 0.0;
	double val;
	double ival;
    vector<double> colors11;
    vector<double> colors12;
    vector<double> colors13;
    for(unsigned int i=0; i<align.size(); i++){
        switch(OPTION){
            case 1:
                score = align[i].min;
                break;
            case 2:
                score = align[i].central;
                break;
            case 3:
                score = align[i].sideRMS;
                break;
            case 4:
                score = align[i].sideAV;
                break;
            case 5:
                score = 1000*align[i].rot;
                break;
            case 6:
                if(align[i].rot<0)
                    score = -1;
                else
                    score = 1000*(1-cos(align[i].rot*(atan(1.0)/45.0)));
                break;
           case 7:
              score = align[i].MaxDist;
              break;
            default:
                break;
        }
        val = score/score_cutoff;
        if(val < 0.0){
            colors11.push_back(1);			//not aligned: color = white
            colors12.push_back(1);
            colors13.push_back(1);
        } else {
            if(val < 1){			//green = similar  ->  red = comparitively dissimilar
                ival = 1-val;
                colors11.push_back(prosmart_green1.x*ival+prosmart_red1.x*val);
                colors12.push_back(prosmart_green1.y*ival+prosmart_red1.y*val);
                colors13.push_back(prosmart_green1.z*ival+prosmart_red1.z*val);
            } else {
                colors11.push_back(prosmart_red1.x);		//structurally dissimilar: color = dark red
                colors12.push_back(prosmart_red1.y);
                colors13.push_back(prosmart_red1.z);
            }
        }
    }
    
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        /*for(unsigned int i=0; i<info.size(); i++){
         outfile << info[i] << endl;
         }*/
		for(unsigned int i=0; i<obj1.size(); i++){
			outfile << "color white, (" << obj1[i] << "//" << domain1[i] << "/*/*)" << endl;
		}
		for(unsigned int i=0; i<align.size(); i++){
			outfile << "set_color newcolor" << i << " = [" << colors11[i] << "," << colors12[i] << "," << colors13[i] << "]";
			for(unsigned int j=0; j<obj1.size(); j++){
				outfile << "; color newcolor" << i << ", (" << obj1[j] << "//" << domain1[j] << "/" << align[i].res1 << "/*)";
			}
			outfile << endl;
		}
        outfile.close();
		//cout << "Extra PyMOL colour script written to: " << file << endl;
    } else {
        cout << "Unable to open output file " << file << " for writing" << endl << endl;
    }
    
    return;
}

void write_pymol_nonidentical(int OPTION, vector<res_alignment> &align, vector<vector<res_alignment> > &align_v, string &file, string &obj1, string &chain1, vector<string> &obj2, vector<string> &chain2, double score_cutoff)
{
    ofstream outfile;
    double score = 0.0;
	double val;
	double ival;
    vector<double> colors11;
    vector<double> colors12;
    vector<double> colors13;
    for(unsigned int i=0; i<align.size(); i++){
        switch(OPTION){
            case 1:
                score = align[i].min;
                break;
            case 2:
                score = align[i].central;
                break;
            case 3:
                score = align[i].sideRMS;
                break;
            case 4:
                score = align[i].sideAV;
                break;
            case 5:
                score = 1000*align[i].rot;
                break;
            case 6:
                if(align[i].rot<0)
                    score = -1;
                else
                    score = 1000*(1-cos(align[i].rot*(atan(1.0)/45.0)));
                break;
            case 7:
                score = align[i].MaxDist;
                break;
            default:
                break;
        }
        val = score/score_cutoff;
        if(val < 0.0){
            colors11.push_back(1);			//not aligned: color = white
            colors12.push_back(1);
            colors13.push_back(1);
        } else {
            if(val < 1){			//green = similar  ->  red = comparitively dissimilar
                ival = 1-val;
                colors11.push_back(prosmart_green1.x*ival+prosmart_red1.x*val);
                colors12.push_back(prosmart_green1.y*ival+prosmart_red1.y*val);
                colors13.push_back(prosmart_green1.z*ival+prosmart_red1.z*val);
            } else {
                colors11.push_back(prosmart_red1.x);		//structurally dissimilar: color = dark red
                colors12.push_back(prosmart_red1.y);
                colors13.push_back(prosmart_red1.z);
            }
        }
    }
    
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        /*for(unsigned int i=0; i<info.size(); i++){
         outfile << info[i] << endl;
         }*/
        outfile << "color white, (" << obj1 << "*//" << chain1 << "/*/*)" << endl;
        for(unsigned int k=0; k<align.size(); k++){
            outfile << "set_color newcolor" << k << " = [" << colors11[k] << "," << colors12[k] << "," << colors13[k] << "]";
            outfile << "; color newcolor" << k << ", (" << obj1 << "*//" << chain1 << "/" << align[k].res1 << "/*)";
            outfile << endl;
        }
		for(unsigned int i=0; i<obj2.size(); i++){
			outfile << "color white, (" << obj2[i] << "*//" << chain2[i] << "/*/*)" << endl;
            for(unsigned int j=0; j<align_v[i].size(); j++){
                for(unsigned int k=0; k<align.size(); k++){
                    if(align[k].res1 == align_v[i][j].res1){
                        outfile << "color newcolor" << k << ", (" << obj2[i] << "*//" << chain2[i] << "/" << align_v[i][j].res2 << "/*)" << endl;
                        break;
                    }
                }
                
            }
		}
        outfile.close();
		//cout << "Extra PyMOL colour script written to: " << file << endl;
    } else {
        cout << "Unable to open output file " << file << " for writing" << endl << endl;
    }
    
    return;
}

void write_pymol_nonidentical(int OPTION, vector<vector<res_alignment> > &max_scores, string &file, vector<string> &objs, vector<string> &chains, double score_cutoff)
{
    ofstream outfile;
    double score = 0.0;
	double val;
	double ival;
    double tmp_pi = atan(1.0)/45.0;
    
    vector<double> colors11;
    vector<double> colors12;
    vector<double> colors13;
    int NO_COL = 100;   //this will actually cause NO_COL+1 colours to be created...
    for(int i=0; i<=NO_COL; i++){
        val = (double)i/NO_COL;
        ival = 1.0-val;
        colors11.push_back(prosmart_green1.x*ival+prosmart_red1.x*val);
        colors12.push_back(prosmart_green1.y*ival+prosmart_red1.y*val);
        colors13.push_back(prosmart_green1.z*ival+prosmart_red1.z*val);
    }
    
    vector<vector<int> > indexes;
    vector<int> empty;
    for(unsigned int i=0; i<max_scores.size(); i++){
        indexes.push_back(empty);
        for(unsigned int j=0; j<max_scores[i].size(); j++){
            switch(OPTION){
                case 1:
                    score = max_scores[i][j].min;
                    break;
                case 2:
                    score = max_scores[i][j].central;
                    break;
                case 3:
                    score = max_scores[i][j].sideRMS;
                    break;
                case 4:
                    score = max_scores[i][j].sideAV;
                    break;
                case 5:
                    score = 1000*max_scores[i][j].rot;
                    break;
                case 6:
                    if(max_scores[i][j].rot<0)
                        score = -1;
                    else
                        score = 1000*(1-cos(max_scores[i][j].rot*tmp_pi));
                    break;
                case 7:
                    score = max_scores[i][j].MaxDist;
                    break;
                default:
                    break;
            }
            val = (double)NO_COL*score/score_cutoff;
            if(val < 0.0){
                indexes[i].push_back(-1);
            } else {
                if(val < (double)NO_COL){			//green = similar  ->  red = comparitively dissimilar
                    indexes[i].push_back(floor(val));
                } else {
                    indexes[i].push_back(NO_COL);
                }
            }
        }
    }
    
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        for(unsigned int i=0; i<colors11.size(); i++){
            outfile << "set_color newcolor" << i << " = [" << colors11[i] << "," << colors12[i] << "," << colors13[i] << "]" << endl;
        }
        vector<string> str_v;
        for(unsigned int i=0; i<objs.size(); i++){
            outfile << "color white, (" << objs[i] << "*//" << chains[i] << "/*/*)" << endl;
            vector<string> strv;
            for(unsigned int j=0; j<indexes[i].size(); j++){
                if(indexes[i][j]>=0){
                    strv.push_back(max_scores[i][j].res1);
                    /*if(str.size()==0){
                        if(max_scores[i][j].res1[0]=='-')
                            str.push_back(max_scores[i][j].res1);
                        else 
                            str = max_scores[i][j].res1;
                    } else {
                        if(max_scores[i][j].res1[0]=='-')
                            str += "+\\" + max_scores[i][j].res1;
                        else 
                            str += "+" + max_scores[i][j].res1;
                    }*/
                    outfile << "color newcolor" << indexes[i][j] << ", (" << objs[i] << "*//" << chains[i] << "/";
                    if(max_scores[i][j].res1[0]=='-')
                        outfile << "\\";
                    outfile << max_scores[i][j].res1 << "/*)" << endl;
                }
            }
            string str = "";
            if(strv.size()>0){
                if(strv[0][0]=='-'){
                    str = " and not i. \\" + strv[0];
                } else {
                    str = " and not i. " + strv[0];
                }
                bool GOT_LINK = 0;
                for(unsigned int i=1; i<strv.size(); i++){
                    if(int_to_str(str_to_int(strv[i-1])) == strv[i-1]){
                        if(int_to_str(str_to_int(strv[i])) == strv[i]){
                            if(str_to_int(strv[i-1]) == str_to_int(strv[i])-1){
                                GOT_LINK = 1;
                                continue;
                            }
                        }
                    }
                    if(GOT_LINK){
                        if(strv[i-1][0]=='-'){
                            str += "-\\" + strv[i-1];
                        } else {
                            str += "-" + strv[i-1];
                        }
                        GOT_LINK = 0;;
                    }
                    if(strv[i][0]=='-'){
                        str += " and not i. \\" + strv[i];
                    } else {
                        str += " and not i. " + strv[i];
                    }
                }
                if(GOT_LINK){
                    if(strv[strv.size()-1][0]=='-'){
                        str += "-\\" + strv[strv.size()-1];
                    } else {
                        str += "-" + strv[strv.size()-1];
                    }
                    GOT_LINK = 0;;
                }
            }
            str_v.push_back(str);
        }
        for(unsigned int i=0; i<objs.size(); i++){
            outfile << "hide everything, " << objs[i] << "* " << str_v[i] << endl;
        }
        outfile.close();
		//cout << "Extra PyMOL colour script written to: " << file << endl;
    } else {
        cout << "Unable to open output file " << file << " for writing" << endl << endl;
    }
    
    return;
}

