//
// This file is part of ProSMART.
//

#ifndef prosmart_array_tools_H
#define prosmart_array_tools_H

#include <fstream>
#include <vector>
#include <string>
#include <cstring>
#include "jama_svd.h"
#include "jama_lu.h"
#include "tnt_array2d_utils.h"
#include "prosmart_flags.h"
using namespace TNT;
using namespace std;
using namespace JAMA;

struct LIST_ELEMENT
{
    double val;
    int idx;
    bool operator() (LIST_ELEMENT i, LIST_ELEMENT j) { return (i.val>j.val);}
};

template<class T> ostream& operator<<(ostream&, vector<T>&);
template<class T> T sum(vector<T>&);
template<class T> double average(vector<T> &);

double sum(Array1D<double>&);
double average(Array1D<double>&);
vector<double> average_v(vector<double> &, vector<double> &);
double stddev(vector<double> &, double);

Array1D<double> SV(Array2D<double>&);
void SVDuv(Array2D<double> &, Array2D<double> &, Array2D<double> &);
double get_angle(Array2D<double> &);
template <class T>
Array2D<T> crossprod(Array2D<T> &, Array2D<T> &);
template <class T> 
Array1D<T> vprod(Array2D<T> &, vector<T> &);
template <class T>
Array1D<T> vprod_t(Array2D<T> &, Array1D<T> &);
Array2D<double> crossprod_tt(Array2D<double> &, Array2D<double> &);
Array2D<double> crossprod_weight(Array2D<double> &, Array2D<double> &, vector<double> &);
template <class T>
Array2D<T> t(const Array2D<T> &);
template <class T>
T tr(const Array2D<T> &);
template <class T>
T tr_cov(const Array2D<T> &);
double tr_cov(const Array2D<double> &, const Array2D<double> &);
double det3x3(const Array2D<double> &);
template <class T>
T absol(const T&, const T&);
double absol(const double&);
Array2D<double> absol(Array2D<double>&);
Array1D<double> decimal_places (Array1D<double> &, int);
double decimal_places(double, int);
void fileoutput_array(Array2D<double>, string, string, bool);

double spearman_rank_corr_coef(vector<double> &, vector<double> &);
vector<double> prosmart_rank(vector<double> &);
template<class T>
Array2D<T> inverse(const Array2D<T> &);
template<class T>
Array2D<T> pseudoinverse(const Array2D<T> &);

//////////////////

template<class T> ostream& operator<<(ostream& out, vector<T>& v)
{
    for(unsigned int i=0; i<v.size(); i++){
        cout << v[i] << " ";
    }
    
    return out;
}

template<class T> T sum(vector<T> &A)
{
    T ctr=(T)0;
    for(unsigned int i=0; i<A.size(); i++)
        ctr+=A[i];
    return ctr;
}

template<class T> T min(vector<T> &A)
{
    T result=(T)0;
    if(A.size()>0){
        result = A[0];
    }
    for(unsigned int i=1; i<A.size(); i++){
        if(result>A[i]){
            result = A[i];
        }
    }
    return result;
}

template<class T> T max(vector<T> &A)
{
    T result=(T)0;
    if(A.size()>0){
        result = A[0];
    }
    for(unsigned int i=1; i<A.size(); i++){
        if(result<A[i]){
            result = A[i];
        }
    }
    return result;
}

template<class T> unsigned int n_greater_than(vector<T> &A, T val)
{
    unsigned int result = 0;
    for(unsigned int i=0; i<A.size(); i++)
        if(A[i]>val)
            result++;
    return result;
}

template<class T> double average(vector<T> &A)
{
    T result = sum(A);
    return (double)result/A.size();
}

template<class T> double median(vector<T> &A)
{
    T result;
    unsigned int n = A.size();
    sort(A.begin(), A.end());
    if(n%2 == 0){
        result = 0.5*(A[(n/2)-1]+A[n/2]);
    } else {
        result = A[n/2];
    }
    return result;
}

template <class T>
T tr(const Array2D<T> &A)
{
    int m = A.dim1();
    int n = A.dim2();
    
    if(m!=n){
        cout << "Error in calculating trace - not a square matrix.";
    }
    
    T B=0.0;
    for(int i=0; i<m; i++){
        B += A[i][i];
    }
    return B;
}

template <class T> Array2D<T> crossprod(Array2D<T> &A, Array2D<T> &B)
{
    int nA = A.dim2();
    int nB = B.dim2();
    Array2D<T> temp(nA,nB);
    int N = A.dim1();
    
    for(int i=0; i<nA; i++){
        for(int j=0; j<nB; j++){
            temp[i][j] = 0;
            for(int k=0; k<N; k++){
                temp[i][j] += A[k][i]*B[k][j];
            }
        }
    }
    
    return temp;
}

template <class T> Array1D<T> vprod(Array2D<T> &A, vector<T> &B)
// A[n1,n2]^T*B[n1,1] = C[n2,1]
{
    int n1 = A.dim1();
    int n2 = A.dim2();
    Array1D<T> temp(n2,0.0);
    
    if((int)B.size()!=n1){
        return temp;
    }
    
    for(int i=0; i<n1; i++){
        for(int j=0; j<n2; j++){
            temp[j] += A[i][j]*B[i];
        }
    }
    
    return temp;
}

template <class T> Array1D<T> vprod(Array2D<T> &A, Array1D<T> &B)
// A[n1,n2]^T*B[n1,1] = C[n2,1]
{
    int n1 = A.dim1();
    int n2 = A.dim2();
    Array1D<T> temp(n1,0.0);
    
    if(B.dim1()!=n2){
        return temp;
    }
    
    for(int i=0; i<n1; i++){
        for(int j=0; j<n2; j++){
            temp[i] += A[i][j]*B[j];
            //cout << endl << A[i][j] << "\t" << B[j];
        }
    }
    
    return temp;
}

template <class T> Array2D<T> t(const Array2D<T> &A)
{
    int m = A.dim1();
    int n = A.dim2();
    
    Array2D<T> B(n,m);
    
    for(int i=0; i<n; i++){
        for(int j=0; j<m; j++){
            B[i][j] = A[j][i];
        }
    }
    return B;
}

template <class T> Array2D<T> operator*(Array2D<T> A, T &B)
{
    Array2D<double> result(A.dim1(),A.dim2());
    for(int i=0; i<A.dim1(); i++){
        for(int j=0; j<A.dim2(); j++){
            result = A[i][j]*B;
        }
    }
    return result;
}

template<class T>
Array2D<T> inverse(const Array2D<T> &M)
//taken from http://wiki.cs.princeton.edu/index.php/TNT
{    
	// solve for inverse with LU decomposition
	JAMA::LU<T> lu(M);
    
	// create identity matrix
	Array2D<T> id(M.dim1(), M.dim2(), (T)0);
	for(int i=0; i<M.dim1(); i++) id[i][i] = 1;
    
    // solves A * A_inv = Identity
	return lu.solve(id);
}

template<class T>
Array2D<T> pseudoinverse(const Array2D<T> &M)
{
    SVD<T> tempsvd(M,1);
    Array2D<T> U;
    Array2D<T> V;
    Array1D<T> D;
    tempsvd.getU(U);
    tempsvd.getV(V);
    tempsvd.getSingularValues(D);
    vector<T> Dinv;
    for(int i=0; i<D.dim1(); i++){
        //cout << endl << i << " " << D[i];
        if(D[i]<=D[0]*0.0000000001){
            Dinv.push_back(0.0);
        } else {
            Dinv.push_back(1.0/D[i]);
        }
    }    
    return crossprod_weight(V,U,Dinv);
}

#endif
