//
// This file is part of ProSMART.
//

#include "prosmart_multiple_superposition.h"

Array1D<PDBfile> get_pdb_files(string &workingdirectory, vector<string> &filenames, vector<string> &chains)
{
    unsigned int N = filenames.size();
    Array1D<PDBfile> pdb_v(N);
    stringstream ss;
    
    //cout << endl << "getting PDB objects..." << endl << flush;
    //initialise PDB objects
    for(unsigned int i=0; i<N; i++){
        string input = workingdirectory + ".Input_Chains/" + get_filename(filenames[i]) + "_" + chains[i][0];
        pdb_v[i].read_formatted_pdb(input);
        vector<res_corresp> original_res = read_original_res(input+"_orig");
        for(int j=0; j<pdb_v[i].size(); j++){
            pdb_v[i].set_resnum(j,original_res[pdb_v[i].get_resnum(j)].res);
        }
    }
    return pdb_v;
}

Array1D<vector<vector<unsigned int> > > get_residues(Array1D<PDBfile> &pdb_v)
{
    Array1D<vector<vector<unsigned int> > > residues(pdb_v.dim1());
    vector<vector<unsigned int> > empty_vv;
    vector<unsigned int> empty_v;

    //cout << endl << "getting residues..." << endl << flush;
    for(int i=0; i<pdb_v.dim1(); i++){
        residues[i] = empty_vv;
        residues[i].push_back(empty_v);
        residues[i].back().push_back(0);
        for(int j=1; j<pdb_v[i].size(); j++){
            if(pdb_v[i].get_orig_resnum(j)!=pdb_v[i].get_orig_resnum(j-1)){
                residues[i].push_back(empty_v);
            }
            residues[i].back().push_back(j);
        }
    }
    /*for(unsigned int i=0; i<residues.size(); i++){
        cout << endl << i;
        for(unsigned int j=0; j<residues[i].size(); j++){
            cout << endl << "\t" << j;
            for(unsigned int k=0; k<residues[i][j].size(); k++){
                cout << endl << "\t\t" << k << "\t" << residues[i][j][k] << "\t" << pdb_v[i].get_orig_resnum(residues[i][j][k]);
            }
        }
    }*/
        
    return residues;
}

Array2D<vector<unsigned int> > get_residue_alignment(Array1D<PDBfile> &pdb_v, Array1D<vector<vector<unsigned int> > > &res_v, Array2D<vector<res_alignment> > &align_m)
{
    int N = pdb_v.dim1();
    Array2D<vector<unsigned int> > result(N,N);
    vector<unsigned int> empty_v;
    unsigned int idx1 = 0;
    unsigned int idx2 = 0;
    bool VALID = 0;
    
    //cout << endl << "getting alignment indexes..." << endl << flush;
    for(int i=0; i<N; i++){
        for(int j=i+1; j<N; j++){
            result[i][j] = empty_v;
            result[j][i] = empty_v;
            for(unsigned int k=0; k<align_m[i][j].size(); k++){ //this could be optimised by starting from the last match (+1).
                VALID = 0;
                for(unsigned int l=0; l<res_v[i].size(); l++){
                    if(delete_spaces(pdb_v[i].get_orig_resnum(res_v[i][l][0])) == align_m[i][j][k].res1){
                        idx1 = l;
                        VALID = 1;
                        break;
                    }
                }
                if(!VALID){break;}
                VALID = 0;
                for(unsigned int l=0; l<res_v[j].size(); l++){
                    if(delete_spaces(pdb_v[j].get_orig_resnum(res_v[j][l][0])) == align_m[i][j][k].res2){
                        idx2 = l;
                        VALID = 1;
                        break;
                    }
                }
                if(!VALID){break;}
                result[i][j].push_back(idx1);
                result[j][i].push_back(idx2);
            }
            if(!VALID){break;}
        }
        if(!VALID){break;}
    }
    if(!VALID){
        cout << endl << "Error getting residue alignment indexes." << endl << endl;
        exit(-1);
    }
    /*for(int i=0; i<N; i++){
        for(int j=i+1; j<N; j++){
            for(unsigned int k=0; k<result[i][j].size(); k++){
                cout << endl << i << " " << j << "\t" << result[i][j][k] << "\t" << result[j][i][k];
            }
        }
    }*/
    return result;
}

Array2D<vector<unsigned int> > get_atom_alignment(Array1D<PDBfile> &pdb_v, Array1D<vector<vector<unsigned int> > > &res_v, Array2D<vector<unsigned int> > &res_align)
{
    int N = pdb_v.dim1();
    Array2D<vector<unsigned int> > result(N,N);
    vector<unsigned int> empty_v;
    
    //cout << endl << "getting alignment indexes..." << endl << flush;
    for(int i=0; i<N; i++){
        for(int j=i+1; j<N; j++){
            for(unsigned int k=0; k<res_align[i][j].size(); k++){   // for each aligned residue
                for(unsigned int l=0; l<res_v[i][res_align[i][j][k]].size(); l++){  // for each atom in structure1
                    for(unsigned int m=0; m<res_v[j][res_align[j][i][k]].size(); m++){  // for each atom in structure2
                        if(pdb_v[i].get_atom(res_v[i][res_align[i][j][k]][l]) == pdb_v[j].get_atom(res_v[j][res_align[j][i][k]][m])){
                            result[i][j].push_back(res_v[i][res_align[i][j][k]][l]);
                            result[j][i].push_back(res_v[j][res_align[j][i][k]][m]);
                            break;
                        }
                    }
                }
            }
        }
    }
    /*for(int i=0; i<N; i++){
        for(int j=i+1; j<N; j++){
            for(unsigned int k=0; k<result[i][j].size(); k++){
                cout << endl << i << " " << j << "\t" << result[i][j][k] << "\t" << result[j][i][k];
                cout << "\t" << pdb_v[i].get_atom(result[i][j][k]) << " " << pdb_v[j].get_atom(result[j][i][k]);
                if(pdb_v[i].get_atom(result[i][j][k]) != pdb_v[j].get_atom(result[j][i][k])){
                    cout << endl << "Error getting atomic alignment indexes." << endl << endl;
                    exit(-1);
                }
            }
        }
    }*/
    return result;
}

void sieve_fit_atoms(Array1D<PDBfile> &pdb_v, Array2D<vector<unsigned int> > &aij)
{
    int N = pdb_v.dim1();
    Array2D<vector<unsigned int> > result(N,N);
    
    for(int i=0; i<N; i++){
        for(int j=1; j<N; j++){
            if(i>=j){continue;}
            if(aij[i][j].size()==0){continue;}
            Coords tmp1;
            Coords tmp2;
            pdb_v[i].getcoords(tmp1,aij[i][j]);
            pdb_v[j].getcoords(tmp2,aij[j][i]);
            vector<unsigned int> removed = sieve_fit(tmp1,tmp2);
            for(unsigned int k=0; k<removed.size(); k++){
                aij[i][j].erase(aij[i][j].begin()+removed[k]);
                aij[j][i].erase(aij[j][i].begin()+removed[k]);
            }
        }
    }
    return;
}

/*void sieve_fit_atoms(Array1D<PDBfile> &pdb_v, Array2D<vector<unsigned int> > &aij)
{
    int N = pdb_v.dim1();
    Array2D<vector<unsigned int> > result(N,N);
    
    for(int i=0; i<N; i++){
        for(int j=1; j<N; j++){
            if(i>=j){continue;}
            Coords tmp1;
            Coords tmp2;
            pdb_v[i].getcoords(tmp1,aij[i][j]);
            pdb_v[j].getcoords(tmp2,aij[j][i]);
            vector<unsigned int> removed = sieve_fit(tmp1,tmp2);
            cout << endl << i << " " << j << endl << removed << endl;
            if(removed.size()==0){continue;}
            sort(removed.begin(),removed.end());
            cout << endl << i << " " << j << endl << removed << endl;
            unsigned int ctr = 0;
            for(unsigned int k=0; k<aij[i][j].size(); k++){
                if(aij[i][j][k]==removed[ctr]){
                    if(removed.size()<ctr-1){
                        ctr++;
                    }
                } else {
                    result[i][j].push_back(aij[i][j][k]);
                    result[j][i].push_back(aij[j][i][k]);
                }
            }
        }
    }
    return result;
}*/

void ensure_consensus(Array2D<vector<unsigned int> > &aij)
//note - this should work for atoms, residues, and more generally any strictly positively ordered matrix.
{
    int N = aij.dim1();
    Array1D<vector<unsigned int> > result(N);
    Array2D<vector<unsigned int> > final(N,N);
    bool CONSENSUS = 1;

    /*for(int i=0; i<N; i++){
     for(int j=1; j<N; j++){
     if(i>=j){continue;}
     cout << endl << "here: " << i << " " << j << endl << aij[i][j];
     cout << endl << "here: " << j << " " << i << endl << aij[j][i];
     }
     }*/
    
    //ensure consistency between 0 and j, to get initial vectors of equal length.
    Array1D<unsigned int> idx(N,(unsigned int)0);
    for(unsigned int k=0; k<aij[0][1].size(); k++){
        CONSENSUS = 1;
        for(int j=2; j<N; j++){
            while(aij[0][j][idx[j]]<aij[0][1][k]){
                if(idx[j]>=aij[0][j].size()-1){
                    break;
                }
                idx[j]++;
            }
            if(aij[0][j][idx[j]]!=aij[0][1][k]){
                CONSENSUS = 0;
                break;
            }
        }
        if(CONSENSUS){
            result[0].push_back(k);      // aij[0][1][k]
            result[1].push_back(k);      // aij[1][0][k]
            for(int j=2; j<N; j++){
                result[j].push_back(idx[j]);   // aij[j][0][[idx[j]]]
            }
        }
    }

    /*for(unsigned int k=result[0].size()-3; k<result[0].size(); k++){
        cout << endl << endl << k;
        cout << endl << "\t";
        for(int j=1; j<N; j++){
            cout << "\t" << aij[0][j][result[j][k]];
        }
        cout << endl << "\t";
        for(int j=1; j<N; j++){
            cout << "\t" << aij[j][0][result[j][k]];
        }
    }*/
    
    //check for consistency between i&j, by comparing with i&0 and j&0. Identify any violations.
    Array1D<bool> remove(result[0].size(),(bool)0);
    for(int i=1; i<N-1; i++){
        for(int j=i+1; j<N; j++){
            int ctr1 = 0;
            int ctr2 = 0;
            for(unsigned int k=0; k<result[0].size(); k++){
                if(!remove[k]){
                    while(aij[i][j][ctr1]<aij[i][0][result[i][k]]){
                        if(ctr1>=(int)aij[i][j].size()-1){
                            break;
                        }
                        ctr1++;
                    }
                    while(aij[j][i][ctr2]<aij[j][0][result[j][k]]){
                        if(ctr2>=(int)aij[j][i].size()-1){
                            break;
                        }
                        ctr2++;
                    }
                    //cout << endl << i << "\t" << j << "\t" << k << "\t" << "\t" << aij[i][j][ctr1] << "\t" << aij[j][i][ctr2];
                    if(aij[i][j][ctr1]!=aij[i][0][result[i][k]] || aij[j][i][ctr2]!=aij[j][0][result[j][k]] || aij[j][i][ctr1]!=aij[j][i][ctr2]){
                        remove[k] = 1;
                        //cout << "***";
                    }
                }
            }
        }
    }
    
    //construct final consensus alignment
    for(unsigned int k=0; k<result[0].size(); k++){
        if(!remove[k]){
            for(int j=1; j<N; j++){
                final[0][j].push_back(aij[0][1][result[0][k]]); //note: atoms are same, independent of which structure it's aligned to.
            }
            for(int i=1; i<N; i++){
                for(int j=0; j<N; j++){
                    if(i==j)continue;
                    final[i][j].push_back(aij[i][0][result[i][k]]);
                }
            }
        }
    }
    
    aij = final;
    
    /*for(int i=0; i<N; i++){
        for(int j=1; j<N; j++){
            if(i>=j){continue;}
            cout << endl << endl << "here: " << i << " " << j << endl << aij[i][j];
            cout << endl << "here: " << i << " " << j << endl << final[i][j];
            cout << endl << endl << "here: " << j << " " << i << endl << aij[j][i];
            cout << endl << "here: " << j << " " << i << endl << final[j][i];
        }
    }*/
    
    return;
}

Array2D<vector<double> > get_alignment_weights(Array2D<vector<unsigned int> > &a_ij)
{
    int N = a_ij.dim1();
    Array2D<vector<double> > result(N,N);
    for(int i=0; i<N; i++){
        for(int j=i+1; j<N; j++){
            for(unsigned int k=0; k<a_ij[i][j].size(); k++){
                result[i][j].push_back(1.0);
            }
            result[j][i] = result[i][j];
        }
    }
    return result;
}

Array2D<Coords> get_aligned_coords(Array1D<PDBfile> &pdb_v, Array2D<vector<unsigned int> > &a_ij)
{
    int N = pdb_v.dim1();
    Array2D<Coords> result(N,N);
    
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j){continue;}
            pdb_v[i].getcoords(result[i][j],a_ij[i][j]);
        }
    }
    
    return result;
}

Array1D<Transform> get_initial_transformation(Array2D<Coords> &crds)
{
    int N = crds.dim1();
    Array1D<Transform> result(N);
    
    initialise_identityM();
    result[0].create();
    for(int i=1; i<N; i++){
        result[i] = get_transf(crds[0][i],crds[i][0]);    //performs sieve fitting if flag is set
        //cout << endl << "Initial transformations:" << endl << i << result[i] << endl;
    }
    
    return result;
}

Array2D<Transform> get_all_transformations(Array2D<Coords> &crds)
{
    int N = crds.dim1();
    Array2D<Transform> result(N,N);
    
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j){continue;}
            result[i][j] = get_transf(crds[i][j],crds[j][i]);    //performs sieve fitting if flag is set
            cout << endl << i << " " << j << endl;
            cout << result[i][j].get_r() << endl << result[i][j].get_p();
        }
    }
    
    return result;
}

Array1D<Transform> optimise_rotations(Array2D<Coords> &aijXi, Array2D<vector<double> > &wij, Array1D<Transform> &transf)
{
    int N = aijXi.dim1();
    Array1D<Transform> result(N);
    Array2D<Coords> aijXi_mti(N,N);
    Array2D<Array2D<double> > XTwX(N,N);
    vector<vector<Array2D<double> > > R;
    vector<Array2D<double> > empty;
    Array2D<double> R_diff(3,3,0.0);
    unsigned int idx = 0;
    double angle = 0.0;
    vector<double> max_angle;
    
    //get XTwX, which is independent of R
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j)continue;
            aijXi_mti[i][j] = transf[i].pre_translate(aijXi[i][j]);
        }
    }
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j)continue;
            if(wij.dim1()>0){
                XTwX[i][j] = crossprod_weight(aijXi_mti[i][j],aijXi_mti[j][i],wij[i][j]);
            } else {
                XTwX[i][j] = aijXi_mti[i][j]*aijXi_mti[j][i];
            }
        }
    }
    
    //initialise R
    R.push_back(empty);
    for(int i=0; i<N; i++){
        R[0].push_back(transf[i].get_r());
    }
    
    //iteratively update R
    while(1){
        idx++;
        R.push_back(empty);
//#define DISPLAY_GEN_PROC_ROTATION_COUTS
#ifdef DISPLAY_GEN_PROC_ROTATION_COUTS
#define CALCULATE_TRCOV_FOR_GEN_PROC
#ifdef CALCULATE_TRCOV_FOR_GEN_PROC
        double tr_XkTXlRlRkT = 0.0;
#endif
#endif
        //update R (fix first rotation - don't need to do this, but it makes sense to fix first structure)
        R[idx].push_back(R[0][0]);
        for(int i=1; i<N; i++){
            Array2D<double> sumXTwXR(3,3,(double)0.0);
            for(int j=0; j<N; j++){
                if(i==j)continue;
                sumXTwXR += matmult(XTwX[i][j],R[idx-1][j]);
            }
            R[idx].push_back(rotateM(sumXTwXR));
#ifdef CALCULATE_TRCOV_FOR_GEN_PROC
            tr_XkTXlRlRkT += tr(crossprod(sumXTwXR,R[idx-1][i]));
#endif
        }
        
        //check convergence criteria
        max_angle.push_back(0.0);
        for(int i=0; i<N; i++){
            R_diff = crossprod(R[idx][i],R[idx-1][i]);
            angle = get_angle(R_diff);
            if(angle>max_angle.back()){
                max_angle.back() = angle;
            }
        }
#ifdef DISPLAY_GEN_PROC_ROTATION_COUTS
        cout.precision(6);
        cout << endl << "Iteration:" << idx << "\tMax angle: " << max_angle.back();
#ifdef CALCULATE_TRCOV_FOR_GEN_PROC
        cout.precision(15);
        cout << "\ttr(): " << tr_XkTXlRlRkT;
#endif
#endif
        if(idx>30)break;
        if(max_angle.back()<0.0001)break;
        if(idx>1)if(max_angle.back()>max_angle[idx-2])break;
    }
    
    //get updated transformations
    for(int i=0; i<N; i++){
        result[i] = transf[i];
        result[i].set_r(R.back()[i],transf[i].get_pre());
        //cout << endl << i << " " << transf[i] << result[i];
    }
    
    return result;
}

void optimise_translations(Array2D<Coords> &aijXi, Array2D<vector<double> > &wij, Array1D<Transform> &transf)
{
    int N = aijXi.dim1();
    Array2D<double> wij_hat(N,N,0.0);
    Array1D<double> sumwij_hat(N,0.0);
    Array1D<Array2D<double> > R(N);
    Array2D<double> sumYw(N,3,0.0);
    Array2D<double> A(N-1,N-1,0.0);
    vector<double> empty;
    vector<vector<double> > Z;
    Array1D<double> tmpcrd;
    
    // problem: (xI - M)T = Z
    //
    // T = R^Tt^T
    // x = sumwij_hat
    // Z = sumYw
    
    // get x (don't need for first i)
    for(int i=1; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j)continue;
            if(wij.dim1()>0){
                wij_hat[i][j] = sum(wij[i][j]);
            } else {
                wij_hat[i][j] = aijXi[i][j].size();
            }
            sumwij_hat[i] += wij_hat[i][j];
        }
    }

    // get Z (leaving the first translation at 0.0)
    for(int i=0; i<N; i++){
        R[i] = transf[i].get_r();
    }
    for(int x=0; x<3; x++){
        Z.push_back(empty);
    }
    for(int i=1; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j)continue;
            Coords tmp1 = aijXi[i][j]*R[i];
            Coords tmp2 = aijXi[j][i]*R[j];
            Coords Yij = tmp1-tmp2;
            if(wij.dim1()>0){
                tmpcrd = Yij.weighted_sum(wij[i][j]);
            } else {
                tmpcrd = Yij.sum();
            }
            for(int x=0; x<3; x++){
                sumYw[i][x] += tmpcrd[x];
            }
        }
        for(int x=0; x<3; x++){
            Z[x].push_back(0.0-sumYw[i][x]);
        }
    }

    // get A = xI - M (leaving the first translation at 0.0, apart from first diagonal at 1.0)
    for(int i=0; i<N-1; i++){
        for(int j=0; j<N-1; j++){
            if(i==j){
                A[i][j] = sumwij_hat[i+1];
            } else {
                A[i][j] = 0.0-wij_hat[i+1][j+1];                
            }
        }
    }
    
    //invert A and get new T
    Array2D<double> Ainv = pseudoinverse(A);
    //cout << endl << endl << "A:" << endl << A;
    //cout << endl << "Ainv:" << endl << Ainv;
    //cout << endl << "A*Ainv*A: " << endl << matmult(matmult(A,Ainv),A);
    //cout << endl << "Z:" << endl << Z;
    
    vector<Array1D<double> > Tnew;
    for(unsigned int x=0; x<3; x++){
        Tnew.push_back(vprod(Ainv,Z[x]));
        //cout << "Tnew " << x << endl << Tnew[x];
    }
    
    //note - first structure doesn't change
    coord new_T;
    coord new_RT;    
    for(int i=0; i<N-1; i++){
        new_T.x = Tnew[0][i];
        new_T.y = Tnew[1][i];
        new_T.z = Tnew[2][i];
        new_RT = R[i+1]*new_T;
        transf[i+1].set_r(R[i+1],new_RT);
    }

    return;
}

double brute_force_ss(Array2D<Coords> &aijXi, Array2D<vector<double> > &wij, Array1D<Transform> &transf)
{
    double result = 0.0;
    int N = aijXi.dim1();
    Coords crds_diff;

    for(int i=0; i<N-1; i++){
        for(int j=i+1; j<N; j++){
            crds_diff = transf[i].transform(aijXi[i][j]) - transf[j].transform(aijXi[j][i]);
            if(wij.dim1()>0){
                result += crds_diff.tr_cov(wij[i][j]);  //weighted sum of squares
            } else {
                result += crds_diff.tr_cov();  //sum of squares
            }
            //for(int k=0; k<crds_diff.size(); k++){cout << endl << dist(*crds_diff.get(k));}
        }
    }
    return result;
}

void output_multiple_superposition(vector<string> &files, vector<string> &chains, Array1D<Transform> &transf_v, string ext, bool app)
{
    bool OUT_PDB_FULL = 0;
    vector<string> empty;
    
    //output PDB files and PDB file loader
    string empty_s="";
    string tmp_dir="multiple_superposition/"+ext+"/";
    create_directory(empty_s,tmp_dir);
    ofstream outfile;
    string loader="loader.pml";
    if(app){
        outfile.open(loader.c_str(),ios::app);
    } else {
        outfile.open(loader.c_str());
    }
    if(outfile.is_open()){
        for(unsigned int i=0; i<files.size(); i++){
            string fileout2 = tmp_dir+get_filename(files[i])+"_"+chains[i]+".pdb";
            output_pdb(empty,files[i],fileout2,chains[i][0],transf_v[i],1,OUT_PDB_FULL);
            outfile << "load " << fileout2 << endl;
        }
        outfile.close();
    }
    
    return;
}

void output_multiple_superposition(vector<string> &files, vector<string> &chains, Array1D<Transform> &transf_v, int ctr)
{
    bool OUT_PDB_FULL = 0;
    vector<string> empty;
    
    //output PDB files and PDB file loader
    string empty_s="";
    string tmp_dir="multiple_superposition/";
    create_directory(empty_s,tmp_dir);
    tmp_dir+=int_to_str(ctr)+"/";
    create_directory(empty_s,tmp_dir);
    for(unsigned int i=0; i<files.size(); i++){
        string fileout2 = tmp_dir+get_filename(files[i])+"_"+chains[i]+".pdb";
        output_pdb(empty,files[i],fileout2,chains[i][0],transf_v[i],1,OUT_PDB_FULL);
        system(("echo \"load " + fileout2 + "\" >> loader_all.pml").c_str());
    }
    
    return;
}

void pymol_multiple_superposition(string file, vector<string> &files, vector<string> &chains, Array1D<PDBfile> &pdbs, Array2D<vector<unsigned int> > &aij)
{
    for(int i=2; i<aij.dim1(); i++){
        if(aij[i][0].size()!=aij[1][0].size()){
            return;
        }
    }
    ofstream outfile;
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        outfile << "color white, all" << endl;
        for(unsigned int j=0; j<aij[0][1].size(); j++){
            outfile << "color red, (" << get_filename(files[0]) << "//" << chains[0] << "/" << delete_spaces(pdbs[0].get_orig_resnum(aij[0][1][j])) << "/" << delete_spaces(pdbs[0].get_atom(aij[0][1][j])) << ")" << endl;
        }
        for(int i=1; i<pdbs.dim1(); i++){
            for(unsigned int j=0; j<aij[i][0].size(); j++){
                outfile << "color red, (" << get_filename(files[i]) << "//" << chains[i] << "/" << delete_spaces(pdbs[i].get_orig_resnum(aij[i][0][j])) << "/" << delete_spaces(pdbs[i].get_atom(aij[i][0][j])) << ")" << endl;
            }
        }
        outfile.close();
    }
    
    return;
}

void multiple_structure_scoring(vector<string> &files, Array1D<PDBfile> &pdbs, Array2D<vector<unsigned int> > &aij, Array1D<Transform> &transf, vector<double> &colvec)
{
    const int N = pdbs.dim1();
    Array1D<vector<vector<int> > > index(N);
    const vector<int> empty;
    
    for(int i=0; i<N; i++){
        for(int j=0; j<N; j++){
            if(i==j){continue;}
            for(unsigned int k=0; k<aij[i][j].size(); k++){
                while(index[i].size()<=aij[i][j][k]){
                    index[i].push_back(empty);
                }
                while((int)index[i][aij[i][j][k]].size()<N){
                    index[i][aij[i][j][k]].push_back(-1);
                }
                index[i][aij[i][j][k]][j] = aij[j][i][k];   //store corresponding indices from other structures
            }
        }
        //cout << endl << i << endl;for(unsigned int k=0; k<index[i].size(); k++){cout << k;for(unsigned int j=0; j<index[i][k].size(); j++){cout << "\t" << index[i][k][j];}cout << endl;}
    }
    
    for(int i=0; i<N; i++){
        string obj1 = get_filename(files[i]);
        for(int j=i+1; j<N; j++){
            string obj2 = get_filename(files[j]) + "_L";
            ofstream outfile;
            string file = "max_dist_" + int_to_str(i) + "_" + int_to_str(j) + ".pml";
            outfile.open(file.c_str());
            if(outfile.is_open()){
                outfile << "# ProSMART Colour File" << endl;
                outfile << "color white, all" << endl;
                for(unsigned int k=0; k<index[i].size(); k++){
                    if((int)index[i][k].size()==N){
                        if(index[i][k][j]>=0){
                            outfile << "color red, (" << obj1 << "_" << pdbs[i].get_chain(k) << "//" << pdbs[i].get_chain(k) << "/";
                            if(pdbs[i].get_resnum(k)<0){
                                outfile << "\\";
                            }
                            outfile << delete_spaces(pdbs[i].get_orig_resnum(k)) << "/" << delete_spaces(pdbs[i].get_atom(k)) << ")" << endl;
                            outfile << "color red, (" << obj2 << "//" << pdbs[j].get_chain(index[i][k][j]) << "/";
                            if(pdbs[j].get_resnum(index[i][k][j])<0){
                                outfile << "\\";
                            }
                            outfile << delete_spaces(pdbs[j].get_orig_resnum(index[i][k][j])) << "/" << delete_spaces(pdbs[j].get_atom(index[i][k][j])) << ")" << endl;
                        }
                    }
                    
                }
                
                outfile.close();
            }
        }
    }
    
    
    
    //get transformed coordinates
    vector<Coords> crd_transf;
    for(int i=0; i<pdbs.dim1(); i++){
        Coords crd;
        pdbs[i].getcoords(crd);
        crd_transf.push_back(transf[i].transform(crd));
    }
    
    //get max atomic deviation for one-on-all-chains (-1.0 if unaligned; doesn't require complete consensus)
    Array1D<vector<double> > atom_scores(N);
    for(int i=0; i<N; i++){
        while(atom_scores[i].size()<index[i].size()){
            atom_scores[i].push_back(-1.0);
        }
        for(unsigned int k=0; k<index[i].size(); k++){
            coord tmp1 = crd_transf[i].get_crd(k);
            for(unsigned int j=0; j<index[i][k].size(); j++){
                if(index[i][k][j]>=0){
                    coord tmp2 = crd_transf[j].get_crd(index[i][k][j]);
                    double tmpd = dist(tmp1,tmp2);
                    if(tmpd>atom_scores[i][k]){
                        atom_scores[i][k] = tmpd;
                    }
                }
            }
        }
    }
    
    //get max atomic deviation for all-on-all-chains (-1.0 if unaligned; doesn't require complete consensus)
    Array1D<vector<double> > atom_scores_all(N);
    for(int i=0; i<N; i++){
        for(unsigned int k=0; k<atom_scores[i].size(); k++){
            atom_scores_all[i].push_back(atom_scores[i][k]);
            for(unsigned int j=0; j<index[i][k].size(); j++){
                if(index[i][k][j]>=0){
                    if(atom_scores_all[i][k]<atom_scores[j][index[i][k][j]]){
                        atom_scores_all[i][k] = atom_scores[j][index[i][k][j]];
                    }
                }
            }
        }
    }
    
    //output colour script
    ofstream outfile;
    string file = "max_dist.pml";
    outfile.open(file.c_str());
    if(outfile.is_open()){
        outfile << "# ProSMART Colour File" << endl;
        outfile << "color white, all" << endl;
        
        double val = 0.0;
        double ival = 1.0;
        int NO_COL = 100;   //this will actually cause NO_COL+1 colours to be created...
        for(int i=0; i<=NO_COL; i++){
            val = (double)i/NO_COL;
            ival = 1.0-val;
            double color1 = colvec[0]*ival+colvec[3]*val;
            double color2 = colvec[1]*ival+colvec[4]*val;
            double color3 = colvec[2]*ival+colvec[5]*val;
            outfile << "set_color newcolor" << i << " = [" << color1 << "," << color2 << "," << color3 << "]" << endl;
        }
        
        double score_cutoff = 1.0;
        int idx = -1;
        for(int i=0; i<N; i++){
            string obj = get_filename(files[i]) + "_" + pdbs[i].get_chain(0);
            for(unsigned int j=0; j<atom_scores_all[i].size(); j++){
                val = (double)NO_COL*atom_scores_all[i][j]/score_cutoff;
                if(val < 0.0){
                    idx = -1;
                } else {
                    if(val < (double)NO_COL){
                        idx = floor(val);
                    } else {
                        idx = NO_COL;
                    }
                    outfile << "color newcolor" << idx << ", (" << obj << "//" << pdbs[i].get_chain(j) << "/";
                    if(pdbs[i].get_resnum(j)<0){
                        outfile << "\\";
                    }
                    outfile << delete_spaces(pdbs[i].get_orig_resnum(j)) << "/" << delete_spaces(pdbs[i].get_atom(j)) << ")" << endl;
                }
            }
        }
        
        outfile.close();
    }
    
    return;
}




