//
// This file is part of ProSMART.
//

#include "restrain_hbond.h"

bool test_atompair(PDBfile &pdb, int i, int j, hbond &hbond_params)
{
    //check atoms are separated by sufficient residues
    int diff_dist = pdb.get_resnum(j) - pdb.get_resnum(i);
    if(diff_dist < hbond_params.MIN_RESIDUE_SEPARATION){
        return 0;
    }
    if(diff_dist > hbond_params.MAX_RESIDUE_SEPARATION && hbond_params.MAX_RESIDUE_SEPARATION>=hbond_params.MIN_RESIDUE_SEPARATION){
        return 0;
    }
    for(unsigned int x=0; x<hbond_params.REMOVED_RESIDUE_SEPARATIONS.size(); x++){
        if(diff_dist == hbond_params.REMOVED_RESIDUE_SEPARATIONS[x])
            return 0;
    }
    
    //check atoms are separated by an allowed number of residues
    if(hbond_params.ALLOWED_RESIDUE_SEPARATIONS.size()>0){
        bool ALLOWED = 0;
        for(unsigned int i=0; i<hbond_params.ALLOWED_RESIDUE_SEPARATIONS.size(); i++){
            if(diff_dist == hbond_params.ALLOWED_RESIDUE_SEPARATIONS[i]){
                ALLOWED = 1;
                break;
            }
        }
        if(!ALLOWED)return 0;
    }
    
    //check atoms can form donor and acceptor (and other criteria)
    switch(hbond_params.RESTRAINT_TYPE){
        case 1:         // helix restraints
            if(pdb.get_atom(i)==" O  ")            //O-N
                if(pdb.get_atom(j)==" N  ") return 1;
            break;     
        case 2:         // beta-sheet restraints (or general main-chain restraints)
            if(pdb.get_atom(i)==" N  "){            //N-O
                if(pdb.get_atom(j)==" O  ") return 1;
            } else if(pdb.get_atom(i)==" O  ")     //O-N
                if(pdb.get_atom(j)==" N  ") return 1;
            break;
        case 0:         // need to add option to use monomer library to get info about donors/acceptors
        default:        // no filtering based on atom type
            return 1;
            break;
    }
    return 0;
}

int get_max_bonds(PDBfile &pdb1, int i, int &opt)
{
    switch(opt){
        case 1:
        case 2:         // main-chain restraints
            if(pdb1.get_atom(i)==" N  ")
                return 1;
            else if(pdb1.get_atom(i)==" O  ")
                return 2;
            else 
                return 0;
            break;
        case 0:         // need to add option to use monomer library to get info about donors/acceptors
        default:        // no filtering based on atom type
            return 1;   // don't know max bonds - assume 1.
            break;
    }
    return 0;
}

vector<vector<int> > h_bond_restraints(PDBfile &pdb1, vector<res_corresp> &original_res1, hbond &hbond_params, vector<residue_alignment> &res_align, vector<int> &RANGE_MIN, vector<int> &RANGE_MAX, vector<string> &RM_RESIDUES)
{
    //filter ALLOWED_RESIDUE_SEPARATIONS based on MIN_RESIDUE_SEPARATION, MAX_RESIDUE_SEPARATION and REMOVED_RESIDUE_SEPARATIONS
    for(unsigned int i=0; i<hbond_params.ALLOWED_RESIDUE_SEPARATIONS.size(); i++){
        if(hbond_params.ALLOWED_RESIDUE_SEPARATIONS[i]<hbond_params.MIN_RESIDUE_SEPARATION){
            hbond_params.ALLOWED_RESIDUE_SEPARATIONS.erase(hbond_params.ALLOWED_RESIDUE_SEPARATIONS.begin()+i);
            i--;
        } else if(hbond_params.ALLOWED_RESIDUE_SEPARATIONS[i]>hbond_params.MAX_RESIDUE_SEPARATION && hbond_params.MAX_RESIDUE_SEPARATION>=hbond_params.MIN_RESIDUE_SEPARATION){
            hbond_params.ALLOWED_RESIDUE_SEPARATIONS.erase(hbond_params.ALLOWED_RESIDUE_SEPARATIONS.begin()+i);
            i--;
        } else for(unsigned int j=0; j<hbond_params.REMOVED_RESIDUE_SEPARATIONS.size(); j++){
            if(hbond_params.ALLOWED_RESIDUE_SEPARATIONS[i]==hbond_params.REMOVED_RESIDUE_SEPARATIONS[j]){
                hbond_params.ALLOWED_RESIDUE_SEPARATIONS.erase(hbond_params.ALLOWED_RESIDUE_SEPARATIONS.begin()+i);
                i--;
                break;
            }
        }
    }
    
    cout << endl << endl << "Generating generic restraints, e.g. for hydrogen bonds." << endl;
    
    cout << endl << "Generic restraint type:  " << hbond_params.RESTRAINT_TYPE;
    if(hbond_params.RESTRAINT_TYPE==1){
        cout << " - only allow O-N atom-pairs" << endl << "(intended for helical h-bond restraints)";
    } else if(hbond_params.RESTRAINT_TYPE==2){
        cout << " - allow O-N and N-O atom-pairs" << endl << "(intended for all-main-chain and/or beta-sheet h-bond restraints)";
    } else {
        cout << " - no filtering - allow any main-chain atom-pairs" << endl << "(only recommended for highly customised purposes - use with caution)";
    }

    cout << endl << endl << "Criteria:";
    if(hbond_params.ALLOWED_RESIDUE_SEPARATIONS.size()>0){
        cout << endl << "Residue separation must be equal to:     " << hbond_params.ALLOWED_RESIDUE_SEPARATIONS[0];
        for(unsigned int i=1; i<hbond_params.ALLOWED_RESIDUE_SEPARATIONS.size()-1; i++){
            cout << ", " << hbond_params.ALLOWED_RESIDUE_SEPARATIONS[i];
        }
        if(hbond_params.ALLOWED_RESIDUE_SEPARATIONS.size()>1){
            cout << " or " << hbond_params.ALLOWED_RESIDUE_SEPARATIONS[hbond_params.ALLOWED_RESIDUE_SEPARATIONS.size()-1];
        }
        cout << " residues";
    } else {
        if(hbond_params.MIN_RESIDUE_SEPARATION>0){
            cout << endl << "Residue separation must be at least:     " << hbond_params.MIN_RESIDUE_SEPARATION << " residues";
        }
        if(hbond_params.MAX_RESIDUE_SEPARATION>=0 && hbond_params.MAX_RESIDUE_SEPARATION>=hbond_params.MIN_RESIDUE_SEPARATION){
            cout << endl << "Residue separation must be no more than: " << hbond_params.MAX_RESIDUE_SEPARATION << " residues";
        }
        if(hbond_params.REMOVED_RESIDUE_SEPARATIONS.size()>0){
            cout << endl << "Residue separation cannot be equal to:   " << hbond_params.REMOVED_RESIDUE_SEPARATIONS[0];
            for(unsigned int i=1; i<hbond_params.REMOVED_RESIDUE_SEPARATIONS.size()-1; i++){
                cout << ", " << hbond_params.REMOVED_RESIDUE_SEPARATIONS[i];
            }
            if(hbond_params.REMOVED_RESIDUE_SEPARATIONS.size()>1){
                cout << " or " << hbond_params.REMOVED_RESIDUE_SEPARATIONS[hbond_params.REMOVED_RESIDUE_SEPARATIONS.size()-1];
            }
            cout << " residues";
        }
    }
    cout << endl << "Maximum interatomic distance:            " << hbond_params.MAX_DIST;
    cout << endl << "Minimum interatomic distance:            " << hbond_params.MIN_DIST;
    cout << endl << "Target restraint value:                  " << hbond_params.DIST;
    cout << endl;
    
    ////////////////////////
    //get list of all sufficiently close atom-pairs
    
    vector<coord> outer_box = get_outer_box(pdb1);
    vector<vector<vector<vector<int> > > > boxed_atoms = get_boxed_atoms(pdb1, outer_box, hbond_params.MAX_DIST);
    Array1D<coord> atom_boxes = get_atom_boxes(boxed_atoms, pdb1.size());
    vector<vector<vector<vector<int> > > > array_adjacent_atoms = get_adjacent_atoms_array(boxed_atoms);
    vector<vector<int> > close_atoms;        
    for(int i=0; i<atom_boxes.dim1(); i++){
        close_atoms.push_back(get_close_atoms(i,atom_boxes,array_adjacent_atoms,pdb1,hbond_params.MAX_DIST));
    }
    

    ////////////////////////////////////////////////
    // the rest of this function identifies suitable atom-pairs
    // given the list of sufficiently close atom-pairs and filtering criteria
    ////////////////////////////////////////////////
    
    ////////////////////////
    //remove atoms-pairs that belong to unaligned residues
    if(hbond_params.STRICT_FRAG){
        int tmp_resnum = 0;
        bool VALID = 0;
        for(unsigned int i=0; i<close_atoms.size(); i++){
            tmp_resnum = pdb1.get_resnum(i);
            VALID = 0;
            for(unsigned int x=0; x<res_align.size(); x++)
                if(tmp_resnum==res_align[x].res1){
                    VALID = 1;
                    break;
                }
            if(!VALID)
                close_atoms[i].clear();
            else 
                for(unsigned int j=0; j<close_atoms[i].size(); j++){
                    int tmp_resnum = pdb1.get_resnum(close_atoms[i][j]);
                    bool VALID = 0;
                    for(unsigned int x=0; x<res_align.size(); x++)
                        if(tmp_resnum==res_align[x].res1){
                            VALID = 1;
                            break;
                        }
                    if(!VALID){
                        close_atoms[i].erase(close_atoms[i].begin()+j);
                        j--;
                    }
                }
        }
    }
    
    
    ////////////////////////
    //remove atom-pairs according to custom residue filtering
    if(RANGE_MIN.size()>0 || RM_RESIDUES.size()>0){
        string tmp_str;
        int tmp_resnum;
        bool VALID;
        for(unsigned int i=0; i<close_atoms.size(); i++){
            tmp_resnum = pdb1.get_resnum(i);
            VALID = 0;
            for(unsigned int x=0; x<RANGE_MIN.size(); x++)
                if(original_res1[tmp_resnum].res >= RANGE_MIN[x])
                    if(original_res1[tmp_resnum].res <= RANGE_MAX[x]){
                        VALID = 1;
                        break;
                    }
            if(!VALID){
                close_atoms[i].clear();
            } else {
                for(unsigned int x=0; x<RM_RESIDUES.size(); x++){
                    if(original_res1[tmp_resnum].ins == ' '){
                        tmp_str = int_to_str(original_res1[tmp_resnum].res);
                    } else {
                        tmp_str = int_to_str(original_res1[tmp_resnum].res) + original_res1[tmp_resnum].ins;
                    }
                    if(tmp_str == RM_RESIDUES[x]){
                        close_atoms[i].clear();
                        break;
                    }
                }
            }
            for(unsigned int j=0; j<close_atoms[i].size(); j++){
                tmp_resnum = pdb1.get_resnum(close_atoms[i][j]);
                VALID = 0;
                for(unsigned int x=0; x<RANGE_MIN.size(); x++)
                    if(original_res1[tmp_resnum].res >= RANGE_MIN[x])
                        if(original_res1[tmp_resnum].res <= RANGE_MAX[x]){
                            VALID = 1;
                            break;
                        }
                if(!VALID){
                    close_atoms[i].erase(close_atoms[i].begin()+j);
                    j--;
                } else {
                    for(unsigned int x=0; x<RM_RESIDUES.size(); x++){
                        if(original_res1[tmp_resnum].ins == ' '){
                            tmp_str = int_to_str(original_res1[tmp_resnum].res);
                        } else {
                            tmp_str = int_to_str(original_res1[tmp_resnum].res) + original_res1[tmp_resnum].ins;
                        }
                        if(tmp_str == RM_RESIDUES[x]){
                            close_atoms[i].erase(close_atoms[i].begin()+j);
                            j--;
                            break;
                        }
                    }
                }
            }
        }
    }
    
    
    ////////////////////////
    //remove atoms-pairs that are too close
    for(unsigned int i=0; i<close_atoms.size(); i++)
        for(unsigned int j=0; j<close_atoms[i].size(); j++)
            if(pdb1.get_distance(i,close_atoms[i][j])<hbond_params.MIN_DIST){
                close_atoms[i].erase(close_atoms[i].begin()+j);
                j--;
            }
    
    
    //////////////////////////    
    //check that atoms can be paired (e.g. require donor-acceptor pair, specific requirements for helix/sheet, etc.)
    for(unsigned int i=0; i<close_atoms.size(); i++)
        for(unsigned int j=0; j<close_atoms[i].size(); j++)
            if(!test_atompair(pdb1,i,close_atoms[i][j],hbond_params)){
                close_atoms[i].erase(close_atoms[i].begin()+j);
                j--;
            }
    
    
    //////////////////////////
    //get maximum potential number of restraints per atom
    vector<unsigned int> MAX_BONDS;
    for(unsigned int i=0; i<close_atoms.size(); i++)
        MAX_BONDS.push_back(get_max_bonds(pdb1,i,hbond_params.RESTRAINT_TYPE));
    
    if(hbond_params.OVERRIDE_MAX_BONDS>0)
        for(unsigned int i=0; i<MAX_BONDS.size(); i++)
            if(MAX_BONDS[i]>0)
                MAX_BONDS[i] = hbond_params.OVERRIDE_MAX_BONDS;
    
    
    //////////////////////////
    //need to allow specification of only parallel/antiparallel sheets
    
    
    //////////////////////////
    //add inverse to close_atoms... consequently need to filter at later point to undo this.
    vector<int> tmp_atoms;  //always empty
    for(unsigned int i=0; i<close_atoms.size(); i++){
        for(unsigned int j=0; j<close_atoms[i].size(); j++){
            if(close_atoms[i][j]<=(int)i)continue;
            while((int)close_atoms.size()<=close_atoms[i][j]){
                close_atoms.push_back(tmp_atoms);
            }
            close_atoms[close_atoms[i][j]].push_back(i);
        }
    }
    
    
    //////////////////////////
    //filter by number of restraints for each atom
    vector<vector<int> > removed_atoms;
    filter_restraints_by_max_bonds(pdb1,close_atoms,removed_atoms,MAX_BONDS,hbond_params.DIST);

    
    //////////////////////////
    //restore any removed restraints that can be kept
    restore_hbond_restraints(pdb1,original_res1,close_atoms,removed_atoms,MAX_BONDS,hbond_params.DIST);
    
    
    //////////////////////////
    //filter to ensure that two restraints are not generated for each atom-pair.
    for(unsigned int i=0; i<close_atoms.size(); i++){
        for(unsigned int j=0; j<close_atoms[i].size(); j++){
            if(close_atoms[i][j]<=(int)i){
                close_atoms[i].erase(close_atoms[i].begin()+j);
                j--;
            }
        }
    }

    
    //////////////////////////
    unsigned int total_restraints=0;
    for(unsigned int i=0; i<close_atoms.size(); i++){
        total_restraints += close_atoms[i].size();
    }
    cout << endl << endl << total_restraints << " restraints generated." << endl;
    
    if(hbond_params.TROUBLESHOOT && total_restraints>0){
        //display final interatomic distances:
        cout << endl << "List of generated generic (e.g. h-bond) restraints:";
        cout << endl << "Res1 Atom1";
        cout << endl << "\tRes2 Atom2  \tDistance";
        for(unsigned int i=0; i<close_atoms.size(); i++){
            if(close_atoms[i].size()==0){
                continue;
            }
            cout << endl << original_res1[pdb1.get_resnum(i)].res << original_res1[pdb1.get_resnum(i)].ins << pdb1.get_atom(i);
            for(unsigned int j=0; j<close_atoms[i].size(); j++){
                cout << endl << "\t" << original_res1[pdb1.get_resnum(close_atoms[i][j])].res << original_res1[pdb1.get_resnum(close_atoms[i][j])].ins << pdb1.get_atom(close_atoms[i][j]) << "     \t" << pdb1.get_distance(i,close_atoms[i][j]);
            }
        }
        cout << endl;
    }
    
    return close_atoms;
}

void filter_restraints_by_max_bonds(PDBfile &pdb1, vector<vector<int> > &close_atoms, vector<vector<int> > &removed_atoms, vector<unsigned int> &MAX_BONDS, double &HBOND_DIST)
{
    //filter by number of restraints for each atom
    vector<int> tmp_atoms;
    tmp_atoms.clear();
    removed_atoms.clear();
    for(unsigned int i=0; i<close_atoms.size(); i++){
        removed_atoms.push_back(tmp_atoms);
    }
    
    double test_dist = 0.0;
    double worst_dist = 0.0;
    int worst_idx = 0;
    for(unsigned int i=0; i<close_atoms.size(); i++){
        if(close_atoms[i].size()==0)continue;
        while(close_atoms[i].size()>MAX_BONDS[i]){
            worst_dist = 0.0;
            worst_idx = 0;
            for(unsigned int j=0; j<close_atoms[i].size(); j++){
                test_dist = abs(pdb1.get_distance(i,close_atoms[i][j])-HBOND_DIST);
                if(test_dist>worst_dist){
                    worst_dist = test_dist;
                    worst_idx = j;
                }
            }
            removed_atoms[i].push_back(close_atoms[i][worst_idx]);
            for(unsigned int j=0; j<close_atoms[close_atoms[i][worst_idx]].size(); j++){
                if(close_atoms[close_atoms[i][worst_idx]][j]==(int)i){
                    close_atoms[close_atoms[i][worst_idx]].erase(close_atoms[close_atoms[i][worst_idx]].begin()+j);
                    break;
                }
            }
            close_atoms[i].erase(close_atoms[i].begin()+worst_idx);
        }
    }
    return;
}


void restore_hbond_restraints(PDBfile &pdb1, vector<res_corresp> &original_res1, vector<vector<int> > &close_atoms, vector<vector<int> > &removed_atoms, vector<unsigned int> &MAX_BONDS, double &HBOND_DIST)
{
    //check whether any removed restraints can be restored
    vector<int> restore1;
    vector<int> restore2;
    for(unsigned int i=0; i<removed_atoms.size(); i++){
        if(removed_atoms[i].size()==0)continue;
        //cout << endl << i << " " << pdb1.get_atom(i);
        if(close_atoms[i].size()<MAX_BONDS[i]){
            //cout << "\t*";
            for(unsigned int j=0; j<removed_atoms[i].size(); j++){                
                //cout << endl << "\t" << removed_atoms[i][j] << " " << pdb1.get_atom(removed_atoms[i][j]);
                if(close_atoms[removed_atoms[i][j]].size()<MAX_BONDS[removed_atoms[i][j]]){
                    //cout << "\t***";
                    restore1.push_back(i);
                    restore2.push_back(j);
                }
            }
        }
    }
    /*for(unsigned int i=0; i<restore1.size(); i++){
     cout << endl << restore1[i] << " " << pdb1.get_atom(restore1[i]) << " " << removed_atoms[restore1[i]][restore2[i]] << " " << pdb1.get_atom(removed_atoms[restore1[i]][restore2[i]]) << " " << pdb1.get_distance(restore1[i],removed_atoms[restore1[i]][restore2[i]]);
     }*/
    
    if(restore1.size()>0){
        cout << endl << "Trying to restore " << restore1.size() << " potential restraint";if(restore1.size()>1){cout << "s";}cout << "...";
        
        //determine which restraints can be easily restored without clashes
        vector<unsigned int> lens;
        for(unsigned int i=0; i<close_atoms.size(); i++){       //occupy vector with current number of restraints
            lens.push_back(close_atoms[i].size());
        }
        for(unsigned int i=0; i<restore1.size(); i++){          //add all potential restorations
            lens[restore1[i]]++;
            lens[removed_atoms[restore1[i]][restore2[i]]]++;
        }
        for(unsigned int i=0; i<restore1.size(); i++){          //restore non-clashing restraints
            if(lens[restore1[i]] <= MAX_BONDS[restore1[i]]){
                if(lens[removed_atoms[restore1[i]][restore2[i]]] <= MAX_BONDS[removed_atoms[restore1[i]][restore2[i]]]){
                    close_atoms[restore1[i]].push_back(removed_atoms[restore1[i]][restore2[i]]);
                    close_atoms[removed_atoms[restore1[i]][restore2[i]]].push_back(restore1[i]);
                    restore1.erase(restore1.begin()+i);
                    restore2.erase(restore2.begin()+i);
                    i--;
                }
            }
        }
        
        /*for(unsigned int i=0; i<restore1.size(); i++){
         cout << endl << restore1[i] << " " << pdb1.get_atom(restore1[i]) << " " << removed_atoms[restore1[i]][restore2[i]] << " " << pdb1.get_atom(removed_atoms[restore1[i]][restore2[i]]) << " " << pdb1.get_distance(restore1[i],removed_atoms[restore1[i]][restore2[i]]);
         }*/
        
        
        
        if(restore1.size()>0){            
            cout << endl << "Trying to restore " << restore1.size() << " potential restraint";if(restore1.size()>1){cout << "s";}cout << "...";
            //restore restraints in order of favourability
            int best_idx = 0;
            double best_dist = 0.0;
            double test_dist = 0.0;
            vector<int> fail1;
            vector<int> fail2;
            while(restore1.size()>0){
                best_dist = abs(pdb1.get_distance(restore1[0],removed_atoms[restore1[0]][restore2[0]])-HBOND_DIST);
                best_idx = 0;
                for(unsigned int i=0; i<restore1.size(); i++){
                    if(close_atoms[restore1[i]].size() >= MAX_BONDS[restore1[i]] || close_atoms[removed_atoms[restore1[i]][restore2[i]]].size() >= MAX_BONDS[removed_atoms[restore1[i]][restore2[i]]]){
                        fail1.push_back(restore1[i]);
                        fail2.push_back(restore2[i]);
                        restore1.erase(restore1.begin()+i);
                        restore2.erase(restore2.begin()+i);
                        i--;
                        continue;
                    }
                    test_dist = abs(pdb1.get_distance(restore1[i],removed_atoms[restore1[i]][restore2[i]])-HBOND_DIST);
                    if(test_dist<best_dist){
                        best_dist = test_dist;
                        best_idx = i;
                    }
                }
                if(restore1.size()>0){
                    //cout << endl << "restore: " << restore1[best_idx] << " " << pdb1.get_atom(restore1[best_idx]) << " " << removed_atoms[restore1[best_idx]][restore2[best_idx]] << " " << pdb1.get_atom(removed_atoms[restore1[best_idx]][restore2[best_idx]]) << " " << pdb1.get_distance(restore1[best_idx],removed_atoms[restore1[best_idx]][restore2[best_idx]]);
                    close_atoms[restore1[best_idx]].push_back(removed_atoms[restore1[best_idx]][restore2[best_idx]]);
                    close_atoms[removed_atoms[restore1[best_idx]][restore2[best_idx]]].push_back(restore1[best_idx]);
                    restore1.erase(restore1.begin()+best_idx);
                    restore2.erase(restore2.begin()+best_idx);
                }
            }
            
            if(fail1.size()>0){
                cout << endl << "Warning: " << fail1.size() << " potential restraints not generated (in favour of more likely candidates):";
                cout << endl << "Res1\tAtom1\tRes2\tAtom2\tDistance";
                for(unsigned int i=0; i<fail1.size(); i++){
                    cout << endl << original_res1[pdb1.get_resnum(fail1[i])].res << original_res1[pdb1.get_resnum(fail1[i])].ins
                    << "\t" << pdb1.get_atom(fail1[i])
                    << "\t" << original_res1[pdb1.get_resnum(removed_atoms[fail1[i]][fail2[i]])].res << original_res1[pdb1.get_resnum(removed_atoms[fail1[i]][fail2[i]])].ins
                    << "\t" << pdb1.get_atom(removed_atoms[fail1[i]][fail2[i]])
                    << "\t" << pdb1.get_distance(fail1[i],removed_atoms[fail1[i]][fail2[i]]);
                }
                cout << endl;
            }
        }
    }

    return;
}

bool read_hbond_parameters_file(hbond &hbond_params, string &filein)
{    
    ifstream infile(filein.c_str());
    if(!infile){
		cout << endl << endl << "Error - cannot open " << filein << " for reading." << endl << endl;
		return 0;
	}
    string line;
    bool result = 1;
    while(!infile.eof()) {
        line.clear();
        getline(infile,line);
        hbond_params.DIST = atof(line.c_str());
        
        line.clear();
        getline(infile,line);
        hbond_params.MIN_DIST = atof(line.c_str());
        
        line.clear();
        getline(infile,line);
        hbond_params.MAX_DIST = atof(line.c_str());
        
        line.clear();
        getline(infile,line);
        hbond_params.MIN_RESIDUE_SEPARATION = atoi(line.c_str());
        
        line.clear();
        getline(infile,line);
        hbond_params.MAX_RESIDUE_SEPARATION = atoi(line.c_str());
        
        line.clear();
        getline(infile,line);
        if(line!="#")return 0;
        
        line.clear();
        getline(infile,line);
        while(line!="#") {
            hbond_params.ALLOWED_RESIDUE_SEPARATIONS.push_back(atoi(line.c_str()));
            line.clear();
            getline(infile,line);
        }
        
        line.clear();
        getline(infile,line);
        while(line!="#") {
            hbond_params.REMOVED_RESIDUE_SEPARATIONS.push_back(atoi(line.c_str()));
            line.clear();
            getline(infile,line);
        }
        
        line.clear();
        getline(infile,line);
        hbond_params.RESTRAINT_TYPE = atoi(line.c_str());
        
        line.clear();
        getline(infile,line);
        hbond_params.OVERRIDE_MAX_BONDS = atoi(line.c_str());
        
        line.clear();
        getline(infile,line);
        hbond_params.TROUBLESHOOT = atoi(line.c_str());
        
        line.clear();
        getline(infile,line);
        hbond_params.STRICT_FRAG = atoi(line.c_str());
        
        line.clear();
        getline(infile,line);
        if(line!="#END")return 0;
        
        break;
    }
    infile.close();

    return result;
}




